/*
 * #%L
 * I18n :: Maven Plugin
 * 
 * $Id: ParserValidationMojo.java 1808 2010-11-16 08:59:10Z tchemit $
 * $HeadURL: http://svn.nuiton.org/svn/i18n/tags/i18n-2.0/maven-i18n-plugin/src/main/java/org/nuiton/i18n/plugin/parser/impl/ParserValidationMojo.java $
 * %%
 * Copyright (C) 2007 - 2010 CodeLutin
 * %%
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Lesser General Public License as 
 * published by the Free Software Foundation, either version 3 of the 
 * License, or (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Lesser Public License for more details.
 * 
 * You should have received a copy of the GNU General Lesser Public 
 * License along with this program.  If not, see
 * <http://www.gnu.org/licenses/lgpl-3.0.html>.
 * #L%
 */

package org.nuiton.i18n.plugin.parser.impl;

import org.nuiton.i18n.plugin.parser.I18nSourceEntry;
import org.nuiton.i18n.plugin.parser.SourceEntry;
import org.nuiton.io.FileUpdater;
import org.nuiton.io.MirroredFileUpdater;
import org.xml.sax.EntityResolver;
import org.xml.sax.InputSource;

import javax.xml.parsers.DocumentBuilder;
import javax.xml.xpath.XPath;
import java.io.File;
import java.io.IOException;
import java.net.SocketTimeoutException;
import java.net.URL;
import java.net.URLConnection;

/**
 * Find i18n keys from xworks xml validation files.
 * <p/>
 * <b>Note: </b> this goal must always be invoked before the {@code process-resources}
 * phase, otherwise all files will be considered as uptodate.
 *
 * @author tchemit <chemit@codelutin.com>
 * @goal parserValidation
 * @phase generate-resources
 */
public class ParserValidationMojo extends AbstractParserXmlMojo {

    final URL xworksResource = getClass().getResource("/xwork-validator-1.0.2.dtd");

    /**
     * Root directory of the default entry.
     *
     * @parameter expression="${i18n.defaultBasedir}" default-value="${basedir}/src/main/resources"
     * @required
     */
    protected File defaultBasedir;

    /**
     * Default included files to process (ant-like expression).
     *
     * @parameter expression="${i18n.defaultIncludes}" default-value="**\/**-validation.xml"
     */
    protected String defaultIncludes;

    /**
     * Defines the core rules file used to detect i18n keys in xml validation
     * files.
     * <p/>
     * <b>Note :</b> If you do not want to use it, set it to empty and fill the
     * {@link #userRulesFiles} parameter.
     *
     * @parameter expression="${i18n.coreRuleFile}" default-value="validation.rules"
     * @since 2.0
     */
    protected String coreRuleFile;

    /**
     * Always use the local xowrks dtd to increase performance.
     *
     * @parameter expression="${i18n.useLocalResolver}" default-value="true"
     * @since 1.6.0
     */
    protected boolean useLocalResolver;

    /**
     * Defines the file name of the getter where to put detected i18n keys
     * while getter phase.
     *
     * @parameter expression="${i18n.outputGetter}" default-value="validation.getter"
     * @since 2.0
     */
    protected String outputGetter;


    @Override
    public String[] getDefaultIncludes() {
        return new String[]{defaultIncludes};
    }

    @Override
    public String[] getDefaultExcludes() {
        return I18nSourceEntry.EMPTY_STRING_ARRAY;
    }

    @Override
    public File getDefaultBasedir() {
        return defaultBasedir;
    }

    @Override
    public FileUpdater newFileUpdater(SourceEntry entry) {
        return new MirroredFileUpdater("", "", entry.getBasedir(), cp) {

            @Override
            public File getMirrorFile(File f) {
                String file =
                        f.getAbsolutePath().substring(prefixSourceDirecotory);
                return new File(destinationDirectory + File.separator + file);
            }
        };
    }

    @Override
    protected String getOutGetter() {
        return outputGetter;
    }

    @Override
    protected String getCoreRuleFile() {
        return coreRuleFile;
    }

    protected XmlFileParser newXmlFileParser(final XPath xpath,
                                             final DocumentBuilder builder) {

        // add cached entity resolver
        builder.setEntityResolver(getEntityResolver());

        return new XmlFileParser(getLog(),
                                 getEncoding(),
                                 oldParser,
                                 showTouchedFiles,
                                 rules,
                                 xpath,
                                 builder,
                                 namespaces,
                                 isVerbose()) {
            @Override
            public String extract(String i18nString) {
                String s = null;
                if (!i18nString.trim().isEmpty()) {
                    s = i18nString.trim();
                    int end = s.indexOf("##");
                    if (end > 0) {
                        // remove params from key
                        s = s.substring(0, end);
                    }
                }
                if (getLog().isDebugEnabled()) {
                    getLog().debug(i18nString + " = " + s);
                }
                return s;
            }
        };
    }

    protected EntityResolver getEntityResolver() {

        return new EntityResolver() {

            public static final String XWORK_PUBLIC_ID =
                    "-//OpenSymphony Group//XWork Validator 1.0.2//EN";

            boolean useLocal = useLocalResolver;

            @Override
            public InputSource resolveEntity(String publicId,
                                             String systemId) throws IOException {
                if (getLog().isDebugEnabled()) {
                    getLog().debug("publicID:" + publicId + ", systemId:" +
                                   systemId);
                }
                if (XWORK_PUBLIC_ID.equals(publicId)) {
                    if (!useLocal) {
                        URL uri = new URL(systemId);
                        if (verbose) {
                            getLog().info("try to connect to " + uri);
                        }
                        URLConnection openConnection = uri.openConnection();
                        openConnection.setUseCaches(true);
                        openConnection.setConnectTimeout(1000);
                        try {
                            openConnection.connect();
                            return new InputSource(
                                    openConnection.getInputStream());
                        } catch (SocketTimeoutException e) {
                            useLocal = true;
                        } catch (IOException e) {
                            useLocal = true;
                        }
                    }

                    // use directly local resource
                    InputSource inputSource =
                            new InputSource(xworksResource.openStream());
                    return inputSource;
                }
                // use the default behaviour
                return null;
            }
        };
    }

    @Override
    @Deprecated
    protected String getKeyModifierStart() {
        return "=\\s*[\"\']";
    }

    @Override
    @Deprecated
    protected String getKeyModifierEnd() {
        return "[\"\']";
    }
}
