/*
 * *##% Plugin maven pour i18n
 * Copyright (C) 2007 - 2009 CodeLutin
 *
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Lesser General Public License as
 * published by the Free Software Foundation, either version 3 of the
 * License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Lesser Public License for more details.
 *
 * You should have received a copy of the GNU General Lesser Public
 * License along with this program.  If not, see
 * <http://www.gnu.org/licenses/lgpl-3.0.html>. ##%* */
package org.nuiton.i18n.plugin;

import java.net.MalformedURLException;
import org.apache.maven.plugin.MojoExecutionException;
import org.apache.maven.plugin.MojoFailureException;

import java.io.File;
import java.io.FileOutputStream;
import java.io.IOException;
import java.net.URL;
import java.util.Arrays;
import java.util.LinkedHashMap;
import java.util.List;
import java.util.Locale;
import java.util.Map;
import java.util.Map.Entry;
import org.apache.maven.artifact.Artifact;
import org.apache.maven.artifact.factory.ArtifactFactory;
import org.apache.maven.artifact.metadata.ArtifactMetadataSource;
import org.apache.maven.artifact.repository.ArtifactRepository;
import org.apache.maven.artifact.resolver.ArtifactCollector;
import org.apache.maven.artifact.resolver.filter.ArtifactFilter;
import org.apache.maven.artifact.resolver.filter.ScopeArtifactFilter;
import org.apache.maven.project.MavenProject;
import org.apache.maven.project.MavenProjectBuilder;
import org.apache.maven.shared.dependency.tree.DependencyNode;
import org.apache.maven.shared.dependency.tree.DependencyTreeBuilder;
import org.apache.maven.shared.dependency.tree.DependencyTreeBuilderException;
import org.nuiton.i18n.bundle.I18nBundleEntry;
import org.nuiton.util.PluginHelper.SortedProperties;
import org.nuiton.i18n.bundle.I18nBundleFactory;
import org.nuiton.util.DependencyUtil;
import org.nuiton.util.PluginHelper;

/**
 * Créer un bundle pour une application finale.
 *
 * Cela génère un merge de tous les fichiers i18n utilisés en un seul.
 *
 * On utilise la dépendance sur les artifacts pour connaitre l'ordre le chargement
 * des bundles.
 *
 * Si dans un bundle childs, la valeur de la clef est vide, on conserve alors celui
 * du parent,
 *
 * Ainsi on obtient un bundle dont toutes les clefs sont traduites.
 *
 * Le but aussi d'utiliser un unique bundle est de gagner du temps au runtime
 * car la recherche des bundles devient trop couteuse en temps lorsque l'on a de
 * nombreuses dépendances (au dessus de 100 deps cela peut prendre plusieurs
 * secondes, ce qui 'est pas acceptable).
 *
 * On a ajoute un second mode d'initialisation dans la clesse I18n pour n'utiliser
 * qu'un seul bundle et courcircuiter le chargement couteux...
 *  *
 *
 * @author chemit
 * @goal bundle
 * @phase generate-resources
 * @execute goal=gen
 * @requiresProject true
 * @requiresDependencyResolution runtime
 * 
 * @since 0.12
 */
public class Bundle extends AbstractI18nPlugin {

    /**
     * Repertoire ou generer les bundles.
     *
     * @parameter expression="${i18n.bundleOutputDir}" default-value="${basedir}/target/generated-sources/resources/META-INF"
     * @required
     * @since 1.0.0
     */
    protected File bundleOutputDir;
    /**
     * Nom du bundle a generer.
     *
     * @parameter expression="${i18n.bundleOutputName}" default-value="${project.artifactId}-i18n"
     * @required
     * @since 1.0.0
     */
    protected String bundleOutputName;
    /**
     * Un drapeau pour vérifier que les bundles ne contiennent pas d'entrées vides.
     *
     * @parameter expression="${i18n.checkBundle}" default-value="true"
     * @required
     * @since 1.0.0
     */
    protected boolean checkBundle;
    /**
     * Un drapeau pour afficher les entrées vides. (nécessite {@link #checkBundle} activé).
     *
     * @parameter expression="${i18n.showEmpty}" default-value="false"
     * @required
     * @since 1.0.0
     */
    protected boolean showEmpty;
    /**
     * Dependance du projet.
     *
     * @parameter default-value="${project}"
     * @required
     * @since 1.0.0
     */
    protected MavenProject project;
    /**
     * Local Repository.
     *
     * @parameter expression="${localRepository}"
     * @required
     * @readonly
     * @since 1.0.0
     */
    protected ArtifactRepository localRepository;
    /**
     * Remote repositories used for the project.
     *
     * @parameter expression="${project.remoteArtifactRepositories}"
     * @required
     * @readonly
     * @since 1.0.0
     */
    protected List<?> remoteRepositories;
    /**
     * Dependency tree builder component.
     *
     * @component
     */
    protected DependencyTreeBuilder dependencyTreeBuilder;
    /**
     * Artifact Factory component.
     *
     * @component
     */
    protected ArtifactFactory factory;
    /**
     * Artifact metadata source component.
     *
     * @component
     */
    protected ArtifactMetadataSource artifactMetadataSource;
    /**
     * Artifact collector component.
     *
     * @component
     */
    protected ArtifactCollector collector;
    /**
     * Maven Project Builder component.
     *
     * @component
     */
    protected MavenProjectBuilder mavenProjectBuilder;
    protected I18nArtifact[] i18nArtifacts;
    protected ClassLoader loader;
    protected URL[] urls;

    @Override
    public void init() {
        super.init();

        if (locales == null || locales.length == 0) {
            throw new IllegalStateException("il faut au moins une locale declaree (utiliser la propriete 'bundles')");
        }

        if (!bundleOutputDir.exists()) {
            bundleOutputDir.mkdirs();
        }

        try {
            // calcul des artifacts qui ont un bundle i18n et trie selon les
            // dependances

            i18nArtifacts = detectI18nArtifacts();

            if (!silent) {
                getLog().info("detected " + i18nArtifacts.length + " i18n artifact(s) : ");
                for (I18nArtifact a : i18nArtifacts) {
                    getLog().info(" - " + a);
                }
            }
        } catch (Exception ex) {
            throw new RuntimeException(ex);
        }

        // ajout de repertoire de generation (le parent en fait)
        // dans les resources du projet

        String newresourceDir = bundleOutputDir.getParentFile().getAbsolutePath();

        PluginHelper.addResourceDir(newresourceDir, project);
    }

    @Override
    public void execute() throws MojoExecutionException, MojoFailureException {

        if ("pom".equals(project.getPackaging()) || "site".equals(project.getPackaging())) {
            return;
        }


        long t00 = System.nanoTime();

        init();

        if (!silent) {
            getLog().info("config - bundle name : " + bundleOutputName);
            getLog().info("config - baseidr     : " + bundleOutputDir);
            getLog().info("config - locales     : " + Arrays.toString(locales));
        }
        // la locale par defaut est la première
        Locale defaultLocale = locales[0];

        Map<Locale, String> bundleDico = new LinkedHashMap<Locale, String>(locales.length);

        try {

            for (Locale locale : locales) {

                long t0 = System.nanoTime();

                File bundleOut = getI18nFile(bundleOutputDir, bundleOutputName, locale, false);

                if (!silent) {
                    getLog().info("generate bundle for locale " + locale);
                }

                SortedProperties propertiesOut = new SortedProperties(encoding, false);
                StringBuilder buffer = new StringBuilder();
                for (I18nArtifact artifact : i18nArtifacts) {
                    I18nBundleEntry[] bundleEntries = artifact.getBundleEntries(locale, defaultLocale);
                    for (I18nBundleEntry bundleEntry : bundleEntries) {

                        bundleEntry.load(propertiesOut);
                        String strPath = bundleEntry.getPath().toString();
                        int index = strPath.indexOf("i18n/");

                        buffer.append(',').append(strPath.substring(index));
                        if (verbose) {
                            getLog().info("loaded " + bundleEntry.getPath() + " in " + PluginHelper.convertTime(t0, System.nanoTime()));
                        }
                    }
                }
                if (buffer.length() > 0) {
                    bundleDico.put(locale, buffer.substring(1));
                    if (!silent) {
                        getLog().info("bundles for locale : " + bundleDico.get(locale));
                    }
                }
                propertiesOut.store(bundleOut);
                if (!silent && verbose) {
                    getLog().info("bundle created in " + PluginHelper.convertTime(t0, System.nanoTime()) + " (detected sentences : " + propertiesOut.size() + ")");
                }
                if (checkBundle) {
                    checkBundle(locale, propertiesOut, showEmpty);
                }
            }

            // ecriture du ficher des definitions i18n (permet de faire une
            // recherche extact sur un fichier puis d'en deduire les bundles a
            // charger
            String f = String.format(I18nBundleFactory.UNIQUE_BUNDLE_DEF, bundleOutputName);
            File defOut = new File(bundleOutputDir, f);
            if (!silent) {
                getLog().info("prepare i18n definition " + defOut.getAbsolutePath());
            }
            SortedProperties p = new SortedProperties(encoding, false);
            p.setProperty(I18nBundleFactory.BUNDLE_DEF_LOCALES, bundles);
            for (Entry<Locale, String> e : bundleDico.entrySet()) {
                p.setProperty(I18nBundleFactory.BUNDLES_FOR_LOCALE + e.getKey().toString(), e.getValue());
            }
            p.store(new FileOutputStream(defOut), null);

            if (!silent && verbose) {
                getLog().info("done in " + PluginHelper.convertTime(t00, System.nanoTime()));
            }
        } catch (IOException e) {
            getLog().error("File Error I/O ", e);
            throw new MojoFailureException("File Error I/O ");
        }
    }

    /**
     * Detecte les {@link I18nArtifact} et les retourne dans l'ordre de chargement
     * dans le système i18n, i.e l'ordre des dependances entre artifacts.
     *
     * @return les artifacts  i18nables triés par leur ordre de chargement dans le système i18n.
     * 
     * @throws MalformedURLException
     * @throws IOException
     * @throws DependencyTreeBuilderException
     */
    protected I18nArtifact[] detectI18nArtifacts() throws MalformedURLException, IOException, DependencyTreeBuilderException {

        Map<Artifact, I18nArtifact> dico = new java.util.HashMap<Artifact, I18nArtifact>();

        I18nArtifact i18nArtifact;
        for (Object o : project.getArtifacts()) {
            i18nArtifact = new I18nArtifact((Artifact) o);
            if (i18nArtifact.detectBundles()) {
                if (!silent && getLog().isDebugEnabled()) {
                    getLog().debug("detected artifact " + i18nArtifact);
                }
                dico.put(i18nArtifact.getArtifact(), i18nArtifact);
            } else {
                if (!silent && getLog().isDebugEnabled()) {
                    getLog().debug("reject artifact " + i18nArtifact);
                }
            }
        }

        ArtifactFilter artifactFilter = new ScopeArtifactFilter(Artifact.SCOPE_RUNTIME);

        DependencyNode rootNode = dependencyTreeBuilder.buildDependencyTree(project, localRepository, factory,
                artifactMetadataSource, artifactFilter, collector);

        List<Artifact> artifacts = new java.util.ArrayList<Artifact>(dico.keySet());

        DependencyUtil.sortArtifacts(rootNode, artifacts, getLog().isDebugEnabled());

        // l'artifact du projet est traite en dernier car s'il possède des
        // bundles alors ils doivent etre charge en dernier

        Artifact projectArtifact = project.getArtifact();
        i18nArtifact = new I18nArtifact(projectArtifact, src.getParentFile());

        if (i18nArtifact.detectBundles()) {
            if (!silent && verbose) {
                getLog().info("detected artifact " + i18nArtifact);
            }
            artifacts.add(i18nArtifact.getArtifact());
            dico.put(i18nArtifact.getArtifact(), i18nArtifact);
        }

        I18nArtifact[] result = new I18nArtifact[artifacts.size()];
        int i = 0;
        for (Artifact artifact : artifacts) {
            result[i++] = dico.get(artifact);
        }
        return result;
    }
}
