/**
 * *##% guix-compiler-gwt
 * Copyright (C) 2009 CodeLutin
 *
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Lesser General Public License as
 * published by the Free Software Foundation, either version 3 of the
 * License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Lesser Public License for more details.
 *
 * You should have received a copy of the GNU General Lesser Public
 * License along with this program.  If not, see
 * <http://www.gnu.org/licenses/lgpl-3.0.html>. ##%*
 */
package org.nuiton.guix.generator;

import com.google.gwt.user.client.ui.Label;
import com.google.gwt.user.client.ui.MenuBar;
import java.beans.IntrospectionException;
import java.io.File;
import java.io.FileInputStream;
import java.io.FileNotFoundException;
import java.io.InputStreamReader;
import java.io.Reader;
import org.nuiton.guix.model.GuixModelObject;

import java.lang.reflect.Modifier;
import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import org.nuiton.guix.BindingUtils;
import org.nuiton.guix.GwtEventHandler;
import org.nuiton.guix.model.AttributeDescriptor;
import org.nuiton.guix.model.StyleSheet;
import org.nuiton.guix.tags.ScriptHandler;
import org.nuiton.guix.tags.ScriptHandler.ScriptPart;
import org.nuiton.guix.tags.TagHandler;
import org.nuiton.guix.tags.TagManager;
import org.nuiton.guix.tags.gwt.CellHandler;
import org.nuiton.guix.tags.gwt.MenuBarHandler;
import org.nuiton.guix.tags.gwt.MenuHandler;
import org.nuiton.guix.tags.gwt.MenuItemHandler;
import org.nuiton.guix.tags.gwt.RowHandler;
import org.nuiton.guix.tags.gwt.TableHandler;

/**
 * Generates a GWT abstract class
 *
 * @author kmorin
 */
public class GwtAbstractClassGenerator extends GwtJavaFileGenerator {

    private static final String SETTER_PATTERN = "%1$s oldValue = this.%2$s;\n%3$s\nsupport.firePropertyChange(\"%2$s\", oldValue, %4$s);";
    private static final String GWT_DEFAULT_PACKAGE = "com.google.gwt.user.client.ui";

    /** Stylesheets applied to the objects of the class */
    List<StyleSheet> styleSheets = new ArrayList<StyleSheet>();
    /** map of the different parts of the script associated with gmo */
    Map<ScriptPart, Object> script;
    /** Bindings to generate */
    Map<String,Map<String, String>> bindings2Generate = new HashMap<String,Map<String, String>>();
    /** the GWT classloader */
    ClassLoader gwtClassLoader = Label.class.getClassLoader();
    GwtGenerator gwtGenerator;

    /**
     * Constructor
     *
     * @param gmo GuixModelObject representing the class to generate
     * @param classes List of the classes to generate
     */
    public GwtAbstractClassGenerator(GuixModelObject gmo, Map<String, JavaFile> classes) {
        super(gmo, classes);
        jf = new JavaFile(Modifier.PUBLIC | Modifier.ABSTRACT,
                JavaFile.ABSTRACT_CLASS,
                gmo.getClassDescriptor().getPackageName(),
                gmo.getClassDescriptor().getName() + "Abstract",
                null, gmo.getJavadoc());

        script = ScriptHandler.decomposeScript(gmo.getClassDescriptor().getScript());
    }

    public void setGwtGenerator(GwtGenerator gwtGenerator) {
        this.gwtGenerator = gwtGenerator;
    }

    @Override
    public JavaFile generate() {
        super.addImports();

        //add imports
        for (String s : (List<String>) script.get(ScriptPart.IMPORTS)) {
            jf.addImport(s);
        }

        //add fields
        for(JavaField field : (List<JavaField>) script.get(ScriptPart.FIELDS)) {
            jf.addField(field, true);
        }

        //add bodycode
        jf.addBodyCode((String) script.get(ScriptPart.BODYCODE));

        try {
            //get creation, setting and displying methods content
            Map<Method, String> methodBodies = browseModelObjects(gmo);

            //add constructor
            jf.addMethod(new JavaMethod(Modifier.PUBLIC, null, gmo.getClassDescriptor().getName() + "Abstract",
                    null, null, "super(" + (gmo.getConstructor() != null ? gmo.getConstructor() : "") + ");" + (!jf.isSuperclassIsGuixObject() ? "\ninitialize();" : ""), "Constructor"));

            //add initialization method
            jf.addMethod(new JavaMethod(Modifier.PUBLIC, "void", "initialize", null, null,
                    getMethodName(Method.BEFORE_CREATION) + "();\n" + getMethodName(Method.COMPONENTS_CREATION) + "();\n" +
                    (script.get(ScriptPart.INITIALIZERS) != null ? script.get(ScriptPart.INITIALIZERS) + "\n" : "") +
                    getMethodName(Method.BEFORE_SETTINGS) + "();\n" + getMethodName(Method.COMPONENTS_SETTINGS) + "();\n" +
                    getMethodName(Method.BEFORE_TREE) + "();\n" + getMethodName(Method.COMPONENTS_TREE) + "();\n" +
                    getMethodName(Method.BEFORE_BINDING) + "();\n" + getMethodName(Method.DATABINDING_INIT) + "();\n" + getMethodName(Method.IN_THE_END) + "();",
                    "initialization"));

            //add overridable methods
            jf.addMethod(new JavaMethod(Modifier.PUBLIC, "void", getMethodName(Method.BEFORE_CREATION), null, null, jf.isSuperclassIsGuixObject() ? "super." + getMethodName(Method.BEFORE_CREATION) + "();\n" : "", "Method called before the creation of the components"));
            jf.addMethod(new JavaMethod(Modifier.PUBLIC, "void", getMethodName(Method.BEFORE_TREE), null, null, jf.isSuperclassIsGuixObject() ? "super." + getMethodName(Method.BEFORE_TREE) + "();\n" : "", "Method called before the creation of the component tree"));
            jf.addMethod(new JavaMethod(Modifier.PUBLIC, "void", getMethodName(Method.BEFORE_SETTINGS), null, null, jf.isSuperclassIsGuixObject() ? "super." + getMethodName(Method.BEFORE_SETTINGS) + "();\n" : "", "Method called before the settings of the components"));
            jf.addMethod(new JavaMethod(Modifier.PUBLIC, "void", getMethodName(Method.BEFORE_BINDING), null, null, jf.isSuperclassIsGuixObject() ? "super." + getMethodName(Method.BEFORE_BINDING) + "();\n" : "", "Method called before the binding"));
            jf.addMethod(new JavaMethod(Modifier.PUBLIC, "void", getMethodName(Method.IN_THE_END), null, null, jf.isSuperclassIsGuixObject() ? "super." + getMethodName(Method.IN_THE_END) + "();\n" : "", "Method called in the end of the initialization"));

            //add creation and initialization methods
            jf.addMethod(new JavaMethod(Modifier.PUBLIC, "void", getMethodName(Method.COMPONENTS_CREATION), null, null,
                    (jf.isSuperclassIsGuixObject() ? "super." + getMethodName(Method.COMPONENTS_CREATION) + "();\n" : "") +
                    methodBodies.get(Method.COMPONENTS_CREATION), "Components creation"));
            jf.addMethod(new JavaMethod(Modifier.PUBLIC, "void", getMethodName(Method.COMPONENTS_SETTINGS), null, null,
                    (jf.isSuperclassIsGuixObject() ? "super." + getMethodName(Method.COMPONENTS_SETTINGS) + "();\n" : "") +
                    methodBodies.get(Method.COMPONENTS_SETTINGS), "Components settings"));
            jf.addMethod(new JavaMethod(Modifier.PUBLIC, "void", getMethodName(Method.COMPONENTS_TREE), null, null,
                    (jf.isSuperclassIsGuixObject() ? "super." + getMethodName(Method.COMPONENTS_TREE) + "();\n" : "") +
                    methodBodies.get(Method.COMPONENTS_TREE), "components layout"));

            //add databinding method
            jf.addMethod(new JavaMethod(Modifier.PUBLIC, "void", getMethodName(Method.DATABINDING_INIT), null, null,
                     jf.isSuperclassIsGuixObject() ? "super." + getMethodName(Method.DATABINDING_INIT) + "();\n" :
                         "", "initilization of databinding"));
            
            //add script methods
            for (JavaMethod m : (List<JavaMethod>) script.get(ScriptPart.METHODS)) {
                //checks if the method override a generated method
                if ((m.getName().equals(getMethodName(Method.BEFORE_BINDING)) || m.getName().equals(getMethodName(Method.BEFORE_CREATION)) || m.getName().equals(getMethodName(Method.BEFORE_TREE)) || m.getName().equals(getMethodName(Method.BEFORE_SETTINGS)) || m.getName().equals(getMethodName(Method.IN_THE_END))) && m.getReturnType() != null && m.getReturnType().equals("void") && m.getArguments() == null) {
                    int i = 0;
                    boolean methodFound = false;
                    while (i < jf.getMethods().length && !methodFound) {
                        methodFound = jf.getMethods()[i].getName().equals(m.getName()) && jf.getMethods()[i].getReturnType() != null && jf.getMethods()[i].getReturnType().equals("void") && jf.getMethods()[i].getArguments() == null;
                        i++;
                    }
                    if (methodFound) {
                        jf.getMethods()[i - 1].appendBodyCode(m.getBodyCode(), "\n");
                    }
                    else {
                        jf.addMethod(m);
                    }
                }
                else {
                    jf.addMethod(m);
                }
            }
            //add the method for the PropertChange management
            if(!jf.isSuperclassIsGuixObject()) {
                jf.addField(new JavaField(Modifier.PROTECTED, "java.beans.PropertyChangeSupport", "support", "new java.beans.PropertyChangeSupport(this)", null, null));
                jf.addMethod(new JavaMethod(Modifier.PUBLIC, "void", "addPropertyChangeListener", new JavaArgument[]{new JavaArgument("java.beans.PropertyChangeListener", "listener")},
                        null, "support.addPropertyChangeListener(listener);", null));
                jf.addMethod(new JavaMethod(Modifier.PUBLIC, "void", "addPropertyChangeListener", new JavaArgument[]{new JavaArgument("String", "property"), new JavaArgument("java.beans.PropertyChangeListener", "listener")},
                        null, "support.addPropertyChangeListener(property, listener);", null));
                jf.addMethod(new JavaMethod(Modifier.PUBLIC, "void", "removePropertyChangeListener", new JavaArgument[]{new JavaArgument("java.beans.PropertyChangeListener", "listener")},
                        null, "support.addPropertyChangeListener(listener);", null));
                jf.addMethod(new JavaMethod(Modifier.PUBLIC, "void", "removePropertyChangeListener", new JavaArgument[]{new JavaArgument("String", "property"), new JavaArgument("java.beans.PropertyChangeListener", "listener")},
                        null, "support.addPropertyChangeListener(property, listener);", null));
            }
            //Modify the setters by adding the firePropertyChange call
            for(JavaMethod method : jf.getMethods()) {
                if(method.getName().startsWith("set") && method.getName().length() > 3
                        && Character.isUpperCase(method.getName().charAt(3)) && method.getArguments() != null) {
                    method.setBodyCode(String.format(SETTER_PATTERN, method.getArguments()[0].getType(),
                            Character.toLowerCase(method.getName().charAt(3)) + method.getName().substring(4),
                            method.getBodyCode(), method.getArguments()[0].getName()));
                }
            }
        }
        catch (ClassNotFoundException eee) {
            if (log.isErrorEnabled()) {
                log.error(eee);
            }
        }
        catch (ClassCastException eee) {
            if (log.isErrorEnabled()) {
                log.error(eee);
            }
        }
        return jf;
    }

    /**
     * Browse the GuixModelObjects to add the creation, initialization and bindings methods to the file to gnerate
     *
     * @param gmo the GuixModelObject to analyze
     * @return A map containing the creation, initialization and bindings methods
     */
    private Map<Method, String> browseModelObjects(GuixModelObject gmo) throws ClassNotFoundException, ClassCastException {
        StringBuffer componentsCreation = new StringBuffer("");
        StringBuffer componentsSettings = new StringBuffer("");
        StringBuffer componentsTree = new StringBuffer("");
        GwtEventHandler geh = new GwtEventHandler();
        //the class represented by gmo
        Class clazz = null;
        //do we need to overcharge the field
        boolean createField = true;
        //id with the first letter capitalized
        String capitalizedId = (gmo.getId().length() > 0) ? Character.toUpperCase(gmo.getId().charAt(0)) + gmo.getId().substring(1) : gmo.getId();

        //if gmo represnts a table tag
        if ((gmo.getClassDescriptor().getSuperClass() != null
                && ((gmo.getClassDescriptor().getSuperClass().getPackageName() == null
                && (gmo.getClassDescriptor().getSuperClass().getName().equalsIgnoreCase("Table") || gmo.getClassDescriptor().getSuperClass().getName().equalsIgnoreCase("FlexTable")))
                || gmo.getClassDescriptor().getSuperClass().toString().equals("com.google.gwt.user.client.ui.FlexTable")))
                || ((gmo.getClassDescriptor().getPackageName() == null
                && (gmo.getClassDescriptor().getName().equalsIgnoreCase("Table") || gmo.getClassDescriptor().getName().equalsIgnoreCase("FlexTable")))
                || gmo.getClassDescriptor().toString().equals("com.google.gwt.user.client.ui.FlexTable"))) {
            //matrix of the gridbaglayout : the free zones are represented by null or false
            List<List<int[]>> layout = new ArrayList<List<int[]>>();
            StringBuffer settingsMethod = new StringBuffer();
            //Checks in the table tag if any attribute defines constraints for the children
            TableHandler tableHandler = new TableHandler(gmo);
            //the table is a FlexPanel
            clazz = tableHandler.getClassToGenerate();

            //if the gmo is the parent or if the field is inherited and has the same class, we don't need to overcharge the field
            if(gmo.getParent() == null || (jf.getInheritedField(gmo.getId()) != null && jf.getInheritedField(gmo.getId()).getType().equals(clazz.getName()))) {
                createField = false;
            }
            //if the field is inherited but doesn't hav the same class, check if the types are compatible
            if(createField && jf.getInheritedField(gmo.getId()) != null) {
                try {
                    Class inheritedClazz = gwtClassLoader.loadClass(jf.getInheritedField(gmo.getId()).getType());
                    if(!inheritedClazz.isAssignableFrom(clazz)) {
                         throw new ClassCastException();
                    }
                }
                catch(ClassNotFoundException eee) {
                    try {
                        Class inheritedClazz = ClassLoader.getSystemClassLoader().loadClass(jf.getInheritedField(gmo.getId()).getType());
                        if(!inheritedClazz.isAssignableFrom(clazz)) {
                             throw new ClassCastException();
                        }
                    }
                    catch(ClassNotFoundException eeee) {
                        throw new ClassCastException();
                    }
                }
            }
            if(createField) {
                jf.addField(new JavaField(Modifier.PROTECTED,
                            clazz.getName(), gmo.getId(), gmo.getJavadoc(), tableHandler), gmo.isJavaBean());

                componentsCreation.append(gmo.getId()).append(" = new ").append(clazz.getName())
                            .append("(").append(gmo.getConstructor() != null ? gmo.getConstructor() : "").append(");\n");
                
                if(gmo.getParent() != null && jf.getInheritedField(gmo.getId()) != null) {
                    componentsCreation.append("super.set").append(capitalizedId).append("(").append(gmo.getId()).append(");\n");
                }

            }
            
            if(jf.isSuperclassIsGuixObject() && jf.getInheritedField(gmo.getId()) != null) {
                settingsMethod.insert(0, "super." + gmo.getId() + "Settings();\n");
            }
            settingsMethod.append(gmo.getId())
                    .append(".addStyleName(\"Table\");\n");
            processAttributes(clazz, gmo, settingsMethod, geh, tableHandler);

            //add the component to its parent
            if (!ignoreTag(gmo)) {
                if(gmo.getParent().getParent() == null && jf.getInheritedField(gmo.getId()) != null) {
                    componentsTree.append(gmo.getParent().getParent() != null ? gmo.getParent().getId() : "this").append(".set").append(capitalizedId).append("(").append(gmo.getId()).append(");\n");
                }
                else if(addItemComponent(gmo)) {
                    componentsTree.append(gmo.getParent().getParent() != null ? gmo.getParent().getId() : "this").append(".addItem(").append(gmo.getId()).append(");\n");
                }
                else {
                    componentsTree.append(gmo.getParent().getParent() == null ? "this" : gmo.getParent().getId()).append(".add(").append(gmo.getId()).append(");\n");
                }
            }

            for (int r = 0; r < gmo.getChildren().size(); r++) {
                GuixModelObject row = gmo.getChildren().get(r);
                //the children of a table must be rows
                if (!row.getClassDescriptor().getName().equalsIgnoreCase("Row")) {
                    if (log.isErrorEnabled()) {
                        log.error("Table children must be Rows !");
                    }
                }
                else {
                    //Checks in the row tag if any attribute defines constraints for the children
                    RowHandler rowHandler = new RowHandler(row, tableHandler);
                    componentsTree.append(gmo.getId())
                            .append(".getRowFormatter().addStyleName(")
                            .append(rowHandler.getRow() > 0 ? rowHandler.getRow() : r).append(",\"Row\");\n");
                    int iC = 0;
                    for (int c = 0; c < row.getChildren().size(); c++) {
                        GuixModelObject cell = row.getChildren().get(c);
                        //the children of a row must be cells
                        if (!cell.getClassDescriptor().getName().equalsIgnoreCase("Cell")) {
                            if (log.isErrorEnabled()) {
                                log.error("Row children must be Cells !");
                            }
                        }
                        //the cell can contain only one component
                        else if (cell.getChildren().size() != 1) {
                            if (log.isErrorEnabled()) {
                                log.error("Cells must contain only one tag ! : " + cell.getId() + " has " + cell.getChildren().size() + "children");
                            }
                        }
                        else {
                            CellHandler cellHandler = new CellHandler(cell, rowHandler);
                            componentsTree.append(TableHandler.createCell(gmo.getId(), cellHandler, layout, iC, r));

                            //browse the children of the child of the cell
                            Map<Method, String> methodBodies = browseModelObjects(cell.getChildren().get(0));
                            componentsCreation.append(methodBodies.get(Method.COMPONENTS_CREATION));
                            componentsSettings.append(methodBodies.get(Method.COMPONENTS_SETTINGS));
                            componentsTree.append(methodBodies.get(Method.COMPONENTS_TREE));
                        }
                    }
                }
            }
            //create the table
            componentsSettings.append(gmo.getId()).append("Settings();\n");
        }
        else if ((gmo.getClassDescriptor().getSuperClass() != null && gmo.getClassDescriptor().getSuperClass().getPackageName() == null && gmo.getClassDescriptor().getSuperClass().getName().equalsIgnoreCase("Tab"))
                || (gmo.getClassDescriptor().getPackageName() == null && gmo.getClassDescriptor().getName().equalsIgnoreCase("Tab"))) {
            //if the parent has a superclass and this superclass is a TabPanel or if the parent is null or if the parent is a TabPanel
            if(((gmo.getParent().getClassDescriptor().getSuperClass() != null
                    && ((gmo.getParent().getClassDescriptor().getSuperClass().getPackageName() == null
                    && gmo.getParent().getClassDescriptor().getSuperClass().getName().equals("TabPanel"))
                    || (gmo.getParent().getClassDescriptor().getSuperClass().toString().equals("com.google.gwt.user.client.ui.TabPanel"))))
                    ||(gmo.getParent() == null
                    || (gmo.getParent().getClassDescriptor().getPackageName() == null
                    && gmo.getParent().getClassDescriptor().getName().equals("TabPanel"))
                    || (gmo.getParent().getClassDescriptor().toString().equals("com.google.gwt.user.client.ui.TabPanel"))))
                    && gmo.getChildren().size() == 1) {
                String title = null;
                for(AttributeDescriptor attribute : gmo.getAttributeDescriptors()) {
                    if(attribute.getName().equals("title")) {
                        title = attribute.getValue();
                        break;
                    }
                }

                //add the component to its parent
                componentsTree.append(gmo.getParent().getParent() != null ? gmo.getParent().getId() : "this").append(".add(")
                            .append(gmo.getChildren().get(0).getId()).append(",").append(title != null ? title : "\"" + gmo.getId() + "\"").append(");\n");

                Map<Method, String> methodBodies = browseModelObjects(gmo.getChildren().get(0));
                componentsCreation.append(methodBodies.get(Method.COMPONENTS_CREATION));
                componentsSettings.append(methodBodies.get(Method.COMPONENTS_SETTINGS));
                componentsTree.append(methodBodies.get(Method.COMPONENTS_TREE));
            }
            else if(log.isErrorEnabled()) {
                log.error("Tab tags can only be children of TabPanel tags, and can have only one child.");
            }
        }
        //if gmo represnts a menubar tag
        else if ((gmo.getClassDescriptor().getSuperClass() != null
                && ((gmo.getClassDescriptor().getSuperClass().getPackageName() == null
                && gmo.getClassDescriptor().getSuperClass().getName().equalsIgnoreCase("MenuBar"))
                || gmo.getClassDescriptor().getSuperClass().toString().equals("com.google.gwt.user.client.ui.MenuBar")))
                || ((gmo.getClassDescriptor().getPackageName() == null
                && gmo.getClassDescriptor().getName().equalsIgnoreCase("MenuBar"))
                || gmo.getClassDescriptor().toString().equals("com.google.gwt.user.client.ui.MenuBar"))) {
            StringBuffer settingsMethod = new StringBuffer();
            MenuBarHandler menuBarHandler = new MenuBarHandler();
            clazz = menuBarHandler.getClassToGenerate();
            //if the gmo is the parent or if the field is inherited and has the same class, we don't need to overcharge the field
            if(gmo.getParent() == null || (jf.getInheritedField(gmo.getId()) != null && jf.getInheritedField(gmo.getId()).getType().equals(clazz.getName()))) {
                createField = false;
            }
            //if the field is inherited but doesn't hav the same class, check if the types are compatible
            if(createField && jf.getInheritedField(gmo.getId()) != null) {
                try {
                    Class inheritedClazz = gwtClassLoader.loadClass(jf.getInheritedField(gmo.getId()).getType());
                    if(!inheritedClazz.isAssignableFrom(clazz)) {
                         throw new ClassCastException();
                    }
                }
                catch(ClassNotFoundException eee) {
                    try {
                        Class inheritedClazz = ClassLoader.getSystemClassLoader().loadClass(jf.getInheritedField(gmo.getId()).getType());
                        if(!inheritedClazz.isAssignableFrom(clazz)) {
                             throw new ClassCastException();
                        }
                    }
                    catch(ClassNotFoundException eeee) {
                        throw new ClassCastException();
                    }
                }
            }

            if(createField) {
                jf.addField(new JavaField(Modifier.PROTECTED,
                            clazz.getName(), gmo.getId(), gmo.getJavadoc(), menuBarHandler),
                            gmo.isJavaBean());
                
                componentsCreation.append(gmo.getId()).append(" = new ").append(clazz.getName())
                            .append("(").append(gmo.getConstructor() != null ? gmo.getConstructor() : "").append(");\n");

                if(gmo.getParent() != null && jf.getInheritedField(gmo.getId()) != null) {
                    componentsCreation.append("super.set").append(capitalizedId).append("(").append(gmo.getId()).append(");\n");
                }
            }
            
            if(jf.isSuperclassIsGuixObject() && jf.getInheritedField(gmo.getId()) != null) {
                settingsMethod.insert(0, "super." + gmo.getId() + "Settings();\n");
            }
            processAttributes(clazz, gmo, settingsMethod, geh, menuBarHandler);

            //add the component to its parent
            if (!ignoreTag(gmo)) {
                if(gmo.getParent().getParent() == null && jf.getInheritedField(gmo.getId()) != null) {
                    componentsTree.append(gmo.getParent().getParent() != null ? gmo.getParent().getId() : "this").append(".set").append(capitalizedId).append("(").append(gmo.getId()).append(");\n");
                }
                else if(addItemComponent(gmo)) {
                    componentsTree.append(gmo.getParent().getParent() != null ? gmo.getParent().getId() : "this").append(".addItem(").append(gmo.getId()).append(");\n");
                }
                else {
                    componentsTree.append(gmo.getParent().getParent() == null ? "this" : gmo.getParent().getId()).append(".add(").append(gmo.getId()).append(");\n");
                }
            }

            for (GuixModelObject child : gmo.getChildren()) {
                //browse the children of the child of the menubar
                Map<Method, String> methodBodies = browseMenuBar(child);
                componentsCreation.append(methodBodies.get(Method.COMPONENTS_CREATION));
                componentsSettings.append(methodBodies.get(Method.COMPONENTS_SETTINGS));
                componentsTree.append(methodBodies.get(Method.COMPONENTS_TREE));
            }
            //create the menu
            componentsSettings.append(gmo.getId()).append("Settings();\n");
        }

        //if gmo represents a "normal" tag
        else {
            //TagHandler of gmo's class
            TagHandler th = null;
            //TagHandler of gmo's first not generated superclass
            TagHandler superTh = null;
            //JavaFile representing the class of the field (if not a real class)
            JavaFile jFile = null;
            //gmo's first not generated superclass
            Class superClazz = null;
            StringBuffer settingsMethod = new StringBuffer();
            //List of gmo's generated superclasses
            List<String> generatedSuperClasses = new ArrayList<String>();

            //check if gmo's class has a TagHandler and get the class to generate instead
            if(TagManager.getGuixClassHandler(gmo.getClassDescriptor().getName()) != null) {
                th = TagManager.getGuixClassHandler(gmo.getClassDescriptor().getName());
                if(th != null) {
                    clazz = th.getClassToGenerate();
                }
            }
            //check if gmo's class has a TagHandler and get the class to generate instead
            else {
                if(gmo.getClassDescriptor().getPackageName() == null) {
                    gmo.getClassDescriptor().setPackageName(GWT_DEFAULT_PACKAGE);
                }
                if(TagManager.getGuixClassHandler(gmo.getClassDescriptor().toString()) != null) {
                    th = TagManager.getGuixClassHandler(gmo.getClassDescriptor().toString());
                    if(th != null) {
                        clazz = th.getClassToGenerate();
                    }
                }
                //check if gmo's superclass has a TagHandler and get the class to generate instead
                else if(gmo.getParent() == null && gmo.getClassDescriptor().getSuperClass() != null && TagManager.getGuixClassHandler(gmo.getClassDescriptor().getSuperClass().getName()) != null) {
                    th = TagManager.getGuixClassHandler(gmo.getClassDescriptor().getSuperClass().getName());
                    if(th != null) {
                        clazz = th.getClassToGenerate();
                    }
                }
                //check if gmo's type is a generated class
                else if(gmo.getClassDescriptor() != null && classes.get(gmo.getClassDescriptor().toString()) != null) {
                    jFile = classes.get(gmo.getClassDescriptor().toString());
                }
                //check if gmo is the first tag and its superclass is a generated class
                else if(gmo.getParent() == null && gmo.getClassDescriptor().getSuperClass() != null && classes.get(gmo.getClassDescriptor().getSuperClass().toString()) != null) {
                    jFile = classes.get(gmo.getClassDescriptor().getSuperClass().toString());
                }
                else {
                    try{
                        //get the class represented by cd
                        clazz = gwtClassLoader.loadClass(gmo.getClassDescriptor().toString());
                    }
                    catch (ClassNotFoundException eee) {
                        try {
                            clazz = ClassLoader.getSystemClassLoader().loadClass(gmo.getClassDescriptor().toString());
                        }
                        catch(ClassNotFoundException eeee) {
                            File f = new File(gwtGenerator.getSrcDir(), gmo.getClassDescriptor().toString().replace('.', File.separatorChar) + ".java");
                            try {
                                Reader isr = new InputStreamReader(new FileInputStream(f));
                                jFile = JavaFileParser.parseJavaFile(gmo.getClassDescriptor().toString(), isr);
                                gwtGenerator.getPropertyChangeListenerDependencies().add(jFile);
                            }
                            catch (FileNotFoundException eeeee) {
                                log.error("error : " + gmo.getClassDescriptor().toString());
                                log.error(eeeee);
                                eeeee.printStackTrace();
                                throw new ClassNotFoundException();
                            }
                        }
                    }
                }
            }
            if(clazz != null) {
                superClazz = clazz.getSuperclass();
            }
            else {
                //get the first not generated superclass of gmo, the corresponding taghandler and record all the generated superclasses of gmo
                JavaFile jFile2 = jFile;
                while(superClazz == null && jFile2 != null) {
                    try{
                        //get the class represented by cd
                        superClazz = gwtClassLoader.loadClass(jFile2.getSuperClass());
                        superTh = TagManager.getGuixClassHandler(superClazz.getName());
                    }
                    catch (ClassNotFoundException eee) {
                        try {
                            superClazz = ClassLoader.getSystemClassLoader().loadClass(jFile2.getSuperClass());
                            superTh = TagManager.getGuixClassHandler(superClazz.getName());
                        }
                        catch(ClassNotFoundException eeee) {
                            generatedSuperClasses.add(jFile2.getSuperClass());
                            jFile2 = classes.get(jFile2.getSuperClass().endsWith("Abstract") ? jFile2.getSuperClass().substring(0, jFile2.getSuperClass().length() - 8)
                                    : jFile2.getSuperClass());
                        }
                    }
                }
            }
            //check if we need to generate the field, and check if its type is compatible with an eventual inherited field
            if(gmo.getParent() == null) {
                createField = false;
            }
            else if(jf.getInheritedField(gmo.getId()) != null) {
                if((clazz != null && jf.getInheritedField(gmo.getId()).getType().equals(clazz.getName()))
                        || (jFile != null && jf.getInheritedField(gmo.getId()).getType().equals(jFile.getPackageName() + "." + jFile.getClassName()))) {
                    createField = false;
                }
                else {
                    Class inheritedClazz = null;
                    try {
                        inheritedClazz = gwtClassLoader.loadClass(jf.getInheritedField(gmo.getId()).getType());
                        if(clazz != null) {
                            if(!inheritedClazz.isAssignableFrom(clazz)) {
                                throw new ClassCastException();
                            }
                        }
                        else if(jFile != null && superClazz != null) {
                            if(!inheritedClazz.isAssignableFrom(superClazz)) {
                                throw new ClassCastException();
                            }
                        }
                        else {
                            throw new ClassCastException();
                        }
                    }
                    catch(ClassNotFoundException eee) {
                        try {
                            inheritedClazz = ClassLoader.getSystemClassLoader().loadClass(jf.getInheritedField(gmo.getId()).getType());
                            if(clazz != null) {
                                if(!inheritedClazz.isAssignableFrom(clazz)) {
                                    throw new ClassCastException();
                                }
                            }
                            else if(jFile != null && superClazz != null) {
                                if(!inheritedClazz.isAssignableFrom(superClazz)) {
                                    throw new ClassCastException();
                                }
                            }
                            else {
                                throw new ClassCastException();
                            }
                        }
                        catch(ClassNotFoundException eeee) {
                            if(!generatedSuperClasses.contains(jf.getInheritedField(gmo.getId()).getType())) {
                                throw  new ClassCastException();
                            }
                        }
                    }
                }
            }
            if(createField) {
                //if gmo has a TagHandler, add a field to the JavaFile with this TagHandler (used for the databinding generation)
                if(th != null && th.getClassToGenerate() != null) {
                    jf.addField(new JavaField(Modifier.PROTECTED,
                            clazz.getName(), gmo.getId(), gmo.getJavadoc(), th), gmo.isJavaBean());
                }
                else {
                    jf.addField(new JavaField(Modifier.PROTECTED,
                            (classes != null && classes.containsKey(gmo.getClassDescriptor().toString())) ? gmo.getClassDescriptor().toString() + "Abstract" 
                            : gmo.getClassDescriptor().toString(), gmo.getId(), gmo.getJavadoc(), superTh), gmo.isJavaBean());
                }

                componentsCreation.append(gmo.getId()).append(" = new ");
                if(classes != null && classes.containsKey(gmo.getClassDescriptor().toString())) {
                    componentsCreation.append(gmo.getClassDescriptor().toString() + "Impl");
                }
                else if(clazz != null) {
                    componentsCreation.append(clazz.getName());
                }
                else {
                    componentsCreation.append(gmo.getClassDescriptor().toString());
                }
                componentsCreation.append("(").append(gmo.getConstructor() != null ? gmo.getConstructor() : "").append(");\n");
                
            }
                
            //if gmo is not the root ModelObject
            if (gmo.getParent() != null) {
                if(jf.isSuperclassIsGuixObject() && jf.getInheritedField(gmo.getId()) != null) {
                    settingsMethod.insert(0, "super." + gmo.getId() + "Settings();\n");
                }
                if(createField) {
                    if(jf.getInheritedField(gmo.getId()) != null) {
                        componentsCreation.append("super.set").append(capitalizedId).append("(").append(gmo.getId()).append(");\n");
                    }
                }
                componentsSettings.append(gmo.getId()).append("Settings();\n");

                if(clazz != null) {
                    processAttributes(clazz, gmo, settingsMethod, geh, th);
                }
                else {
                    processAttributes(jFile, superClazz, gmo, settingsMethod, geh, superTh);
                }

                //if gmo is the child of a cell
                if (!ignoreTag(gmo)) {
                    if((jFile != null && (jFile.getField(gmo.getId()) != null || jFile.getInheritedField(gmo.getId()) != null))
                            || (gmo.getParent().getParent() == null && jf.getInheritedField(gmo.getId()) != null)) {
                        componentsTree.append(gmo.getParent().getParent() != null ? gmo.getParent().getId() : "this").append(".set").append(capitalizedId).append("(").append(gmo.getId()).append(");\n");
                    }
                    else if(addItemComponent(gmo)) {
                        componentsTree.append(gmo.getParent().getParent() != null ? gmo.getParent().getId() : "this").append(".addItem(").append(gmo.getId()).append(");\n");
                    }
                    //add the component to its parent
                    else if(superClazz != null && com.google.gwt.user.client.ui.Widget.class.isAssignableFrom(superClazz)) {
                        componentsTree.append(gmo.getParent().getParent() != null ? gmo.getParent().getId() : "this").append(".add(").append(gmo.getId()).append(");\n");
                    }
                }
            }
            //if gmo represents the first tag
            else {
                //set superclass and interface
                if(th != null) {
                    jf.setSuperClass(th.getClassToGenerate().getName());
                }
                else {
                    jf.setSuperClass(classes.containsKey(gmo.getClassDescriptor().getSuperClass().toString()) ? gmo.getClassDescriptor().getSuperClass().toString() + "Abstract" : gmo.getClassDescriptor().getSuperClass().toString());
                }
                jf.addInterface(gmo.getClassDescriptor().getName());

                //get the inherited methods from the not generated superclass
                if(clazz != null) {
                    jf.setSuperclassIsGuixObject(false);
                    for(java.lang.reflect.Method m : clazz.getMethods()) {
                        if(Modifier.isPublic(m.getModifiers()) || Modifier.isProtected(m.getModifiers())) {
                            JavaArgument[] args = new JavaArgument[m.getParameterTypes().length];
                            for(int i = 0 ; i < m.getParameterTypes().length ; i++) {
                                args[i] = new JavaArgument(m.getParameterTypes()[i].getName(), "arg" + i);
                            }
                            jf.addInheritedMethod(new JavaMethod(m.getModifiers(), m.getReturnType().getName(), m.getName(), args, null, "", null));
                        }
                    }
                    processAttributes(clazz, gmo, settingsMethod, geh, th);
                }
                //get the inherited methods and fiel from the generated superclass
                else if(jFile != null) {
                    jf.setSuperclassIsGuixObject(true);
                    for(JavaMethod m : jFile.getInheritedMethods()) {
                        if(Modifier.isPublic(m.getModifiers()) || Modifier.isProtected(m.getModifiers())) {
                            jf.addInheritedMethod(m);
                        }
                    }
                    for(JavaMethod m : jFile.getMethods()) {
                        if(Modifier.isPublic(m.getModifiers()) || Modifier.isProtected(m.getModifiers())) {
                            jf.addInheritedMethod(m);
                        }
                    }
                    for(JavaField f : jFile.getInheritedFields()) {
                        if(Modifier.isPublic(f.getModifiers()) || Modifier.isProtected(f.getModifiers())) {
                            jf.addInheritedField(f);
                        }
                    }
                    for(JavaField f : jFile.getFields()) {
                        if(Modifier.isPublic(f.getModifiers()) || Modifier.isProtected(f.getModifiers())) {
                            jf.addInheritedField(f);
                        }
                    }
                    processAttributes(jFile, superClazz, gmo, settingsMethod, geh, superTh != null ? superTh: th);
                }
                else {
                    log.error("Allo Houston, on a un probleme !");
                }
                componentsSettings.append(gmo.getId()).append("Settings();\n");
            }
            for (GuixModelObject child : gmo.getChildren()) {
                Map<Method, String> methodBodies = browseModelObjects(child);
                componentsCreation.append(methodBodies.get(Method.COMPONENTS_CREATION));
                componentsSettings.append(methodBodies.get(Method.COMPONENTS_SETTINGS));
                componentsTree.append(methodBodies.get(Method.COMPONENTS_TREE));
            }
            
        }
        Map<Method, String> result = new HashMap<Method, String>();
        result.put(Method.COMPONENTS_CREATION, componentsCreation.toString());
        result.put(Method.COMPONENTS_SETTINGS, componentsSettings.toString());
        result.put(Method.COMPONENTS_TREE, componentsTree.toString());
        return result;
    }

    /**
     *
     * @return true if the tag does not represent a real object but is just useful to ease the creation of the UI
     */
    private boolean ignoreTag(GuixModelObject gmo) {
        return (gmo.getParent() != null && (gmo.getParent().getClassDescriptor().getPackageName() == null
                && (gmo.getParent().getClassDescriptor().getName().equalsIgnoreCase("Cell") || gmo.getParent().getClassDescriptor().getName().equalsIgnoreCase("Tab")))
                        && !(gmo.getParent().getClassDescriptor().getPackageName() == null && gmo.getParent().getClassDescriptor().getName().equalsIgnoreCase("VerticalSplitPanel"))
                        && !gmo.getParent().getClassDescriptor().toString().equals("com.google.gwt.user.client.ui.VerticalSplitPanel")
                        && !(gmo.getParent().getClassDescriptor().getPackageName() == null && gmo.getParent().getClassDescriptor().getName().equalsIgnoreCase("HorizontalSplitPanel"))
                        && !gmo.getParent().getClassDescriptor().toString().equals("com.google.gwt.user.client.ui.HorizontalSplitPanel"));
    }

    /**
     *
     * @return true if the component parent needs the "addItem" method to add the child
     */
    private boolean addItemComponent(GuixModelObject gmo) {
        return (gmo.getParent() != null && (gmo.getParent().getClassDescriptor().getPackageName() == null && gmo.getParent().getClassDescriptor().getName().equalsIgnoreCase("Tree"))
                            || gmo.getParent().getClassDescriptor().toString().equals("com.google.gwt.user.client.ui.Tree")
                        || (gmo.getParent().getClassDescriptor().getPackageName() == null && gmo.getParent().getClassDescriptor().getName().equalsIgnoreCase("TreeNode"))
                        || gmo.getParent().getClassDescriptor().toString().equals("com.google.gwt.user.client.ui.TreeItem")
                        || (gmo.getParent().getClassDescriptor().getPackageName() == null && gmo.getParent().getClassDescriptor().getName().equalsIgnoreCase("ComboBox"))
                        || gmo.getParent().getClassDescriptor().toString().equals("com.google.gwt.user.client.ui.ListBox"));
    }

    /**
     * Generates the code to set the attributes of the gmo
     *
     * @param clazz class represented by gmo
     * @param gmo the GuixModelObject which contains the attribute
     * @param creationMethod the StringBuffer containing the code of the creation method for the object represented by gmo (null if gmo represents the first tag)
     * @param componentsTree the StringBuffer containing the code of the settings of the attribute of the class (null if gmo does not represent the first tag)
     * @param geh the event handler
     * @param th gmo's TagHandler
     */
    private void processAttributes(Class clazz, GuixModelObject gmo, StringBuffer settingMethod, GwtEventHandler geh, TagHandler th) {
        //browses the attributes
        for (AttributeDescriptor attr : gmo.getAttributeDescriptors()) {
            boolean addQuote = false;
            if(th != null && th.getAttrToGenerate(attr.getName()) != null) {
                attr.setName(th.getAttrToGenerate(attr.getName()));
            }
            //the attribute name with the first letter capitalized
            String capitalizedAttribute = (attr.getName().length() > 0) ? Character.toUpperCase(attr.getName().charAt(0)) + attr.getName().substring(1) : attr.getName();
            //is the attribute a "real" attribute ? false if it represents an event or a constraint
            boolean realAttribute = true;

            try {
                if (gmo.getParent() != null || (!attr.getName().equals("title") && !attr.getName().equals("theme"))) {
                    //checks if the attribute is an event
                    if (geh != null && attr.getName().startsWith("on") && Character.isUpperCase(attr.getName().charAt(2))) {
                        realAttribute = !geh.addEvent(clazz, attr.getName(), attr.getValue());
                    }

                    if (realAttribute) {
                        //get the code of the binding (removes braces mostly)
                        String binding = BindingUtils.processDataBindings(attr.getValue());
                        java.lang.reflect.Method[] methods = clazz.getMethods();
                        int m = 0;
                        //checks if the setter for this attribute exists
                        while (m < methods.length && !methods[m].getName().equals("set" + capitalizedAttribute)) {
                            m++;
                        }
                        //if yes
                        if (m < methods.length) {
                            //if the attribute binds the value of another component attribute
                            if (binding != null) {
                                if(!bindings2Generate.containsKey(gmo.getId())) {
                                    bindings2Generate.put(gmo.getId(), new HashMap<String, String>());
                                }
                                bindings2Generate.get(gmo.getId()).put(attr.getName(), binding);
                            }
                            else {
                                //checks if the parameter of the setter is a String
                                addQuote = methods[m].getParameterTypes()[0].equals(String.class);
                                //generates the code to set the attribute to object
                                if (settingMethod != null) {
                                    settingMethod.append(gmo.getParent() != null ? gmo.getId() : "this").append(".set").append(capitalizedAttribute).append("(").append(addQuote ? "\"" : "").append(attr.getValue()).append(addQuote ? "\"" : "").append(");\n");
                                }
                            }
                        }
                        else if (log.isWarnEnabled()) {
                            log.warn(attr.getName() + " cannot be set.");
                        }
                    }
                }
            }
            catch (IntrospectionException ex) {
                if (log.isErrorEnabled()) {
                    log.error(ex);
                }
            }
        }

        if (gmo.getId() != null && settingMethod != null) {
            //generates the code for the events
            settingMethod.append(geh.generate(gmo.getId()));
            //add to the file the creation method for this object
            jf.addMethod(new JavaMethod(Modifier.PROTECTED, "void", gmo.getId() + "Settings", null, null, settingMethod.toString(), gmo.getId() + " settings"));
        }
    }

    /**
     * Generates the code to set the attributes of the gmo when gmo's class is an uncompiled java file
     *
     * @param jFile the JavaFile describing th uncompiled java file
     * @param gmo the GuixModelObject instance of the clas represented by jFile
     * @param creationMethod the StringBuffer containing the creation method
     */
    private void processAttributes(JavaFile jFile, Class superClazz, GuixModelObject gmo, StringBuffer settingsMethod, GwtEventHandler geh, TagHandler th) {
        //processCSSAttributes(gmo, seh, clazz);
        //browses the attributes
        for (AttributeDescriptor attr : gmo.getAttributeDescriptors()) {
            boolean addQuote = false;
            if(th != null && th.getAttrToGenerate(attr.getName()) != null) {
                attr.setName(th.getAttrToGenerate(attr.getName()));
            }
            //the attribute name with the first letter capitalized
            String capitalizedAttribute = (attr.getName().length() > 0) ? Character.toUpperCase(attr.getName().charAt(0)) + attr.getName().substring(1) : attr.getName();
            //is the attribute a "real" attribute ? false if it represents an event or a constraint
            boolean realAttribute = true;

            try {
                if (gmo.getParent() != null || (!attr.getName().equals("title") && !attr.getName().equals("theme"))) {
                    //checks if the attribute is an event
                    if (geh != null && attr.getName().startsWith("on") && Character.isUpperCase(attr.getName().charAt(2))) {
                        realAttribute = !geh.addEvent(superClazz, attr.getName(), attr.getValue());
                    }

                    if (realAttribute) {
                        //get the code of the binding (removes braces mostly)
                        String binding = BindingUtils.processDataBindings(attr.getValue());

                        JavaMethod method = null;
                        for(JavaMethod m : jFile.getMethods()) {
                            if(m.getName().equals("set" + capitalizedAttribute)) {
                                method = m;
                                break;
                            }
                        }
                        if(method == null) {
                            for(JavaMethod m : jFile.getInheritedMethods()) {
                                if(m.getName().equals("set" + capitalizedAttribute)) {
                                    method = m;
                                    break;
                                }
                            }
                        }
                        if(method != null) {
                            //if the attribute binds the value of another component attribute
                            if (binding != null) {
                                if(!bindings2Generate.containsKey(gmo.getId())) {
                                    bindings2Generate.put(gmo.getId(), new HashMap<String, String>());
                                }
                                bindings2Generate.get(gmo.getId()).put(attr.getName(), binding);
                            }
                            else {
                                //checks if the parameter of the setter is a String
                                addQuote = method.getArguments()[0].getType().equals(String.class.getName());
                                //generates the code to set the attribute to object
                                settingsMethod.append(gmo.getParent() != null ? gmo.getId() : "this").append(".set").append(capitalizedAttribute).append("(").append(addQuote ? "\"" : "").append(attr.getValue()).append(addQuote ? "\"" : "").append(");\n");
                            }
                        }
                        else if (log.isWarnEnabled()) {
                            log.warn(attr.getName() + " cannot be set.");
                        }
                    }
                }
            }
            catch (IntrospectionException ex) {
                if (log.isErrorEnabled()) {
                    log.error(ex);
                }
            }
        }

        if(gmo.getId() != null && settingsMethod != null) {
           //add to the file the creation method for this object
            jf.addMethod(new JavaMethod(Modifier.PROTECTED, "void", gmo.getId() + "Settings", null, null, settingsMethod.toString(), gmo.getId() + " settings"));
        }
    }
    
    /**
     * Browse the MenuBar children and generate the code
     * @param gmo the GuixModelObject representing the MenuBar
     * @return A map containing the creation, initialization and bindings methods
     */
    private Map<Method, String> browseMenuBar(GuixModelObject gmo) {
        StringBuffer componentsCreation = new StringBuffer("");
        StringBuffer componentsSettings = new StringBuffer("");
        StringBuffer componentsTree = new StringBuffer("");
        StringBuffer creationMethod = new StringBuffer("");
        String text = null, cmd = null;
        //gmo's id with the first letter capitalized
        String capitalizedId = (gmo.getId().length() > 0) ? Character.toUpperCase(gmo.getId().charAt(0)) + gmo.getId().substring(1) : gmo.getId();

        if(gmo.getClassDescriptor().getPackageName() == null && gmo.getClassDescriptor().getName().equals("Menu")) {
            TagHandler th = new MenuHandler();
            Class clazz = th.getClassToGenerate();
            jf.addField(new JavaField(Modifier.PROTECTED,
                            clazz.getName(), gmo.getId(), gmo.getJavadoc(), th), gmo.isJavaBean());

            componentsCreation.append(gmo.getId()).append(" = new ").append(clazz.getName())
                            .append("(").append(gmo.getConstructor() != null ? gmo.getConstructor() : "").append(");\n");

                if(gmo.getParent() != null && jf.getInheritedField(gmo.getId()) != null) {
                    componentsCreation.append("super.set").append(capitalizedId).append("(").append(gmo.getId()).append(");\n");
                }

            for(AttributeDescriptor attr : gmo.getAttributeDescriptors()) {
                if(attr.getName().equalsIgnoreCase("text")) {
                    text = attr.getValue();
                }
                else {
                    if(th.getAttrToGenerate(attr.getName()) != null) {
                        attr.setName(th.getAttrToGenerate(attr.getName()));
                    }
                    //the attribute name with the first letter capitalized
                    String capitalizedAttribute = (attr.getName().length() > 0) ? Character.toUpperCase(attr.getName().charAt(0)) + attr.getName().substring(1) : attr.getName();
            
                    java.lang.reflect.Method[] methods = MenuBar.class.getMethods();
                    int m = 0;
                    //checks if the setter for this attribute exists
                    while (m < methods.length && !methods[m].getName().equals("set" + capitalizedAttribute)) {
                        m++;
                    }
                    //if yes
                    if (m < methods.length) {
                        //checks if the parameter of the setter is a String
                        boolean addQuote = methods[m].getParameterTypes()[0].equals(String.class);
                        //generates the code to set the attribute to object
                        creationMethod.append(gmo.getId()).append(".set").append(capitalizedAttribute).append("(").append(addQuote ? "\"" : "").append(attr.getValue()).append(addQuote ? "\"" : "").append(");\n");
                    }
                    else if (log.isErrorEnabled()) {
                        log.error(attr.getName() + " cannot be set.");
                    }
                }
            }
            componentsTree.append(gmo.getParent().getId()).append(".addItem(\"").append(text != null ? text : gmo.getId()).append("\",").append(gmo.getId()).append(");\n");

            for (GuixModelObject child : gmo.getChildren()) {
                Map<Method, String> methodBodies = browseMenuBar(child);
                componentsCreation.append(methodBodies.get(Method.COMPONENTS_CREATION));
                componentsSettings.append(methodBodies.get(Method.COMPONENTS_SETTINGS));
                componentsTree.append(methodBodies.get(Method.COMPONENTS_TREE));
            }
        }
        else if(gmo.getClassDescriptor().getPackageName() == null && gmo.getClassDescriptor().getName().equals("MenuItem")) {
            TagHandler th = new MenuItemHandler();
            for(AttributeDescriptor attr : gmo.getAttributeDescriptors()) {
                if(attr.getName().equalsIgnoreCase("text")) {
                    text = attr.getValue();
                }
                if(attr.getName().equalsIgnoreCase("action")) {
                    cmd = attr.getValue();
                }
                else {
                    if(th.getAttrToGenerate(attr.getName()) != null) {
                        attr.setName(th.getAttrToGenerate(attr.getName()));
                    }
                    else if (log.isErrorEnabled()) {
                        log.error(attr.getName() + " cannot be set.");
                    }
                }
            }
            componentsTree.append(gmo.getParent().getId()).append(".addItem(\"").append(text != null ? text : gmo.getId()).append("\",").append(cmd).append(");\n");
        }
        else {
            if(log.isErrorEnabled()) {
                log.error("The menu can only contain Menu or MenuItem tags.");
            }
        }
        
        Map<Method, String> result = new HashMap<Method, String>();
        result.put(Method.COMPONENTS_CREATION, componentsCreation.toString());
        result.put(Method.COMPONENTS_SETTINGS, componentsSettings.toString());
        result.put(Method.COMPONENTS_TREE, componentsTree.toString());
        return result;
    }

    public Map<String, Map<String, String>> getBindings2Generate() {
        return bindings2Generate;
    }

}

