package org.nuiton.eugene.java;

/*
 * #%L
 * EUGene :: Java templates
 * %%
 * Copyright (C) 2012 - 2015 CodeLutin
 * %%
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Lesser General Public License as
 * published by the Free Software Foundation, either version 3 of the
 * License, or (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Lesser Public License for more details.
 * 
 * You should have received a copy of the GNU General Lesser Public
 * License along with this program.  If not, see
 * <http://www.gnu.org/licenses/lgpl-3.0.html>.
 * #L%
 */




import com.google.common.base.Function;
import com.google.common.base.Predicate;
import com.google.common.collect.ImmutableMap;
import com.google.common.collect.ImmutableSet;
import com.google.common.collect.Iterables;
import com.google.common.collect.Maps;
import org.apache.commons.collections.CollectionUtils;
import org.apache.commons.lang3.StringUtils;
import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.nuiton.eugene.models.object.ObjectModel;
import org.nuiton.eugene.models.object.ObjectModelAttribute;
import org.nuiton.eugene.models.object.ObjectModelClass;
import org.nuiton.eugene.models.object.ObjectModelInterface;
import org.nuiton.eugene.models.object.ObjectModelJavaModifier;
import org.nuiton.eugene.models.object.ObjectModelOperation;
import org.nuiton.eugene.models.object.ObjectModelPackage;
import org.nuiton.util.beans.Binder;
import org.nuiton.util.beans.BinderFactory;

import java.util.ArrayList;
import java.util.Collection;
import java.util.Collections;
import java.util.List;
import java.util.Objects;

/**
 * Generates a java bean and a utility class around it. This transformer acts like {@link SimpleJavaBeanTransformer}
 * but with no interface generation (plus there is not factory generation at all).
 *
 * For example:
 * <pre>
 *     AbstractBoat
 *     Boat (extends AbstractBoat)
 *     AbstractBoats
 *     Boats (extends AbstractBoats)
 * </pre>
 *
 * @author Tony Chemit - chemit@codelutin.com
 * @plexus.component role="org.nuiton.eugene.Template" role-hint="org.nuiton.eugene.java.SimpleJavaBeanWithNoInterfaceTransformer"
 * @since 3.0
 */
public class SimpleJavaBeanWithNoInterfaceTransformer extends AbstractJavaBeanTransformer {

    /** Logger. */
    private static final Log log = LogFactory.getLog(SimpleJavaBeanWithNoInterfaceTransformer.class);

    ImmutableMap<ObjectModelClass, String> beanNameTranslation;

    ImmutableMap<ObjectModelClass, String> beanDefaultsNameTranslation;

    ImmutableSet<ObjectModelClass> beanClasses;

    ImmutableSet<ObjectModelClass> beanDefaultClasses;

    @Override
    public void transformFromModel(ObjectModel model) {
        super.transformFromModel(model);

        ImmutableMap.Builder<ObjectModelClass, String> beanNameTranslationBuilder = new ImmutableMap.Builder<>();
        ImmutableMap.Builder<ObjectModelClass, String> beanDefaultsNameTranslationBuilder = new ImmutableMap.Builder<>();
        ImmutableSet.Builder<ObjectModelClass> beanClassesBuilder = new ImmutableSet.Builder<>();
        ImmutableSet.Builder<ObjectModelClass> beanDefaultClassesBuilder = new ImmutableSet.Builder<>();

        JavaTemplatesTagValues javaTemplatesTagValues = getJavaTemplatesTagValues();
        JavaTemplatesStereoTypes javaTemplatesStereoTypes = getJavaTemplatesStereoTypes();

        for (ObjectModelClass aClass : model.getClasses()) {

            ObjectModelPackage aPackage = model.getPackage(aClass.getPackageName());
            if (JavaTemplatesStereoTypes.hasBeanStereotype(aClass, aPackage)) {

                beanClassesBuilder.add(aClass);

                String classNamePrefix = javaTemplatesTagValues.getSimpleBeanWithNoInterfaceClassNamePrefixTagValue(aClass, aPackage, model);
                String classNameSuffix = javaTemplatesTagValues.getSimpleBeanWithNoInterfaceClassNameSuffixTagValue(aClass, aPackage, model);

                String generateName = generateName(classNamePrefix, aClass.getName(), classNameSuffix);
                beanNameTranslationBuilder.put(aClass, generateName);

                boolean canGenerateDefaults = !javaTemplatesStereoTypes.isSimpleBeanWithNoInterfaceSkipGenerateDefaults(aClass, aPackage, model);

                if (canGenerateDefaults) {

                    beanDefaultClassesBuilder.add(aClass);

                    String classDefaultsNamePrefix = javaTemplatesTagValues.getSimpleBeanWithNoInterfaceDefaultsClassNamePrefixTagValue(aClass, aPackage, model);
                    String classDefaultsNameSuffix = javaTemplatesTagValues.getSimpleBeanWithNoInterfaceDefaultsClassNameSuffixTagValue(aClass, aPackage, model);

                    String generateDefaultsName = generateName(classDefaultsNamePrefix, aClass.getName(), classDefaultsNameSuffix);
                    beanDefaultsNameTranslationBuilder.put(aClass, generateDefaultsName);

                }

            }
        }

        beanClasses = beanClassesBuilder.build();
        beanDefaultClasses = beanDefaultClassesBuilder.build();
        beanNameTranslation = beanNameTranslationBuilder.build();
        beanDefaultsNameTranslation = beanDefaultsNameTranslationBuilder.build();

        ImmutableMap<String, ObjectModelClass> beanClassesByFqn = Maps.uniqueIndex(beanClasses, new Function<ObjectModelClass, String>() {

            @Override
            public String apply(ObjectModelClass input) {
                return input.getQualifiedName();
            }
        });
        ArrayList<String> beanClassesFqn = new ArrayList<>(beanClassesByFqn.keySet());
        Collections.sort(beanClassesFqn);

        String defaultPackageName = getDefaultPackageName();

        String modelBeanInitializeClassName = model.getName() + "ModelInitializer";
        boolean generateModelInitializer = !isInClassPath(defaultPackageName + "." + modelBeanInitializeClassName);
        if (generateModelInitializer) {

            ObjectModelInterface anInterface = createInterface(modelBeanInitializeClassName, defaultPackageName);

            addOperation(anInterface, "start", "void");
            addOperation(anInterface, "end", "void");

            for (String fqn : beanClassesFqn) {
                ObjectModelClass beanClass = beanClassesByFqn.get(fqn);
                String beanName = beanNameTranslation.get(beanClass);
                addImport(anInterface, beanName);
                addOperation(anInterface, "init" + beanName, "void");

            }
        }

        String modelInitializerRunnerClassName = model.getName() + "ModelInitializerRunner";
        boolean generateInitializerRunnerClassName = !isInClassPath(defaultPackageName + "." + modelInitializerRunnerClassName);
        if (generateInitializerRunnerClassName) {

            ObjectModelClass aClass = createClass(modelInitializerRunnerClassName, defaultPackageName);

            StringBuilder bodyBuilder = new StringBuilder();
            bodyBuilder.append(""
+"\n"
+"        initializer.start();"
            );
            for (String fqn : beanClassesFqn) {
                ObjectModelClass beanClass = beanClassesByFqn.get(fqn);
                String beanName = beanNameTranslation.get(beanClass);
                addImport(aClass, beanName);
                bodyBuilder.append(""
+"\n"
+"        initializer.init"+beanName+"();"
                );

            }

            bodyBuilder.append(""
+"\n"
+"        initializer.end();"
            );
            ObjectModelOperation operation = addOperation(aClass, "init", "void", ObjectModelJavaModifier.STATIC);
            addParameter(operation, modelBeanInitializeClassName, "initializer");
            setOperationBody(operation, bodyBuilder.toString());
        }

    }

    @Override
    public void transformFromClass(ObjectModelClass input) {

        ObjectModelPackage aPackage = getPackage(input);

        if (beanClasses.contains(input)) {

            String prefix = getConstantPrefix(input);
            setConstantPrefix(prefix);

            String className = beanNameTranslation.get(input);
            String abstractClassName = "Abstract" + className;

            boolean generateClass = canGenerateClassWithMethods(aPackage, input, className)
                                    && notFoundInClassPath(input, className);
            if (generateClass) {
                generateBeanClass(input, className, abstractClassName);
            }

            boolean generateAbstractClass = canGenerateAbstractClass(aPackage, input, abstractClassName);
            if (generateAbstractClass) {
                generateAbstractBeanClass(aPackage, input, abstractClassName);
            }

            boolean generateDefaults = beanDefaultClasses.contains(input);
            if (generateDefaults) {

                String defaultClassName = beanDefaultsNameTranslation.get(input);
                String abstractDefaultClassName = "Abstract" + defaultClassName;

                if (notFoundInClassPath(input, defaultClassName)) {

                    generateBeanDefaults(input, abstractDefaultClassName, defaultClassName);
                }

                if (canGenerateAbstractClass(aPackage, input, abstractDefaultClassName)) {

                    generateAbstractBeanDefaults(aPackage, input, className, abstractDefaultClassName);
                }

            }

        }

    }

    protected ObjectModelClass generateBeanClass(ObjectModelClass input,
                                                 String className,
                                                 String abstractClassName) {

        ObjectModelClass output;

        if (input.isAbstract()) {
            output = createAbstractClass(className, input.getPackageName());
        } else {
            output = createClass(className, input.getPackageName());
        }

        setSuperClass(output, abstractClassName);

        if (log.isDebugEnabled()) {
            log.debug("will generate " + output.getQualifiedName());
        }

        addSerializable(input, output, true);

        return output;
    }

    protected ObjectModelClass generateAbstractBeanClass(ObjectModelPackage aPackage,
                                                         ObjectModelClass input,
                                                         String className) {

        String superClass = null;

        // test if a super class has bean stereotype
        boolean superClassIsBean = false;
        Collection<ObjectModelClass> superclasses = input.getSuperclasses();
        if (CollectionUtils.isNotEmpty(superclasses)) {
            for (ObjectModelClass superclass : superclasses) {
                superClassIsBean = beanClasses.contains(superclass);
                if (superClassIsBean) {
                    superClass = superclass.getPackageName() + "." + beanNameTranslation.get(superclass);
                    break;
                }
                superClass = superclass.getQualifiedName();
            }
        }

        if (!superClassIsBean) {

            // try to find a super class by tag-value
            superClass = getJavaTemplatesTagValues().getSimpleBeanWithNoInterfaceSuperClassTagValue(input, aPackage, model);
            if (superClass != null) {

                // will act as if super class is a bean
                superClassIsBean = true;
            }
        }

        ObjectModelClass output;

        output = createAbstractClass(className, input.getPackageName());

        if (superClass != null) {
            setSuperClass(output, superClass);
        }
        if (log.isDebugEnabled()) {
            log.debug("will generate " + output.getQualifiedName());
        }

        boolean serializableFound;

        serializableFound = addInterfaces(input, output, null);

        generateI18nBlockAndConstants(aPackage, input, output);

        addSerializable(input, output, serializableFound || superClassIsBean);

        // Get available properties
        List<ObjectModelAttribute> properties = getProperties(input);

        boolean usePCS = !getJavaTemplatesStereoTypes().isSkipGeneratePropertyChangeSupport(input, aPackage, model);
        boolean generateBooleanGetMethods = getEugeneTagValues().isGenerateBooleanGetMethods(input, aPackage, model);
        boolean generateNotEmptyCollections = !getJavaTemplatesStereoTypes().isSkipGenerateNotEmptyCollections(input, aPackage, model);

        // Add properties field + javabean methods
        for (ObjectModelAttribute attr : properties) {

            createProperty(output,
                           attr,
                           usePCS,
                           generateBooleanGetMethods,
                           generateNotEmptyCollections);
        }

        if (!superClassIsBean) {
            addDefaultMethodForNoneBeanSuperClass(output, usePCS, properties);
        }
        return output;
    }

    protected void generateBeanDefaults(ObjectModelClass aClass, String abstractClassName, String defaultClassName) {

        String packageName = aClass.getPackageName();

        ObjectModelClass output = createClass(defaultClassName, packageName);
        setSuperClass(output, packageName + "." + abstractClassName);
        if (log.isDebugEnabled()) {
            log.debug("will generate " + output.getQualifiedName());
        }

    }

    protected void generateAbstractBeanDefaults(ObjectModelPackage aPackage,
                                                ObjectModelClass aClass,
                                                String typeName,
                                                String abstractClassName) {

        ObjectModelClass output = createAbstractClass(abstractClassName, aPackage.getName());
        String superClassName = getAbstractDefaultsSuperClassName(aPackage, aClass);

        if (StringUtils.isNotBlank(superClassName)) {
            setSuperClass(output, superClassName);
        }

        if (log.isDebugEnabled()) {
            log.debug("will generate " + output.getQualifiedName());
        }

        addImport(output, Binder.class);
        addImport(output, BinderFactory.class);

        ObjectModelOperation operation = addOperation(
                output,
                "typeOf" + typeName,
                "<BeanType extends " + typeName + "> Class<BeanType>",
                ObjectModelJavaModifier.STATIC,
                ObjectModelJavaModifier.PUBLIC
        );
        setOperationBody(operation, ""
    +"\n"
+"        return (Class<BeanType>) "+typeName+".class;\n"
+"    "
        );

//        boolean generateContructors = !aClass.isAbstract();
//        if (generateContructors) {
//            generateAbstractBeanDefaultsConstructors(output, typeName);
//        }

        generateAbstractBeanDefaultsCopyMethods(output, typeName);

        boolean generatePredicates = !getJavaTemplatesStereoTypes().isSimpleBeanWithNoInterfaceSkipGeneratePredicates(aClass, aPackage, model);
        if (generatePredicates) {
            generateAbstractBeanDefaultsPredicates(aClass, output, typeName);
        }

        boolean generateFunctions = !getJavaTemplatesStereoTypes().isSimpleBeanWithNoInterfaceSkipGenerateFunctions(aClass, aPackage, model);
        if (generateFunctions) {
            generateAbstractBeanDefaultsFunctions(aClass, output, typeName);
        }

    }

    protected void generateAbstractBeanDefaultsConstructors(ObjectModelClass output, String typeName) {

        ObjectModelOperation operation = addOperation(
                output,
                "new" + typeName,
                typeName,
                ObjectModelJavaModifier.STATIC,
                ObjectModelJavaModifier.PUBLIC
        );
        setOperationBody(operation, ""
    +"\n"
+"        return new "+typeName+"();\n"
+"    "
        );

        operation = addOperation(
                output,
                "new" + typeName,
                "<BeanType extends " + typeName + "> BeanType",
                ObjectModelJavaModifier.STATIC,
                ObjectModelJavaModifier.PUBLIC
        );
        addParameter(operation, "BeanType", "source");
        setOperationBody(operation, ""
    +"\n"
+"        Class<BeanType> sourceType = typeOf"+typeName+"();\n"
+"        Binder<BeanType, BeanType> binder = BinderFactory.newBinder(sourceType);\n"
+"        BeanType result = new"+typeName+"(source, binder);\n"
+"        return result;\n"
+"    "
        );

        operation = addOperation(
                output,
                "new" + typeName,
                "<BeanType extends " + typeName + "> BeanType",
                ObjectModelJavaModifier.STATIC,
                ObjectModelJavaModifier.PUBLIC
        );
        addParameter(operation, "BeanType", "source");
        addParameter(operation, "Binder<BeanType, BeanType>", "binder");
        setOperationBody(operation, ""
    +"\n"
+"        BeanType result = (BeanType) new"+typeName+"();\n"
+"        binder.copy(source, result);\n"
+"        return result;\n"
+"    "
        );

    }

    protected void generateAbstractBeanDefaultsCopyMethods(ObjectModelClass output, String typeName) {

        ObjectModelOperation operation = addOperation(
                output,
                "copy" + typeName,
                "<BeanType extends " + typeName + "> void",
                ObjectModelJavaModifier.STATIC,
                ObjectModelJavaModifier.PUBLIC
        );
        addParameter(operation, "BeanType", "source");
        addParameter(operation, "BeanType", "target");
        setOperationBody(operation, ""
    +"\n"
+"        Class<BeanType> sourceType = typeOf"+typeName+"();\n"
+"        Binder<BeanType, BeanType> binder = BinderFactory.newBinder(sourceType);\n"
+"        binder.copy(source, target);\n"
+"    "
        );

        operation = addOperation(
                output,
                "copy" + typeName,
                "<BeanType extends " + typeName + "> void",
                ObjectModelJavaModifier.STATIC,
                ObjectModelJavaModifier.PUBLIC
        );
        addParameter(operation, "BeanType", "source");
        addParameter(operation, "BeanType", "target");
        addParameter(operation, "Binder<BeanType, BeanType>", "binder");
        setOperationBody(operation, ""
    +"\n"
+"        binder.copy(source, target);\n"
+"    "
        );

    }

    protected void generateAbstractBeanDefaultsPredicates(ObjectModelClass input, ObjectModelClass output, String typeName) {

        boolean atLeastOnePropertyFound = false;
        for (ObjectModelAttribute attribute : getProperties(input)) {

            boolean multiple = JavaGeneratorUtil.isNMultiplicity(attribute);

            if (multiple) {
                continue;
            }

            atLeastOnePropertyFound = true;
            String attrName = getAttributeName(attribute);
            String attrType = getAttributeType(attribute);
            addImport(output, attrType);

            String simpleType = JavaGeneratorUtil.getSimpleName(attrType);

            String capitalizeAttrName = JavaGeneratorUtil.capitalizeJavaBeanPropertyName(attrName);
            String newPreficateMethodName = "new" + capitalizeAttrName + "Predicate";
            ObjectModelOperation operation = addOperation(
                    output,
                    newPreficateMethodName,
                    "<BeanType extends " + typeName + "> Predicate<BeanType>",
                    ObjectModelJavaModifier.STATIC,
                    ObjectModelJavaModifier.PUBLIC
            );
            addParameter(operation, simpleType, attrName);

            String getterName = getGetterName(attribute, attrName);
            setOperationBody(operation, ""
    +"\n"
+"        final "+simpleType+" $tmp = "+attrName+";\n"
+"        return new Predicate<BeanType>() {\n"
+"\n"
+"            @Override\n"
+"            public boolean apply(BeanType input) {\n"
+"                return Objects.equals($tmp, input."+getterName+"());\n"
+"            }\n"
+"        };\n"
+"\n"
+"    "
            );

            operation = addOperation(
                    output,
                    "filterBy" + capitalizeAttrName,
                    "<BeanType extends " + typeName + "> Iterable<BeanType>",
                    ObjectModelJavaModifier.STATIC,
                    ObjectModelJavaModifier.PUBLIC
            );
            addParameter(operation, "Iterable<BeanType>", "source");
            addParameter(operation, simpleType, attrName);
            setOperationBody(operation, ""
    +"\n"
+"        return Iterables.filter(source, "+newPreficateMethodName+"("+attrName+"));\n"
+"    "
            );
        }

        if (atLeastOnePropertyFound) {
            addImport(output, Predicate.class);
            addImport(output, Objects.class);
            addImport(output, Iterable.class);
            addImport(output, Iterables.class);
        }

    }

    protected void generateAbstractBeanDefaultsFunctions(ObjectModelClass input, ObjectModelClass output, String typeName) {

        boolean atLeastOnePropertyFound = false;
        for (ObjectModelAttribute attribute : getProperties(input)) {

            boolean multiple = JavaGeneratorUtil.isNMultiplicity(attribute);

            if (multiple) {
                continue;
            }

            atLeastOnePropertyFound = true;

            String attrName = getAttributeName(attribute);
            String attrType = getAttributeType(attribute);
            addImport(output, attrType);

            String simpleType = JavaGeneratorUtil.getSimpleName(attrType);
            simpleType = wrapPrimitiveType(simpleType);
            String capitalizeAttrName = JavaGeneratorUtil.capitalizeJavaBeanPropertyName(attrName);

            String newFunctionMethodName = "new" + capitalizeAttrName + "Function";
            String getFunctionMethodName = "get" + capitalizeAttrName + "Function";
            String functionTypeName = "Function<BeanType, " + simpleType + ">";

            String functionFieldName = JavaGeneratorUtil.convertVariableNameToConstantName(capitalizeAttrName + "Function");
            addAttribute(
                    output,
                    functionFieldName,
                    "Function<" + typeName + ", " + simpleType + ">",
                    newFunctionMethodName + "()",
                    ObjectModelJavaModifier.FINAL,
                    ObjectModelJavaModifier.STATIC,
                    ObjectModelJavaModifier.PROTECTED
            );


            ObjectModelOperation operation = addOperation(
                    output,
                    getFunctionMethodName,
                    "<BeanType extends " + typeName + "> " + functionTypeName,
                    ObjectModelJavaModifier.STATIC,
                    ObjectModelJavaModifier.PUBLIC
            );


            setOperationBody(operation, ""
    +"\n"
+"        return ("+functionTypeName+") "+functionFieldName+";\n"
+"\n"
+"    "
            );

            operation = addOperation(
                    output,
                    newFunctionMethodName,
                    "<BeanType extends " + typeName + "> " + functionTypeName,
                    ObjectModelJavaModifier.STATIC,
                    ObjectModelJavaModifier.PUBLIC
            );

            String getterName = getGetterName(attribute, attrName);
            setOperationBody(operation, ""
    +"\n"
+"        return new "+functionTypeName+"() {\n"
+"\n"
+"            @Override\n"
+"            public "+simpleType+" apply(BeanType input) {\n"
+"                return input."+getterName+"();\n"
+"            }\n"
+"        };\n"
+"\n"
+"    "
            );

            operation = addOperation(
                    output,
                    "uniqueIndexBy" + capitalizeAttrName,
                    "<BeanType extends " + typeName + "> ImmutableMap<" + simpleType + ", BeanType>",
                    ObjectModelJavaModifier.STATIC,
                    ObjectModelJavaModifier.PUBLIC
            );
            addParameter(operation, "Iterable<BeanType>", "source");
            setOperationBody(operation, ""
    +"\n"
+"        return Maps.uniqueIndex(source, "+functionFieldName+");\n"
+"    "
            );
        }

        if (atLeastOnePropertyFound) {
            addImport(output, Function.class);
            addImport(output, Objects.class);
            addImport(output, ImmutableMap.class);
            addImport(output, Iterable.class);
            addImport(output, Maps.class);
        }

    }

    protected String getAbstractDefaultsSuperClassName(ObjectModelPackage aPackage, ObjectModelClass aClass) {
        String superClassName = null;

        // test if a super class has bean stereotype
        boolean superClassIsBean = false;
        Collection<ObjectModelClass> superclasses = aClass.getSuperclasses();
        if (CollectionUtils.isNotEmpty(superclasses)) {
            for (ObjectModelClass superclass : superclasses) {
                superClassIsBean = beanDefaultClasses.contains(superclass);
                if (superClassIsBean) {
                    superClassName = superclass.getPackageName() + "." + beanDefaultsNameTranslation.get(superclass);
                    break;
                }
                superClassName = superclass.getQualifiedName();
            }
        }

        if (!superClassIsBean) {

            // try to find a super class by tag-value
            superClassName = getJavaTemplatesTagValues().getSimpleBeanWithNoInterfaceDefaultsSuperClassTagValue(aClass, aPackage, model);

        }
        return superClassName;
    }

    @Override
    protected String getAttributeType(ObjectModelAttribute attr) {
        String attrType = super.getAttributeType(attr);
        if (!JavaGeneratorUtil.isPrimitiveType(attrType)) {
            boolean hasClass = model.hasClass(attrType);
            if (hasClass) {
                ObjectModelClass attributeClass = model.getClass(attrType);
                String attributeType = beanNameTranslation.get(attributeClass);
                if (attributeType != null) {
                    attrType = attributeClass.getPackageName() + "." + attributeType;
                }
                String generic = getJavaTemplatesTagValues().getSimpleBeanWithNoInterfaceAttributeGenericTagValue(attr);

                if (generic != null) {
                    attrType += "<" + getAttributeType(generic) + ">";
                }
            }
        }
        return attrType;
    }

    protected String getAttributeType(String attrType) {
        if (!JavaGeneratorUtil.isPrimitiveType(attrType)) {
            boolean hasClass = model.hasClass(attrType);
            if (hasClass) {
                ObjectModelClass attributeClass = model.getClass(attrType);
                String attributeType = beanNameTranslation.get(attributeClass);
                if (attributeType != null) {
                    attrType = attributeClass.getPackageName() + "." + attributeType;
                }
            }
        }
        return attrType;
    }

}

