/* *##% Pollen
 * Copyright (C) 2009 CodeLutin
 *
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>. ##%*/

package org.chorem.pollen.ui.pages.user;

import java.util.HashMap;
import java.util.Map;

import org.apache.tapestry5.BindingConstants;
import org.apache.tapestry5.annotations.Component;
import org.apache.tapestry5.annotations.IncludeStylesheet;
import org.apache.tapestry5.annotations.Parameter;
import org.apache.tapestry5.annotations.Property;
import org.apache.tapestry5.annotations.SessionState;
import org.apache.tapestry5.corelib.components.Form;
import org.apache.tapestry5.corelib.components.PasswordField;
import org.apache.tapestry5.corelib.components.TextField;
import org.apache.tapestry5.ioc.Messages;
import org.apache.tapestry5.ioc.annotations.Inject;
import org.chorem.pollen.business.business.PreventRuleManager;
import org.chorem.pollen.business.dto.UserDTO;
import org.chorem.pollen.business.services.ServiceAuth;
import org.chorem.pollen.business.services.ServiceUser;
import org.chorem.pollen.business.utils.MD5;
import org.chorem.pollen.ui.data.Lien;
import org.chorem.pollen.ui.services.Configuration;

/**
 * Classe de la page d'enregistrement d'un utilisateur.
 *
 * @author kmorin
 * @author rannou
 * @version $Id: Register.java 2904 2010-03-08 15:30:41Z echatellier $
 */
@IncludeStylesheet("context:css/register.css")
public class Register {

    /**
     * Objet de session représentant l'url du site.
     */
    @SessionState
    private String siteURL;

    /**
     * Objet de session représentant l'utilisateur identifié.
     */
    @SuppressWarnings("unused")
    @SessionState
    private UserDTO user;
    @SuppressWarnings("unused")
    @Property
    private boolean userExists;

    /**
     * Utilisateur créé via le formulaire
     */
    @Property
    private UserDTO newUser = new UserDTO();

    @Component
    private Form registerForm;

    @Component(id = "login")
    private TextField loginField;

    @Component(id = "email")
    private TextField emailField;

    @Component(id = "password1")
    private PasswordField passwordField;

    /**
     * Champ password1 du formulaire d'enregsitrement d'un utilisateur
     */
    @Property
    private String password1;
    /**
     * Champ password2 du formulaire d'enregsitrement d'un utilisateur
     */
    @Property
    private String password2;

    @Inject
    private Messages messages;

    /**
     * Service contenant la configuration de l'application.
     */
    @Inject
    private Configuration conf;

    @Parameter(defaultPrefix = BindingConstants.MESSAGE, value = "title")
    @Property
    private String title;

    @SuppressWarnings("unused")
    @Property
    private Lien[] address;

    /** Injection des services */
    @Inject
    private ServiceUser serviceUser;
    @Inject
    private ServiceAuth serviceAuth;

    /**
     * Méthode appelée quand l'utilisateur s'enregistre
     *
     * @return la page d'accueil
     */
    Object onSuccessFromRegisterForm() {

        // L'utilisateur s'est pas trompé en répétant son mot de passe
        if (!password1.equals(password2)) {
            password1 = null;
            registerForm.recordError(passwordField, messages
                    .get("passwords-dont-match"));
        }

        // Le login existe déjà
        if (serviceUser.isUserLoginExisting(newUser.getLogin())) {
            registerForm.recordError(loginField, messages
                    .get("loginAlreadyExists"));
        }

        // L'email existe déjà
        if (serviceUser.isUserEmailExisting(newUser.getEmail())) {
            registerForm.recordError(emailField, messages
                    .get("emailAlreadyExists"));
        }

        // Création de l'utilisateur, envoi d'un email de notification et connexion
        if (!registerForm.getHasErrors()) {
            if (serviceUser.createUser(newUser, MD5.encode(password1)) != null) {
                sendMailNotification();
                user = serviceAuth.getUser(newUser.getLogin(), MD5
                        .encode(password1));
                return "Index";
            }
        }

        return this;
    }

    /** Envoi du mail de notification */
    private void sendMailNotification() {
        Map<String, String> data = new HashMap<String, String>();
        data.put("host", conf.getProperty("email_host"));
        data.put("port", conf.getProperty("email_port"));
        data.put("from", conf.getProperty("email_from"));
        data.put("to", newUser.getEmail());
        data.put("title", messages.format("registerEmail_subject", newUser
                .getLogin()));
        data.put("msg", messages.format("registerEmail_msg", newUser
                .getFirstName(), newUser.getLastName(), newUser.getLogin(),
                password1, siteURL));

        // FIXME call directly MailUtil.sendMail()
        // skip fill map, get from map...
        PreventRuleManager.emailAction(data);
    }

    /**
     * Initialisation de l'affichage
     */
    void setupRender() {
        address = new Lien[] { new Lien("Pollen", "Index"),
                new Lien(title, null) };
    }
}
