/* *##% Pollen
 * Copyright (C) 2009 CodeLutin
 *
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>. ##%*/

package org.chorem.pollen.ui.pages.poll;

import java.io.File;
import java.text.DateFormat;
import java.util.ArrayList;
import java.util.Date;
import java.util.HashMap;
import java.util.Iterator;
import java.util.List;
import java.util.Locale;
import java.util.Map;
import java.util.UUID;

import org.apache.tapestry5.BindingConstants;
import org.apache.tapestry5.ValidationException;
import org.apache.tapestry5.annotations.Component;
import org.apache.tapestry5.annotations.IncludeJavaScriptLibrary;
import org.apache.tapestry5.annotations.IncludeStylesheet;
import org.apache.tapestry5.annotations.InjectComponent;
import org.apache.tapestry5.annotations.Log;
import org.apache.tapestry5.annotations.Parameter;
import org.apache.tapestry5.annotations.Persist;
import org.apache.tapestry5.annotations.Property;
import org.apache.tapestry5.annotations.Retain;
import org.apache.tapestry5.annotations.SessionState;
import org.apache.tapestry5.beaneditor.BeanModel;
import org.apache.tapestry5.corelib.components.Form;
import org.apache.tapestry5.corelib.components.TextField;
import org.apache.tapestry5.corelib.components.Zone;
import org.apache.tapestry5.ioc.Messages;
import org.apache.tapestry5.ioc.annotations.Inject;
import org.apache.tapestry5.upload.services.UploadedFile;
import org.chenillekit.tapestry.core.components.DateTimeField;
import org.chorem.pollen.business.business.PreventRuleManager;
import org.chorem.pollen.business.dto.ChoiceDTO;
import org.chorem.pollen.business.dto.CommentDTO;
import org.chorem.pollen.business.dto.PollAccountDTO;
import org.chorem.pollen.business.dto.PollDTO;
import org.chorem.pollen.business.dto.PreventRuleDTO;
import org.chorem.pollen.business.dto.ResultDTO;
import org.chorem.pollen.business.dto.ResultListDTO;
import org.chorem.pollen.business.dto.UserDTO;
import org.chorem.pollen.business.dto.VoteDTO;
import org.chorem.pollen.business.dto.VotingListDTO;
import org.chorem.pollen.business.services.ServiceChoice;
import org.chorem.pollen.business.services.ServiceComment;
import org.chorem.pollen.business.services.ServicePoll;
import org.chorem.pollen.business.services.ServicePollAccount;
import org.chorem.pollen.business.services.ServiceResults;
import org.chorem.pollen.business.services.ServiceVote;
import org.chorem.pollen.business.utils.MD5;
import org.chorem.pollen.common.ChoiceType;
import org.chorem.pollen.common.PollType;
import org.chorem.pollen.common.VoteCountingType;
import org.chorem.pollen.ui.base.ContextLink;
import org.chorem.pollen.ui.components.FeedBack;
import org.chorem.pollen.ui.components.ImageContextLink;
import org.chorem.pollen.ui.data.EvenOdd;
import org.chorem.pollen.ui.data.Lien;
import org.chorem.pollen.ui.data.PollAction;
import org.chorem.pollen.ui.data.uio.DateChoiceUIO;
import org.chorem.pollen.ui.data.uio.ImageChoiceUIO;
import org.chorem.pollen.ui.services.Configuration;
import org.chorem.pollen.ui.utils.FeedUtil;
import org.chorem.pollen.ui.utils.ImageUtil;
import org.chorem.pollen.votecounting.business.NumberMethod;
import org.slf4j.Logger;

/**
 * Classe de la page de vote.
 *
 * @author kmorin
 * @author rannou
 * @version $Id: VoteForPoll.java 2859 2010-02-09 13:50:24Z jruchaud $
 */
@IncludeStylesheet({"context:css/vote.css", "context:css/lightbox.css"})
@IncludeJavaScriptLibrary({"${tapestry.scriptaculous}/builder.js","context:js/lightbox.js"})
public class VoteForPoll {

    @Inject
    private Logger log;

    @Parameter(defaultPrefix = BindingConstants.MESSAGE, value = "title")
    @Property
    private String title;

    @SuppressWarnings("unused")
    @Property
    private Lien[] address;

    /** Paramètres de la page */
    private String param;

    /** Affichage des messages pour l'utilisateur */
    @Component(id = "feedback")
    private FeedBack feedback;

    /**
     * Objet de session représentant l'utilisateur identifié.
     */
    @SessionState
    private UserDTO user;
    @Property
    private boolean userExists;

    /**
     * Objet de session représentant l'url du site.
     */
    @SessionState
    @Property
    private String siteURL;

    @Component
    private Form voteForm;

    @Component(id = "pollAccountName")
    private TextField nameField;

    /** Composant DateTimeField pour les choix du sondage */
    @SuppressWarnings("unused")
    @Component(parameters = { "timePicker=true", "timePickerAdjacent=true" })
    private DateTimeField dateDTF;

    /** Format des dates */
    @Persist
    @Property
    private DateFormat dateFormat;

    /** Locale courante */
    @Inject
    private Locale currentLocale;

    @Inject
    private Logger logger;

    @Inject
    private Messages messages;

    /**
     * Service contenant la configuration de l'application.
     */
    @Inject
    private Configuration conf;

    /**
     * Sondage pour lequel l'utilisateur vote
     */
    @Property
    @Persist
    private PollDTO poll;

    /**
     * Vote courant de l'utilisateur
     */
    @Property
    private VoteDTO vote;

    /** Choix courant du sondage */
    @Property
    private ChoiceDTO choiceOfPoll;
    /** Choix courant du nouveau vote */
    @Property
    private ChoiceDTO choiceOfVote;

    /** Résultats du sondage */
    @Persist
    private List<ResultDTO> results;

    /**
     * Modèle pour l'affichage de la liste des sondages
     */
    @SuppressWarnings( { "unchecked", "unused" })
    @Property
    @Retain
    private BeanModel voteModel;

    /*@InjectPage
    private ImageDisplay imageDisplay;*/

    @InjectComponent
    private Zone pollZone;

    @InjectComponent
    private Zone commentZone;

    /**
     * Objet servant à changer la couleur à chaque ligne de la liste des
     * sondages
     */
    @SuppressWarnings("unused")
    @Property
    private EvenOdd evenOdd = new EvenOdd();

    /** Choix courant (type texte) */
    private boolean addChoice;

    /** Le votant a déjà voté ? */
    private boolean alreadyVoted;

    /** Le vote est anonyme */
    @Property
    private boolean anonymousVote = false;

    /** Liste des choix */
    private List<ChoiceDTO> voteChoices = new ArrayList<ChoiceDTO>();

    /** Identifiant du compte correspondant à l'adresse forgée */
    private String pollAccountId;

    /** Compte du votant */
    @Property
    private PollAccountDTO pollAccount;

    /** Nouveau choix de type texte */
    @Property
    @Persist
    private ChoiceDTO newChoice;

    /** Nouveau choix de type date */
    @Property
    @Persist
    private DateChoiceUIO newDateChoice;

    /** Nouveau choix de type image */
    @Property
    @Persist
    private ImageChoiceUIO newImageChoice;

    /** Commentaire courant pour l'affichage des commentaires */
    @SuppressWarnings("unused")
    @Property
    @Persist
    private CommentDTO comment;

    /** Nouveau commentaire posté */
    @Property
    @Persist
    private CommentDTO newComment;

    /** Injection des services */
    @Inject
    private ServicePoll servicePoll;
    @Inject
    private ServiceVote serviceVote;
    @Inject
    private ServiceComment serviceComment;
    @Inject
    private ServicePollAccount servicePollAccount;
    @Inject
    private ServiceResults serviceResults;
    @Inject
    private ServiceChoice serviceChoice;
    
    /**
     * Composant pour la gestion des flux RSS
     */
    @InjectComponent
    private ContextLink feedContext;

    /**
     * Context to get feed of the current poll
     * @return a FeedContext defined in .tml file
     */
    public ContextLink getFeedContext() {
        return feedContext;
    }

    /**
     * Composant pour la gestion des upload d'images
     */
    @InjectComponent
    private ImageContextLink imgContext;

    /**
     * Context to get images for choices.
     * @return an ImageContextLink defined in .tml file
     */
    public ImageContextLink getImgContext() {
        return imgContext;
    }

    /** Méthode appelée après la soumission du vote. */
    @Log
    Object onSuccessFromVoteForm() {
        if (initPollAccount()) {
            List<ChoiceDTO> choiceDTOs = new ArrayList<ChoiceDTO>();
            if (poll.getVoteCounting() == VoteCountingType.NORMAL) {
                int nbChoices = 0;
                for (ChoiceDTO choice : poll.getChoiceDTOs()) {
                    if (choice.getValue() == 1) {
                        nbChoices++;
                    }
                    choiceDTOs.add(choice);
                }
                if (nbChoices > poll.getMaxChoiceNb()) {
                    voteForm.recordError(messages.format(("tooManyChoices"),
                            poll.getMaxChoiceNb()));
                    return pollZone.getBody();
                }
                
            } else if (poll.getVoteCounting() == VoteCountingType.PERCENTAGE) {
                int total = 0;
                for (ChoiceDTO choice : poll.getChoiceDTOs()) {
                    total = total + choice.getValue();
                    choiceDTOs.add(choice);
                }
                if (total != 100) {
                    voteForm.recordError(messages.get("not100percent"));
                    return pollZone.getBody();
                }
                
            } else if (poll.getVoteCounting() == VoteCountingType.CONDORCET) {
                for (ChoiceDTO choice : poll.getChoiceDTOs()) {
                    if (choice.getValue() == 0) {
                        choice.setValue(100);
                    }
                    choiceDTOs.add(choice);
                }
                
            } else if (poll.getVoteCounting() == VoteCountingType.NUMBER) {
                String votingId = pollAccount.getVotingId();
                if (anonymousVote) {
                    votingId = "anonymous" + UUID.randomUUID().toString().replaceAll("-", "");
                }
                
                for (ChoiceDTO choice : poll.getChoiceDTOs()) {
                    if (!choice.isHidden()) {
                        choiceDTOs.add(choice);
                        
                        // creates a new hidden choice
                        ChoiceDTO hiddenChoice = new ChoiceDTO();
                        hiddenChoice.setName(NumberMethod.HIDDEN_PREFIX +
                                choice.getName() + "#" + votingId);
                        hiddenChoice.setHidden(true);
                        hiddenChoice.setValue(choice.getValue());
                        hiddenChoice.setPollId(choice.getPollId());
                        hiddenChoice.setVoteId(choice.getVoteId());
                        hiddenChoice.setValidate(false);
                        
                        String choiceId = serviceChoice.createChoice(hiddenChoice);
                        hiddenChoice.setId(choiceId);
                        choiceDTOs.add(hiddenChoice);
                    }
                }
            }

            VoteDTO vote = new VoteDTO(null, poll.getId(), null);
            vote.setChoiceDTOs(choiceDTOs);
            vote.setWeight(pollAccount.getWeight());
            vote.setAnonymous(anonymousVote);

            // mise à jour du vote ou création d'un nouveau vote
            if (alreadyVoted) {
                for (VoteDTO v : poll.getVoteDTOs()) {
                    PollAccountDTO voteAccount = servicePollAccount
                            .findPollAccountById(v.getPollAccountId());
                    
                    if (voteAccount.getVotingId().equals(pollAccount.getVotingId())) {
                        vote.setId(v.getId());
                        deleteVote(vote.getId());
                        serviceVote.createVote(vote, pollAccount);
                    }
                }
            } else {
                serviceVote.createVote(vote, pollAccount);
            }

            // Mise à jour du sondage et des résultats
            poll = servicePoll.findPollByPollId(poll.getPollId());
            countPoll();

            addFeedEntry(PollAction.ADDVOTE, pollAccount.getVotingId(),
                    getResultsAsString());
            sendMailNotification();
        }
        voteChoices.clear();
        return pollZone.getBody();
    }
    
    private void deleteVote(String voteId) {
        for (VoteDTO vote : poll.getVoteDTOs()) {
            if (vote.getId().equals(voteId)) {

                List<ChoiceDTO> choiceDTOs = vote.getChoiceDTOs();
                for (ChoiceDTO choiceDTO : choiceDTOs) {

                    if (choiceDTO.isHidden()) {
                        serviceChoice.deleteChoice(choiceDTO.getId());
                    }
                }

                break;
            }
        }
        
        serviceVote.deleteVote(voteId);
    }
    
    /** Méthode appelée lors de la suppression d'un vote. */
    Object onActionFromDeleteVote(String voteId) {
        deleteVote(voteId);

        // Mise à jour du sondage et des résultats
        poll = servicePoll.findPollByPollId(poll.getPollId());
        countPoll();

        return pollZone.getBody();
    }

    /** Méthode appelée lors de l'édition d'un vote. */
    Object onActionFromEditVote(String votingId) {
        pollAccount.setVotingId(votingId);
        return pollZone.getBody();
    }

    /** Initialisation du pollAccount et contrôle du nom. */
    private boolean initPollAccount() {

        //// Contrôle et définition du votingId
        alreadyVoted = false;
        boolean modifAllowed = false;
        boolean restrictedListsForbidden = false;

        // Suppression des espaces pouvant provoquer un double vote
        String votingId = pollAccount.getVotingId().trim();
        pollAccount.setVotingId(votingId);

        // Contrôle de la présence du votant dans les listes de votants
        // du sondage (si le sondage n'est pas libre)
        if (!isFreePoll()) {
            restrictedListsForbidden = true;
            for (VotingListDTO list : poll.getVotingListDTOs()) {
                for (PollAccountDTO account : list.getPollAccountDTOs()) {
                    if (pollAccount.getVotingId().equals(account.getVotingId())) {
                        restrictedListsForbidden = false;
                        pollAccount = servicePollAccount
                                .findPollAccountById(account.getId());
                        pollAccount.setVotingListId(list.getId());
                        pollAccount.setWeight(account.getWeight());
                        logger.debug("Compte \"" + account.getVotingId()
                                + "\" présent dans la liste \""
                                + list.getName() + "\" (poids="
                                + account.getWeight() + ")");
                    }
                }
            }
            if (restrictedListsForbidden) {
                voteForm.recordError(nameField, messages.format(
                        "restrictedListsForbidden", pollAccount.getVotingId()));
            }
        }

        // Génération d'un identifiant de vote (si le sondage est libre et anonyme)
        if (poll.isAnonymous()) {
            anonymousVote = true;
            if (isFreePoll()) {
                pollAccount.setVotingId("anonymous"
                        + UUID.randomUUID().toString().replaceAll("-", ""));
            }
        }

        //// Définition de l'userId
        if (userExists) {
            pollAccount.setUserId(user.getId());
        } else {
            pollAccount.setUserId("");
        }

        // Contrôle si le votant n'a pas déjà voté
        alreadyVoted = hasAlreadyVoted(pollAccount.getVotingId());
        modifAllowed = isModifAllowed(pollAccount.getVotingId());
        if (alreadyVoted && !modifAllowed) {
            voteForm.recordError(nameField, messages.format("alreadyVoted",
                    pollAccount.getVotingId()));
        }

        logger.debug("votingId: " + pollAccount.getVotingId());
        logger.debug("alreadyVoted: " + alreadyVoted);
        logger.debug("modifAllowed: " + modifAllowed);
        logger.debug("restrictedListsForbidden: " + restrictedListsForbidden);
        return ((!alreadyVoted || modifAllowed) && !restrictedListsForbidden);
    }

    /**
     * Retourne vrai si le votant a déjà voté.
     *
     * @param votingId le votant a rechercher
     * @return vrai si le votant a déjà voté
     */
    private boolean hasAlreadyVoted(String votingId) {

        // comparaison du votant (pollAccount) avec les votants des votes existants
        for (VoteDTO vote : poll.getVoteDTOs()) {
            PollAccountDTO account = servicePollAccount
                    .findPollAccountById(vote.getPollAccountId());

            if (account.getVotingId().equals(votingId)) {
                return true;
            }
        }

        return false;
    }

    /**
     * Retourne vrai si la modification du vote est autorisée. C'est à dire si
     * l'utilisateur connecté est l'auteur du vote ou si le votant défini par
     * l'url est l'auteur du vote.
     *
     * @param votingId le votant a rechercher
     * @return vrai si la modification du vote est autorisée
     */
    public boolean isModifAllowed(String votingId) {
        boolean modifAllowed = false;
        int i = 0;

        // parcours des votes pour trouver celui correspondant au votingId
        // et contrôle du droit de modification
        while (i < poll.getVoteDTOs().size() && !modifAllowed) {

            // account : compte associé au vote courant
            String id = poll.getVoteDTOs().get(i).getPollAccountId();
            PollAccountDTO account = servicePollAccount.findPollAccountById(id);

            // si le votant du vote correspond au votingId
            if (account.getVotingId().equals(votingId)) {
                if (logger.isDebugEnabled()) {
                    logger.debug("vote account: " + account.getId());
                    logger.debug("voting account: " + pollAccountId);
                }

                // si le votant du vote correspond au votant actuel (pollAccountId)
                if (pollAccountId != null
                        && pollAccountId.equals(account.getId())) {
                    modifAllowed = true;
                }

                // si l'utilisateur du vote correspond à l'utilisateur actuel (user)
                if (userExists && user.getId().length() > 0) {
                    modifAllowed = user.getId().equals(account.getUserId());
                }
            }

            i++;
        }

        return modifAllowed;
    }

    /** Ajout d'une entrée dans le flux de syndication */
    private void addFeedEntry(PollAction pollAction, String titleStr,
            String contentStr) {
        String voteURL = siteURL + "poll/VoteFor/" + poll.getPollId();
        File feedFile = getFeedContext().getFile(poll.getPollId());
        String title = null;
        String content = null;

        switch (pollAction) {
        case ADDVOTE:
            if (isVoteAnonymous()) {
                titleStr = messages.get("anonymous");
            }
            title = messages.format("pollFeed_voteTitle", titleStr);
            content = messages.format("pollFeed_voteContent", contentStr);
            break;
        case ADDCHOICE:
            title = messages.format("pollFeed_choiceTitle", titleStr);
            content = messages.format("pollFeed_choiceContent", contentStr);
            break;
        case ADDCOMMENT:
            title = messages.format("pollFeed_commentTitle", titleStr);
            content = messages.format("pollFeed_commentContent", contentStr);
            break;
        }

        if (!feedFile.exists()) {
            FeedUtil.createFeed(feedFile, "atom_1.0", messages.format(
                    "pollFeed_title", poll.getTitle()), siteURL, messages
                    .format("pollFeed_desc", poll.getDescription()));
        }
        FeedUtil.feedFeed(feedFile, title, voteURL, content);
    }

    /** Envoi du mail de notification */
    private void sendMailNotification() {
        String voteURL = siteURL + "poll/VoteFor/" + poll.getPollId();
        String modifURL = siteURL + "poll/Modification/" + poll.getPollId()
                + ":" + MD5.encode(poll.getCreatorId());
        Map<String, String> data = new HashMap<String, String>();
        data.put("host", conf.getProperty("email_host"));
        data.put("port", conf.getProperty("email_port"));
        data.put("from", conf.getProperty("email_from"));

        // Mail au créateur
        data.put("to", poll.getCreatorEmail());
        data
                .put("title", messages.format("voteEmail_subject", poll
                        .getTitle()));
        data.put("msg", messages.format("voteEmail_msg", poll.getTitle(), poll
                .getVoteDTOs().size(), voteURL, modifURL));

        for (PreventRuleDTO rule : poll.getPreventRuleDTOs()) {
            PreventRuleManager manager = new PreventRuleManager(rule);
            manager.execute("vote", poll.getVoteDTOs().size(), data);
        }
    }

    public boolean isAddChoice() {
        return addChoice;
    }

    public void setAddChoice(boolean addChoice) {
        choiceOfVote.setValue((addChoice) ? 1 : 0);
    }

    public void setAddNumberVote(Integer value) {
        if (value != null) {
            choiceOfVote.setValue(value);
            
        } else {
            choiceOfVote.setValue(-1);
        }
    }
    
    public Integer getAddNumberVote() {
        return null;
    }
    
    
    /**
     * Retourne si le choix fait partie du vote (s'il a été renseigné par le
     * votant).
     *
     * @param choice le choix concerné
     * @return true si le choix est dans le vote
     */
    public Boolean isChoiceInVote(ChoiceDTO choice) {
        if (choice != null) {
            switch (poll.getVoteCounting()) {
            case NORMAL:
                return choice.getValue() > 0;
            case PERCENTAGE:
                return true;
            case CONDORCET:
                return choice.getValue() < 100;
            case NUMBER:
                return choice.getValue() >= 0;
            }
        }
        return null;
    }

    public boolean isChoiceHidden() {
        return choiceOfPoll.isHidden();
    }
    
    /**
     * Retourne le choix de vote correspondant au choix de sondage courant.
     *
     * @return un choix de vote
     */
    @Property(write = false)
    private ChoiceDTO currentVoteChoice;
    
    public char setCurrentVoteChoice() {
        currentVoteChoice = null;
        for (ChoiceDTO choice : vote.getChoiceDTOs()) {
            if (choice.getId().equals(choiceOfPoll.getId())) {
                currentVoteChoice = choice;
                break;
            }
        }
        
        return 0;
    }

    /**
     * Retourne le résultat correspondant au choix de sondage courant.
     *
     * @return le résultat du choix
     */
    public String getCurrentChoiceResult() {
        String val = "";
        for (ResultDTO result : results) {
            if (result.getName().equals(choiceOfPoll.getName())) {
                val = removeTrailing0(result.getValue());
            }
        }
        return val;
    }

    /**
     * Vérifie que le fichier de flux de syndication existe.
     *
     * @return vrai si le fichier existe
     */
    @Log
    public boolean isFeedFileExisting() {
        File feedFile = getFeedContext().getFile(poll.getPollId());
        if (log.isDebugEnabled()) {
            log.debug("feed context path : " + getFeedContext().getContextPath());
            log.debug("feed absolute path : " + feedFile.getAbsolutePath());
        }
        return feedFile.exists() ? true : false;
    }
    
    /**
     * Supprime le 0 final d'un nombre à virgule. Le résultat peut-être un
     * double : 1,0 -> 1 et 1,2 -> 1,2.
     *
     * @param val le nombre
     * @return le nombre sans 0 final
     */
    private String removeTrailing0(String val) {
        if (val.endsWith(".0")) {
            val = val.substring(0, val.indexOf('.'));
        }
        return val;
    }

    /**
     * Retourne une chaîne contenant les résultats du sondage.
     *
     * @return les résultats
     */
    public String getResultsAsString() {
        StringBuffer res = new StringBuffer("");
        Iterator<ResultDTO> it = results.iterator();
        while (it.hasNext()) {
            ResultDTO result = it.next();
            if (isDateType()) {
                Date date = new Date(Long.parseLong(result.getName()));
                res.append(dateFormat.format(date));
            } else {
                res.append(result.getName());
            }
            res.append("=" + removeTrailing0(result.getValue()));
            if (it.hasNext()) {
                res.append(", ");
            }
        }
        return res.toString();
    }

    /**
     * Retourne true si le sondage est anonyme ou si le vote soumis est anonyme.
     * Utilisé lors de la soumission d'un vote.
     *
     * @return true si le vote est anonyme
     */
    public boolean isVoteAnonymous() {
        return poll.isAnonymous() || anonymousVote;
    }

    /**
     * Retourne true si le sondage est anonyme ou si le vote courant est
     * anonyme. Utilisé dans la boucle de parcours des votes.
     *
     * @return true si le vote est anonyme
     */
    public boolean isCurrentVoteAnonymous() {
        return poll.isAnonymous() || vote.isAnonymous();
    }

    /** Retourne vrai si la checkbox anonymousVote doit être affichée. */
    public boolean isAnonymousVoteDisplayed() {
        return poll.isAnonymousVoteAllowed() && !poll.isAnonymous();
    }

    /** Retourne l'identifiant du votant du vote courant */
    public String getCurrentVotingId() {
        String id = vote.getPollAccountId();
        String votingId = servicePollAccount.findPollAccountById(id)
                .getVotingId();
        return votingId;
    }

    /** Validation du champs de saisie du choix (texte). */
    void onValidateFromTextName(String value) throws ValidationException {
        validateNewChoice(value);
    }

    /** Validation du champs de saisie du choix (date). */
    void onValidateFromDateDTF(Date value) throws ValidationException {
        validateNewChoice(String.valueOf(value.getTime()));
    }

    /** Validation du champs de saisie du choix (image). */
    void onValidateFromImgFile(UploadedFile value) throws ValidationException {
        validateNewChoice(value.getFileName().replace(' ', '_'));
    }

    /** Validation du champs de saisie du choix. */
    private void validateNewChoice(String value) throws ValidationException {
        for (ChoiceDTO choice : poll.getChoiceDTOs()) {
            if (value.equals(choice.getName())) {
                throw new ValidationException(messages.format("choiceExists",
                        value));
            }
        }
    }

    /** Méthode appelée lors de l'ajout d'un choix. */
    Object onSuccessFromChoiceForm() {
        String choiceName = null;
        String choiceDesc = null;
        if (isTextType()) {
            if (newChoice.getName() != null) {
                newChoice.setValidate(true);
                newChoice.setPollId(poll.getId());
                poll.getChoiceDTOs().add(newChoice);
                choiceName = newChoice.getName();
                choiceDesc = newChoice.getDescription();
            }
        } else if (isDateType()) {
            if (newDateChoice.getDate() != null) {
                newDateChoice.setValidate(true);
                newDateChoice.setPollId(poll.getId());
                newDateChoice.setName(String.valueOf(newDateChoice.getDate()
                        .getTime()));
                poll.getChoiceDTOs().add(newDateChoice);
                Date date = new Date(Long.parseLong(newDateChoice.getName()));
                choiceName = dateFormat.format(date);
                choiceDesc = newDateChoice.getDescription();
            }
        } else if (isImageType()) {
            if (newImageChoice.getImg() != null) {
                newImageChoice.setValidate(true);
                newImageChoice.setPollId(poll.getId());
                newImageChoice.setName(newImageChoice.getImg().getFileName()
                        .replace(' ', '_'));
                poll.getChoiceDTOs().add(newImageChoice);
                //File imgFile = getImgContext().getImageDir(); // dir set in getImgContext()
                File imgFile = getImgContext().getImageDir(); // dir set in getImgContext()
                ImageUtil.saveImage(newImageChoice, imgFile);
                choiceName = newImageChoice.getName();
                choiceDesc = newImageChoice.getDescription();
            }
        }

        servicePoll.updatePoll(poll);
        poll = servicePoll.findPollByPollId(poll.getPollId());
        addFeedEntry(PollAction.ADDCHOICE, choiceName, choiceDesc);
        newChoice = new ChoiceDTO();
        newDateChoice = new DateChoiceUIO();
        newImageChoice = new ImageChoiceUIO();
        return this;
    }

    /** Méthode appelée lors de la suppression d'un choix. */
    Object onActionFromDeleteChoice(String choiceId) {
        Iterator<ChoiceDTO> itChoice = poll.getChoiceDTOs().iterator();
        while (itChoice.hasNext()) {
            ChoiceDTO choice = itChoice.next();
            if (choiceId.equals(choice.getId())) {
                itChoice.remove();
            }
        }
        servicePoll.updatePoll(poll);
        poll = servicePoll.findPollByPollId(poll.getPollId());
        return pollZone.getBody();
    }

    /** Méthode appelée lors de l'ajout d'un commentaire. */
    Object onSuccessFromCommentForm() {
        newComment.setPollId(poll.getId());
        newComment.setPostDate(new Date());
        serviceComment.createComment(newComment);
        poll = servicePoll.findPollByPollId(poll.getPollId());
        addFeedEntry(PollAction.ADDCOMMENT, newComment.getPollAccountId(),
                newComment.getText());
        newComment = new CommentDTO();
        return commentZone.getBody();
    }

    /** Méthode appelée lors de la suppression d'un commentaire. */
    Object onActionFromDeleteComment(String commentId) {
        serviceComment.deleteComment(commentId);
        poll = servicePoll.findPollByPollId(poll.getPollId());
        return commentZone.getBody();
    }

    public boolean isPollNull() {
        return poll == null;
    }

    public boolean isPollChoiceStarted() {
        Date now = new Date();
        boolean started = poll.getBeginChoiceDate() == null || poll.getBeginChoiceDate().before(now);
        boolean ended = poll.getEndChoiceDate() != null && poll.getEndChoiceDate().before(now);
        return started && !ended;
    }

    public boolean isPollStarted() {
        Date now = new Date();
        return poll.getBeginDate() == null || poll.getBeginDate().before(now);
    }

    public boolean isPollFinished() {
        Date now = new Date();
        return poll.getEndDate() != null && poll.getEndDate().before(now);
    }

    public boolean isPollChoiceOrVoteStarted() {
        return isPollChoiceStarted() || isPollStarted();
    }

    public boolean isPollChoiceRunning() {
        return poll.isChoiceAddAllowed() && isPollChoiceStarted();
                //&& !isPollStarted();
    }

    public boolean isPollRunning() {
        return isPollStarted() && !isPollFinished() && !poll.isClosed();
    }

    public boolean isDescNull() {
        return choiceOfPoll.getDescription() == null
                || "".equals(choiceOfPoll.getDescription());
    }

    /** Retourne vrai si le champs pollAccount doit apparaître. */
    public boolean isAccountFieldDisplayed() {
        return !poll.isAnonymous() || isRestrictedPoll() || isGroupPoll();
    }

    public boolean isFreePoll() {
        return poll.getPollType() == PollType.FREE;
    }

    public boolean isRestrictedPoll() {
        return poll.getPollType() == PollType.RESTRICTED;
    }

    public boolean isGroupPoll() {
        return poll.getPollType() == PollType.GROUP;
    }

    public boolean isTextType() {
        return poll.getChoiceType() == ChoiceType.TEXT;
    }

    public boolean isDateType() {
        return poll.getChoiceType() == ChoiceType.DATE;
    }

    public boolean isImageType() {
        return poll.getChoiceType() == ChoiceType.IMAGE;
    }

    public boolean isNormalVoteCounting() {
        return poll.getVoteCounting() == VoteCountingType.NORMAL;
    }

    public boolean isPercentageVoteCounting() {
        return poll.getVoteCounting() == VoteCountingType.PERCENTAGE;
    }

    public boolean isCondorcetVoteCounting() {
        return poll.getVoteCounting() == VoteCountingType.CONDORCET;
    }

    public boolean isNumberVoteCounting() {
        return poll.getVoteCounting() == VoteCountingType.NUMBER;
    }
    
    /** retourne vrai si l'utilisateur est le créateur du sondage */
    public boolean isCreatorUser() {
        PollAccountDTO creator = servicePollAccount.findPollAccountById(poll
                .getCreatorId());
        if (userExists && user.getId().equals(creator.getUserId())) {
            return true;
        }
        return false;
    }

    /**
     * Récupération du créateur du sondage.
     *
     * @return le nom du créateur du sondage.
     */
    public String getCreatorName() {
        PollAccountDTO pollAccount = servicePollAccount
                .findPollAccountById(poll.getCreatorId());
        return pollAccount.getVotingId();
    }

    /** Retourne le message d'aide correspondant au type de sondage. */
    public String getHelpMessage() {
        switch (poll.getVoteCounting()) {
        case NORMAL:
            return messages.get("normalVote-help");
        case PERCENTAGE:
            return messages.get("percentageVote-help");
        case CONDORCET:
            return messages.get("condorcetVote-help");
        case NUMBER:
            return messages.get("numberVote-help");
        default:
            return "";
        }
    }

    /** Retourne la date correspondant au choix courant */
    public Date getChoiceNameAsDate() {
        return new Date(Long.valueOf(choiceOfPoll.getName()));
    }

    /** Action réalisée lorsqu'on clique sur l'image */
    /*Object onDisplayImage(String choiceId) {
        imageDisplay.setPoll(poll);
        imageDisplay.setChoiceId(choiceId);
        imageDisplay.setPageStyle("Vote");
        return imageDisplay;
    }*/

    /** Affichage de "" plutôt que 0 pour le Condorcet */
    /*String onToClientFromCondorcetInput() {
        if (choiceOfVote.getValue() == 0) {
            return "";
        }
        return null;
    }*/

    /** Récupération d'un entier plutôt que "" pour le Condorcet */
    /*Object onParseClientFromCondorcetInput(String input) {
        if ("".equals(input)) return 0; //ne fonctionne pas
        return null;
    }*/

    /** Dépouillement du sondage. Mise à jour des résultats. */
    private void countPoll() {
        ResultListDTO resultListDTO = null;
        
        if (isGroupPoll()) {
            resultListDTO = serviceResults.getGroupResults(poll.getPollId());
        } else {
            resultListDTO = serviceResults.getNormalResults(poll.getPollId());
        }

        results = resultListDTO.getResultDTOs();

        if (logger.isDebugEnabled()) {
            for (ResultDTO res : results) {
                logger.debug(res.getName() + ": " + res.getValue()
                        + ", (voteCounting=" + res.getVoteCounting()
                        + ", byGroup=" + res.isByGroup() + ")");
            }
        }
    }

    /**
     * Méthode appelée au moment de l'activation de la page
     *
     * @param id l'identifiant du sondage et du votant
     */
    void onActivate(String id) {
        param = id;

        // Initialisation du sondage
        if (id != null && !"".equals(id)) {
            String pollId = id.split(":", 2)[0];
            poll = servicePoll.findPollByPollId(pollId);

            if (poll != null) {

                // Identification du votant
                if (id.split(":", 2).length == 2) {
                    String accountId = id.split(":", 2)[1];
                    pollAccount = servicePollAccount
                            .findPollAccountByAccountId(accountId);
                    pollAccountId = pollAccount.getId();
                }
                
                if (pollAccount == null) {
                    pollAccount = new PollAccountDTO();
                    if (userExists) {
                        if (user.getFirstName() != null && user.getLastName() != null) {
                            pollAccount.setVotingId(
                                    user.getFirstName() + " " + user.getLastName());
                            
                        } else if (user.getFirstName() != null) {
                            pollAccount.setVotingId(user.getFirstName());
                            
                        } else if (user.getLastName() != null) {
                            pollAccount.setVotingId(user.getLastName());
                        }
                    }
                }

                // Initialisations pour les formulaires
                newChoice = new ChoiceDTO();
                newDateChoice = new DateChoiceUIO();
                newImageChoice = new ImageChoiceUIO();
                newComment = new CommentDTO();

                newComment.setPollId(poll.getId());
            }
        }

        // Affichage des erreurs
        if (poll == null) {
            feedback.addError(messages.get("pollNotFound"));
        } else {
            if (poll.isClosed()) {
                feedback.addInfo(messages.get("pollClosed"));
            } else if (!isPollStarted()) {
                feedback.addInfo(messages.get("pollNotStarted"));
            } else if (isPollFinished()) {
                feedback.addInfo(messages.get("pollFinished"));
            }
            if (isPollChoiceRunning()) {
                feedback.addInfo(messages.get("pollChoiceRunning"));
            }
        }
    }

    /**
     * Méthode appelée au moment de la désactivation de la page
     *
     * @return l'identifiant du sondage et du votant
     */
    String onPassivate() {
        return param;
    }

    /**
     * Initialisation de l'affichage
     */
    void setupRender() {
        address = new Lien[] { new Lien("Pollen", "Index"),
                new Lien(title, null) };
        dateFormat = DateFormat.getDateTimeInstance(DateFormat.SHORT,
                DateFormat.SHORT, currentLocale);

        countPoll();
    }
}
