/* *##% Pollen
 * Copyright (C) 2009 CodeLutin
 *
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>. ##%*/

package org.chorem.pollen.ui.pages.admin;

import java.util.HashMap;
import java.util.List;
import java.util.Map;

import org.apache.tapestry5.BindingConstants;
import org.apache.tapestry5.ComponentResources;
import org.apache.tapestry5.ValueEncoder;
import org.apache.tapestry5.annotations.Component;
import org.apache.tapestry5.annotations.IncludeStylesheet;
import org.apache.tapestry5.annotations.Parameter;
import org.apache.tapestry5.annotations.Persist;
import org.apache.tapestry5.annotations.Property;
import org.apache.tapestry5.annotations.SessionState;
import org.apache.tapestry5.ioc.Messages;
import org.apache.tapestry5.ioc.annotations.Inject;
import org.chorem.pollen.business.business.PreventRuleManager;
import org.chorem.pollen.business.dto.UserDTO;
import org.chorem.pollen.business.services.ServiceList;
import org.chorem.pollen.business.services.ServiceUser;
import org.chorem.pollen.business.utils.MD5;
import org.chorem.pollen.ui.components.FeedBack;
import org.chorem.pollen.ui.data.Lien;
import org.chorem.pollen.ui.services.Configuration;
import org.chorem.pollen.ui.utils.PasswordGenerator;

/**
 * Classe d'administration des comptes utilisateurs. Si l'utilisateur n'est pas
 * identifié, on lui demande d'abord de s'identifier via un formulaire.
 *
 * @author rannou
 * @version $Id: UsersAdmin.java 2758 2009-08-27 09:58:50Z nrannou $
 */
@IncludeStylesheet("context:css/usersAdmin.css")
public class UsersAdmin {

    /** Affichage des messages pour l'utilisateur */
    @Component(id = "feedback")
    private FeedBack feedback;

    /**
     * Objet de session représentant l'url du site.
     */
    @SessionState
    private String siteURL;

    /**
     * Objet de session représentant l'utilisateur identifié.
     */
    @SessionState
    @Property
    private UserDTO user;
    @Property
    private boolean userExists;

    /**
     * Titre de la page.
     */
    @Parameter(defaultPrefix = BindingConstants.MESSAGE, value = "title")
    @Property
    private String title;

    /**
     * Adresse de la page.
     */
    @SuppressWarnings("unused")
    @Property
    private Lien[] address;

    /**
     * Liste des utilisateurs.
     */
    @Property
    @Persist
    private List<UserDTO> accounts;

    /**
     * Utilisateur courant.
     */
    @Property
    private UserDTO account;

    /**
     * Ressources de la page.
     */
    @Inject
    private ComponentResources resources;

    /**
     * Messages.
     */
    @Inject
    private Messages messages;

    /**
     * Service contenant la configuration de l'application.
     */
    @Inject
    private Configuration conf;

    /** Injection des services */
    @Inject
    private ServiceUser serviceUser;
    @Inject
    private ServiceList serviceList;

    /**
     * Initialisation du formulaire.
     */
    public void onPrepareFromUsersForm() {
        if (accounts == null) {
            accounts = serviceUser.selectUsers(null);
        }
    }

    /**
     * Méthode appelée lors de la soumission du formulaire.
     */
    public Object onSuccessFromUsersForm() {
        for (UserDTO usr : accounts) {
            if (serviceUser.findUserById(usr.getId()) != null) {
                serviceUser.updateUser(usr);
            } else {
                String password = PasswordGenerator.generate(8);
                if (serviceUser.createUser(usr, MD5.encode(password)) != null) {
                    feedback.addInfo(messages.format("userCreated", usr
                            .getLogin()));
                    sendMailNotification(usr, password);
                } else {
                    feedback.addError(messages.format("userNotCreated", usr
                            .getLogin()));
                }
            }
        }
        feedback.addInfo(messages.get("dataSaved"));
        resources.discardPersistentFieldChanges();
        return this;
    }

    /**
     * Méthode appelée lors de l'ajout d'un utilisateur.
     */
    Object onAddRowFromUsersLoop() {
        UserDTO usr = new UserDTO();
        usr.setId(String.valueOf(System.nanoTime()));
        accounts.add(usr);
        return usr;
    }

    /**
     * Méthode appelée lors de la suppression d'un utilisateur.
     */
    void onRemoveRowFromUsersLoop(UserDTO usr) {
        if (serviceUser.findUserById(usr.getId()) != null) {

            // suppression des listes de votants de l'utilisateur
            serviceList.deletePersonLists(usr.getId());

            // suppression de l'utilisateur
            if (serviceUser.deleteUser(usr)) {
                accounts.remove(usr);
            }
        } else {
            accounts.remove(usr);
        }
    }

    /**
     * Encoder permettant de transformer un ID en user.
     */
    public ValueEncoder<UserDTO> getUserEncoder() {
        return new ValueEncoder<UserDTO>() {

            public String toClient(UserDTO value) {
                return value.getId();
            }

            public UserDTO toValue(String key) {
                for (UserDTO usr : accounts) {
                    if (key.equals(usr.getId())) {
                        return usr;
                    }
                }
                throw new IllegalArgumentException("La clé \"" + key
                        + "\" n'existe pas dans la collection : " + accounts);
            }
        };
    }

    /** Envoi du mail de notification */
    private void sendMailNotification(UserDTO newUser, String password) {
        Map<String, String> data = new HashMap<String, String>();
        data.put("host", conf.getProperty("email_host"));
        data.put("port", conf.getProperty("email_port"));
        data.put("from", conf.getProperty("email_from"));
        data.put("to", newUser.getEmail());
        data.put("title", messages.format("registerEmail_subject", newUser
                .getLogin()));
        data.put("msg", messages.format("registerEmail_msg", newUser
                .getFirstName(), newUser.getLastName(), newUser.getLogin(),
                password, siteURL));

        PreventRuleManager.emailAction(data);
    }

    /**
     * Détermine si le compte courant est le compte de l'utilisateur.
     */
    public boolean isMyAccount() {
        if (userExists) {
            return user.getId().equals(account.getId());
        }
        return false;
    }

    /**
     * Méthode d'activation de la page.
     */
    public void onActivate() {

        // Affichage des erreurs
        if (userExists && !user.isAdministrator()) {
            feedback.addError(messages.get("userNotAllowed"));
        }
    }

    /**
     * Initialisation de l'affichage.
     */
    public void setupRender() {
        accounts = serviceUser.selectUsers(null);
        
        address = new Lien[] { new Lien("Pollen", "Index"),
                new Lien(title, null) };
    }
}
