/*
 * #%L
 * Wao :: Web Interface
 * 
 * $Id: Contacts.java 1561 2012-03-13 10:57:45Z bleny $
 * $HeadURL: http://svn.forge.codelutin.com/svn/wao/tags/wao-3.4/wao-ui/src/main/java/fr/ifremer/wao/ui/pages/Contacts.java $
 * %%
 * Copyright (C) 2009 - 2010 Ifremer
 * %%
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Affero General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU Affero General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 * #L%
 */

package fr.ifremer.wao.ui.pages;

import fr.ifremer.wao.WaoBusinessException;
import fr.ifremer.wao.WaoException;
import fr.ifremer.wao.WaoUtils;
import fr.ifremer.wao.bean.ConnectedUser;
import fr.ifremer.wao.bean.ContactFilter;
import fr.ifremer.wao.bean.ContactState;
import fr.ifremer.wao.bean.ObsProgram;
import fr.ifremer.wao.bean.UserRole;
import fr.ifremer.wao.bean.ValidationResult;
import fr.ifremer.wao.entity.Contact;
import fr.ifremer.wao.entity.WaoUser;
import fr.ifremer.wao.io.ImportResults;
import fr.ifremer.wao.service.ServiceBoat;
import fr.ifremer.wao.service.ServiceContact;
import fr.ifremer.wao.service.ServiceReferential;
import fr.ifremer.wao.service.ServiceUser;
import fr.ifremer.wao.ui.components.ContactFilterComponent;
import fr.ifremer.wao.ui.components.Layout;
import fr.ifremer.wao.ui.data.ContactDataSource;
import fr.ifremer.wao.ui.data.ExportStreamResponse;
import fr.ifremer.wao.ui.data.GenericSelectModel;
import fr.ifremer.wao.ui.data.RequiresAuthentication;
import fr.ifremer.wao.ui.services.ContactModelFactory;
import fr.ifremer.wao.ui.services.WaoManager;
import org.apache.commons.lang3.BooleanUtils;
import org.apache.tapestry5.ComponentResources;
import org.apache.tapestry5.StreamResponse;
import org.apache.tapestry5.annotations.Environmental;
import org.apache.tapestry5.annotations.Import;
import org.apache.tapestry5.annotations.InjectComponent;
import org.apache.tapestry5.annotations.Log;
import org.apache.tapestry5.annotations.Persist;
import org.apache.tapestry5.annotations.Property;
import org.apache.tapestry5.annotations.SessionState;
import org.apache.tapestry5.beaneditor.BeanModel;
import org.apache.tapestry5.corelib.components.Zone;
import org.apache.tapestry5.ioc.Messages;
import org.apache.tapestry5.ioc.annotations.Inject;
import org.apache.tapestry5.ioc.services.PropertyAccess;
import org.apache.tapestry5.services.BeanModelSource;
import org.apache.tapestry5.services.javascript.JavaScriptSupport;
import org.apache.tapestry5.upload.services.UploadedFile;
import org.slf4j.Logger;

import java.io.IOException;
import java.io.InputStream;
import java.text.DateFormat;
import java.text.SimpleDateFormat;
import java.util.List;

/**
 * Contacts
 *
 * Created: 9 nov. 2009
 *
 * @author fdesbois <fdesbois@codelutin.com>
 * @version $Id: Contacts.java 1561 2012-03-13 10:57:45Z bleny $
 */
@SuppressWarnings({"UnusedDeclaration"})
@RequiresAuthentication(allowedRoles = {UserRole.ADMIN, UserRole.COORDINATOR, UserRole.OBSERVER, UserRole.PROFESSIONAL})
@Import(stylesheet = "context:css/contacts.css")
public class Contacts {

    @Inject
    private Logger logger;

    @InjectComponent
    private Layout layout;

    @SessionState
    @Property
    private ConnectedUser user;

    @Inject
    private ServiceBoat serviceBoat;

    @Inject
    private ServiceContact serviceContact;

    @Inject
    private ServiceReferential serviceReferential;

    @Environmental
    private JavaScriptSupport javascriptSupport;

    @Inject
    private WaoManager manager;

    @Inject
    private Messages messages;

    @InjectComponent
    private ContactFilterComponent filterComponent;

    public ServiceContact getServiceContact() {
        return serviceContact;
    }

    public String getLabelForEnum(Enum value) {
        return messages.get(value.getDeclaringClass().getSimpleName() + "." + value.name());
    }

    @Log
    void setupRender() throws WaoException {
        contacts = null;
        // Initialize fullView to true for admin user
        if (fullView == null) {
            fullView = user.isAdminOrProfessional();
        }
    }

    @Log
    void afterRender() {
        contactSelectedId = null;
    }

    /**************************** CONTACT FILTERS *****************************/

    @Persist
    private ContactFilter contactFilter;

//    @InjectComponent
//    private Zone filtersZone;

    @InjectComponent
    private Zone importExportZone;

    private boolean reset;

    @Property
    private ContactState stateFilter;

    public ContactFilter getContactFilter() throws WaoException {
        return filterComponent.getFilter();
    }

    public boolean isFiltersVisible() {
//        boolean companyFiltered = getContactFilter().getCompany() != null &&
//                                        user.isAdmin();
//        return (getContactFilter().isFiltered() || companyFiltered) &&
//                        StringUtils.isEmpty(getContactFilter().getBoatName());
        return filterComponent.getFiltersVisible();
    }

    protected ContactFilter getFilter() throws WaoException {
        return getContactFilter();
    }

    Object onActionFromShowImportExport() {
        return importExportZone.getBody();
    }

    Object onActionFromShowFilters() {

        filterComponent.switchFiltersVisible();

        // return filterComponent.getFiltersZone();
        return filterComponent;
    }

    void onSelectedFromReset() {
        reset = true;
    }

//    @Log
//    Object onSuccessFromFiltersForm() throws WaoException {
//        if (StringUtils.isNotEmpty(contactStateMotifId)) {
//            getContactFilter().setContactStateMotif(getContactStateMotifSelectModel().findObject(contactStateMotifId));
//        }
//        if (isEdited()) {
//            return filtersZone.getBody();
//        }
//        if (reset) {
//            contactFilter = null;
//        }
////        else {
////            filtersVisible = true;
////        }
//        return this;
//    }

//    public void onChangeFromSampleRow(String sampleRowId) {
//        getFilter().setSampleRow(getSampleRowSelectModel().findObject(sampleRowId));
//        if (logger.isDebugEnabled()) {
//            logger.debug("change sample row code in filter to " + getFilter().getSampleRow());
//        }
//    }

//    @Property
//    @Persist
//    private String terrestrialLocationId;

//    public Zone onValueChangedTerrestrialDistrict(String terrestrialDistrictId) {
//        updateLocation(terrestrialDistrictId);
//        return filtersZone;
//    }

//    private GenericSelectModel<TerrestrialLocation> terrestrialLocationSelectModel;
//
//    public GenericSelectModel<TerrestrialLocation> getTerrestrialLocationSelectModel() {
//        List<TerrestrialLocation> locations = new ArrayList<TerrestrialLocation>();
//        if (getFilter().getTerrestrialDistrict() != null) {
//            // it may occurs if the user reset the filters
//            serviceReferential.getAllTerrestrialLocations(getContactFilter(),
//                                   getFilter().getTerrestrialDistrict().getDistrictCode());
//        }
//        terrestrialLocationSelectModel = new GenericSelectModel<TerrestrialLocation>(locations,
//                TerrestrialLocation.class, "description", TerrestrialLocation.TOPIA_ID, propertyAccess);
//        return terrestrialLocationSelectModel;
//    }

//    public SelectModel getContactStateSelectModel() {
//        List<ContactState> allowedStates = ContactState.getAllowedStates(user.getProfile().getObsProgram());
//        return new EnumSelectModel(ContactState.class, messages,
//                                 // just convert list to array
//                                 allowedStates.toArray(new ContactState[allowedStates.size()]));
//    }

//    @Property
//    @Persist
//    private String contactStateMotifId;
//
//    private GenericSelectModel<ContactStateMotif> contactStateMotifSelectModel;
//
//    public GenericSelectModel<ContactStateMotif> getContactStateMotifSelectModel() {
//        if (contactStateMotifSelectModel == null) {
//            if (contactFilter.getState() != null) {
//                List<ContactStateMotif> motifs = serviceReferential.getAllContactStateMotifs(contactFilter.getState());
//                if (motifs.size() != 0) {
//                    contactStateMotifSelectModel = new GenericSelectModel<ContactStateMotif>(
//                            motifs, ContactStateMotif.class, "translation", ContactStateMotif.TOPIA_ID, propertyAccess);
//                }
//            }
//        }
//        return contactStateMotifSelectModel;
//    }

//    public Zone onValueChangedFromState(ContactState contactState) {
//        contactFilter.setState(contactState);
//        // refresh
//        contactStateMotifSelectModel = null;
//        contactStateMotifId = null;
//        return filtersZone;
//    }

    public String[] onProvideCompletionsFromBoatName(String input) throws WaoException {
        List<String> results = serviceBoat.getBoatNamesStartWith(input);
        return results.toArray(new String[results.size()]);
    }

    /**************************** CONTACT IMPORT/EXPORT ***********************/

    @Property
    private UploadedFile contactsCsvFile;

    /**
     * Only administrator with no readOnly rights
     * can import/export contacts.
     *
     * @return true if import/export of contacts can be done
     */
    public boolean canImportExport() {
        return (user.isAdmin() || user.isCoordinator()) && ! user.isReadOnly();
    }

    @Log
    void onSuccessFromImportContacts() throws WaoException {
        if (canImportExport()) {
            try {
                ImportResults result = serviceContact.importContactCsv(user,
                        contactsCsvFile.getStream());
                // Suppress persistent list of contacts
                contacts = null;
                layout.addInfo(result.getNbRowsImported() + " contacts " +
                        "importés,  " + result.getNbRowsRefused() +
                        " refusés.");
                for (String error : result.getErrors()) {
                    layout.addInfo(error);
                }
            } catch (WaoBusinessException eee) {
                layout.addError(eee.getMessage());
            }
        }
    }

    StreamResponse onActionFromExportShowContacts() {
        if (canImportExport()) {
            return new ExportStreamResponse("wao-contacts") {

                @Override
                public InputStream getStream() throws IOException {
                    InputStream result;
                    try {
                        result = serviceContact.exportContactCsv(
                                getContactFilter());
                    } catch (WaoException eee) {
                        throw new IOException(eee);
                    }
                    return result;
                }
            };
        }
        return null;
    }

    /**************************** CONTACT DISPLAY MODE ************************/

    @Persist
    @Property
    private Boolean fullView;

    /**
     * ACTION:: Used to change the display mode of the contacts table.
     * This change affect the loading of the css style over the main table.
     *
     * @see #getGridClass()
     */
    void onActionFromToggleDisplayMode() {
        fullView = !fullView;
    }

    /**************************** CONTACT LIST ********************************/

    @Inject
    private ServiceUser serviceUser;

    @Inject
    private BeanModelSource beanModelSource;

    @Inject
    private ComponentResources resources;

    @Persist
    private ContactDataSource contacts;

    @Property
    private Contact contact;

    private BeanModel<Contact> contactModel;

    @Inject
    private PropertyAccess propertyAccess;

    private GenericSelectModel<WaoUser> userSelectModel;

    @Persist
    private String contactSelectedId;

    public void setContactSelectedId(String contactSelectedId) {
        this.contactSelectedId = contactSelectedId;
    }

    @Property
    @Persist
    private String contactUserId;

    private boolean even = true;

    public ContactDataSource getContacts() throws WaoException {
        if (contacts == null) {
            if (logger.isDebugEnabled()) {
                logger.debug("Create DataSource");
            }
            contacts = new ContactDataSource(getContactFilter(), serviceContact);
        }
        return contacts;
    }

    public BeanModel<Contact> getContactModel() {
        if (contactModel == null) {
            ContactModelFactory contactModelFactory = new ContactModelFactory(
                    user.getProfile().getObsProgram(), user.getProfile().getUserRole(), fullView);

            contactModel = contactModelFactory.newModel(beanModelSource, resources);
        }
        return contactModel;
    }

    public GenericSelectModel<WaoUser> getUserSelectModel() {
        if (userSelectModel == null) {
            List<WaoUser> users = serviceUser.getObservers(
                    user.getCompany(), true);
            if (logger.isDebugEnabled()) {
                logger.debug("Nb users : " + users.size());
            }
            userSelectModel = new GenericSelectModel<WaoUser>(users,
                    WaoUser.class, "fullName", "id", propertyAccess);
        }
        return userSelectModel;
    }

    public String getCommentDisplayText(String comment) {
        if (comment != null && comment.length() > 20) {
            return comment.substring(0, 20) + "...";
        }
        return comment;
    }

    public String getCommentTooltip(String comment) {
        return manager.getTooltipText(comment);
    }

    public String getSampleRowDescription() {
        return manager.getTooltipSampleRow(contact.getSampleRow());
    }

    public String getMammalsInfo() {
        return WaoUtils.escapeForToolTip(contact.getMammalsInfo());
    }

    public String getBoatDescription() {
        return manager.getTooltipBoat(contact.getBoat());
    }

    public String getSecondaryObservers() {
        StringBuilder result = new StringBuilder();
        result.append(contact.getMainObserver().getFullName());
        for (WaoUser observer : contact.getSecondaryObservers()) {
            result.append(", ");
            result.append(observer.getFullName());
        }
        return result.toString();
    }

    public String getTooltipExportFrom() throws WaoException {
        if (getContactFilter().getSamplingFilter().getFromDate() != null) {
            return "depuis le " + getDateFormat().format(getContactFilter().getSamplingFilter().getFromDate());
        }
        return "";
    }

    public String getGridClass() {
        return fullView ? "admin" : "user";
    }

    public String getRowClass() {
        String result = manager.getContactStyle(contact, user.isAdmin());
        if (contact.getTopiaId().equals(contactSelectedId)) {
            result = "selected";
        }
        return result;
    }

    public DateFormat getDateFormat() {
        return new SimpleDateFormat("dd/MM/yyyy");
    }

    public DateFormat getDateTimeFormat() {
        return new SimpleDateFormat("dd/MM/yyyy HH:mm");
    }

    public boolean isEmpty(Boolean validation) {
        return validation == null;
    }

    public boolean isBoardingDone() {
        return contact.getContactState() == ContactState.OBSERVATION_DONE;
    }

    /**************************** CONTACT ROW ACTION **************************/

    public boolean hasActions() {
        return !user.isAdmin() && contact.getValidationCompany() == null
                && !user.isReadOnly();
    }

    /**
     * Display validation actions. Evo #2063 : only coordinator can validate
     * for a company.
     *
     * @return true if the validation actions can be displayed
     */
    public boolean hasValidationActions() {
        return (user.isAdmin() || user.isCoordinator()) && ! user.isReadOnly();
    }

    public boolean canEditSampleRow() {
        boolean canEdit = user.getProfile().getCanWrite() &&

                // user is admin and no validation program
                (user.isAdminAndCanWrite() && contact.getValidationProgram() == null

                || // or
                // user is coordinator or observer
                user.isCoordinatorOrObserver() && contact.getValidationCompany() == null);
        return canEdit;
    }

    public boolean canDeleteSampleRow() {
        boolean noValidation = contact.getValidationProgram() == null
                            && contact.getValidationCompany() == null;
        return noValidation && user.getProfile().getCanWrite();
    }

    public boolean canValidate() {
        // Can't validate during edition of the contact row
        switch (user.getRole()) {
            case ADMIN:
                return contact.getValidationProgram() == null &&
                        BooleanUtils.isTrue(contact.getValidationCompany());
            // Evo #2063 : only coordinator can validate
            case COORDINATOR:
                boolean requiredDataAreFilled;
                if (contact.getObsProgram() == ObsProgram.OBSMER) {
                    requiredDataAreFilled = contact.getRestitution() != null;
                } else {
                    requiredDataAreFilled = contact.getDataInputDate() != null;
                }
                ContactState state = contact.getContactState();
                boolean observationDone =
                        state.equals(ContactState.OBSERVATION_DONE) &&
                        requiredDataAreFilled;
                return contact.getValidationCompany() == null &&
                        (state.isUnfinishedState() || observationDone);
            default:
                return false;
        }
    }

    public boolean canUnvalidate() {
        switch (user.getRole()) {
            case ADMIN:
                return contact.getValidationProgram() != null;
            // Evo #2063 : only coordinator can unvalidate
            case COORDINATOR:
                return contact.getValidationCompany() != null &&
                    contact.getValidationProgram() == null;
            default:
                return false;
        }
    }

    /****************************************** CONTACT ACTIONS *******************************************************/

    private Contact modifiedContact;

    @Log
    void onActionFromAcceptContact(String contactId) throws WaoException {
        if (logger.isDebugEnabled()) {
            logger.debug("Accept contact : " + contactId);
        }
        contactSelectedId = contactId;
        modifiedContact = getContacts().get(contactId);
        if (user.isAdmin()) {
            modifiedContact.setValidationProgram(Boolean.TRUE);
        } else {
            modifiedContact.setValidationCompany(Boolean.TRUE);
        }
        saveContact(modifiedContact);
    }

    @Log
    void onActionFromUnvalidateContact(String contactId) throws WaoException {
        if (logger.isDebugEnabled()) {
            logger.debug("Unvalidate contact : " + contactId);
        }
        contactSelectedId = contactId;
        modifiedContact = getContacts().get(contactId);
        if (user.isAdmin()) {
            modifiedContact.setValidationProgram(null);
        } else {
            modifiedContact.setValidationCompany(null);
        }
        saveContact(modifiedContact);
    }

    @Log
    void onActionFromRefuseContact(String contactId) throws WaoException {
        if (logger.isDebugEnabled()) {
            logger.debug("Refuse contact : " + contactId);
        }
        contactSelectedId = contactId;
        modifiedContact = getContacts().get(contactId);
        if (user.isAdmin()) {
            modifiedContact.setValidationProgram(Boolean.FALSE);
        } else {
            modifiedContact.setValidationCompany(Boolean.FALSE);
        }
        saveContact(modifiedContact);
    }

    protected void saveContact(Contact contact) {
        ValidationResult validationResult = serviceContact.validateContact(contact);
        if (validationResult.isSuccess()) {
            serviceContact.saveContact(user, contact, false);
        } else {
            layout.addError(validationResult.getMessage());
        }
    }

    @Log
    void onActionFromDeleteContact(String contactId) throws WaoException {
        if (logger.isDebugEnabled()) {
            logger.debug("Delete contact : " + contactId);
        }
        contactSelectedId = contactId;
        modifiedContact = getContacts().get(contactId);
        serviceContact.saveContact(user, modifiedContact, true);
    }

}
