/*
 * #%L
 * Wao :: Web Interface
 * 
 * $Id: SamplingPlan.java 801 2010-11-25 15:30:03Z bleny $
 * $HeadURL: svn+ssh://bleny@labs.libre-entreprise.org/svnroot/suiviobsmer/tags/wao-1.6/wao-ui/src/main/java/fr/ifremer/wao/ui/pages/SamplingPlan.java $
 * %%
 * Copyright (C) 2009 - 2010 Ifremer
 * %%
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Affero General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU Affero General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 * #L%
 */

package fr.ifremer.wao.ui.pages;

import fr.ifremer.wao.WaoBusinessException;
import fr.ifremer.wao.WaoException;
import fr.ifremer.wao.bean.ConnectedUser;
import fr.ifremer.wao.bean.FacadeRow;
import fr.ifremer.wao.bean.SamplingFilter;
import fr.ifremer.wao.bean.SamplingFilterImpl;
import fr.ifremer.wao.entity.DCF5Code;
import fr.ifremer.wao.entity.FishingZone;
import fr.ifremer.wao.entity.SampleMonth;
import fr.ifremer.wao.entity.SampleRow;
import fr.ifremer.wao.service.ServiceReferential;
import fr.ifremer.wao.service.ServiceSampling;
import fr.ifremer.wao.ui.base.AbstractFilteredPage;
import fr.ifremer.wao.ui.components.Layout;
import fr.ifremer.wao.ui.data.ExportStreamResponse;
import fr.ifremer.wao.ui.data.RequiresAuthentication;
import fr.ifremer.wao.ui.services.WaoManager;
import org.apache.commons.lang.BooleanUtils;
import org.apache.tapestry5.StreamResponse;
import org.apache.tapestry5.annotations.IncludeStylesheet;
import org.apache.tapestry5.annotations.InjectComponent;
import org.apache.tapestry5.annotations.Persist;
import org.apache.tapestry5.annotations.Property;
import org.apache.tapestry5.annotations.SessionState;
import org.apache.tapestry5.corelib.components.Form;
import org.apache.tapestry5.corelib.components.Zone;
import org.apache.tapestry5.ioc.Messages;
import org.apache.tapestry5.ioc.annotations.Inject;
import org.nuiton.util.PeriodDates;
import org.nuiton.util.StringUtil;
import org.nuiton.util.StringUtil.ToString;
import org.slf4j.Logger;

import java.io.IOException;
import java.io.InputStream;
import java.text.DateFormat;
import java.text.NumberFormat;
import java.text.SimpleDateFormat;
import java.util.ArrayList;
import java.util.Date;
import java.util.HashMap;
import java.util.List;
import java.util.Map;

/**
 * SampingPlan
 *
 * Created: 9 nov. 2009
 *
 * @author fdesbois <fdesbois@codelutin.com>
 */
@RequiresAuthentication
@IncludeStylesheet("context:css/sampling.css")
public class SamplingPlan extends AbstractFilteredPage {

    @Inject
    private Logger logger;

    @Inject
    private WaoManager manager;

    @Inject
    private Messages messages;

    @Inject
    private ServiceSampling serviceSampling;

    @Inject
    private ServiceReferential serviceReferential;

    @InjectComponent
    private Layout layout;

    @SessionState
    @Property
    private ConnectedUser user;

    private long nbTidesExpectedTime;

    private long nbTidesRealTime;

    private long totalTidesExpectedTime;

    private long totalTidesRealTime;

    /**
     * Page initialization
     */
    void setupRender() {
        if (isFiltersVisible()) {
            // Initialize filters
            initSelectFilters(true, false, true);
        }
        // Initialize fullView depends on user admin role
        if (fullView == null) {
            fullView = user.isAdmin();
        }
        // The company of connected user will be contributed to abstractFilteredPage
        initCompanyFilter();
    }

    /**************************** IMPORT (ADMIN) *******************************/

//    /**
//     * Fichier CSV contenant un plan d'échantillonnage
//     */
//    @Property
//
//    public boolean canImportSamplingPlan() {
//        return user.isAdmin() && !user.isReadOnly();
//    }
//
//    @Log
//    void onSuccessFromImportSamplingPlan() throws WaoException {
//        if (canImportSamplingPlan()) {
//            try {
//                ImportResults result = serviceSampling.importSamplingPlanCsv(
//                        samplingPlanCsvFile.getStream());
//                layout.addInfo(result.getNbRowsImported() +
//                        " lignes du plan importés,  " +
//                        result.getNbRowsRefused() + " refusés.");
//                for (String error : result.getErrors()) {
//                    layout.addInfo(error);
//                }
//            } catch (WaoBusinessException eee) {
//                layout.addError(eee.getMessage());
//            }
//        }
//    }
    
    /**************************** EXPORT **************************************/

    StreamResponse onActionFromExportSamplingPlan() {
        return new ExportStreamResponse("wao-echantillonnage") {

            @Override
            public InputStream getStream() throws IOException {
                InputStream result = null;
                try {
                    result = serviceSampling.exportSamplingPlanCsv(
                                                    user, getFilter());
                } catch (WaoException eee) {
                    throw new IOException(eee);
                }
                return result;
            }
        };
    }

    /**************************** FILTERS *************************************/

    @Persist
    private SamplingFilter filter;

    @InjectComponent
    private Zone filtersZone;

    @InjectComponent
    private Form filtersForm;

    private boolean reset;

    public PeriodDates getPeriod() {
        return getFilter().getPeriod();
    }

    @Override
    public SamplingFilter getFilter() throws WaoException {
        if (filter == null) {
            filter = new SamplingFilterImpl();
            // Initialize period
            PeriodDates period = PeriodDates.createMonthsPeriodFromToday(11);
            filter.setPeriod(period);
        }
        return filter;
    }

    @Override
    protected boolean isAvailableDataForFiltersOnly() {
        return false;
    }

    @Persist
    private Boolean showFilters;

    public boolean isFiltersVisible() {
        if (showFilters == null) {
            showFilters = false;
        }
        return BooleanUtils.isTrue(showFilters);
    }

    Object onActionFromShowFilters() {
        showFilters = ! showFilters;

        // Initialize filters
        initSelectFilters(true, false, true);

        if (isFiltersVisible()) {
            return filtersZone.getBody();
        } else {
            return filtersZone;
        }
    }

    void onSelectedFromReset() {
        reset = true;
    }

    
    Object onSuccessFromFiltersForm() {
        if (isEdited()) {
            return filtersZone.getBody();
        }
        if (reset) {
            // Don't reset period in filters
            PeriodDates period = getFilter().getPeriod();
            filter = null;
            getFilter().setPeriod(period);
        }
        return this;
    }

    /**************************** MAIN ACTIONS ********************************/

    @Persist
    @Property
    private Boolean fullView;

    /**
     * ACTION:: Used to change the display mode of the samplingPlan table.
     * This change affect the loading of the css style over the main table.
     *
     * @see #getMainClass()
     */
    void onActionFromToggleDisplayMode() {
        fullView = !fullView;
    }

    void onActionFromChangeFilterEstimatedTides() {
        boolean oldValue = getFilter().getEstimatedTides();
        if (logger.isDebugEnabled()) {
            logger.debug("Change estimatedTides in filter to : " + !oldValue);
        }
        getFilter().setEstimatedTides(!oldValue);
    }

    /**************************** MAIN TABLE **********************************/

    /** ------------------------- DATA ------------------------------------- **/

    /**
     * Main data for samplingPlan : List of SampleRow ordered by FishingZone.
     */
//    @Persist
    private List<SampleRow> data;

    private List<Date> months;

    @Property
    private Date month;

    /* variable used in template */
    @Property
    private DCF5Code dcf5code;

    /**
     * Current SampleRow from loop
     */
    @Property
    private SampleRow row;

    @Property
    private String currentFacadeName;

    @Property
    private String currentSectorName;

    /**
     * Return List of SampleRow from suiviobsmer-business
     * @return List of SampleRow
     * @throws WaoException
     */
    public List<SampleRow> getData() throws WaoException {
        if (data == null) {
            if (logger.isDebugEnabled()) {
                logger.debug("sampleRows filter sent to service : " +
                getFilter().getSampleRows());
            }

            FacadeRow facade =
                    serviceSampling.getSampleRowsOrderedByFishingZone(getFilter());
            data = facade.getValues();
        }
        return data;
    }

    public List<Date> getMonths() {
        if (months == null) {
            months = getPeriod().getMonths();
        }
        return months;
    }

    public boolean isFacadeChanged() {
        if (!row.getFacade().equals(currentFacadeName)) {
            currentFacadeName = row.getFacade();
            return true;
        }
        return false;
    }

    public boolean isSectorChanged() {
        if (!row.getSectors().equals(currentSectorName)) {
            currentSectorName = row.getSectors();
            return true;
        }
        return false;
    }

    public String getCompanyName() {
        return row.getCompany() != null ? row.getCompany().getName() : "";
    }

    public boolean isCurrentMonth() {
        String currentStr = getDateFormat().format(new Date());
        String monthStr = getDateFormat().format(month);
        return currentStr.equals(monthStr);
    }

    public Integer getNbTidesExpected() {
        Integer result = null;
        SampleMonth sampleMonth = row.getSampleMonth(month);
        if (sampleMonth != null) {
            result = sampleMonth.getExpectedTidesValue();
        }
        return result;
    }

    private static final ToString<FishingZone> FISHING_ZONE_TO_STRING_DISTRICT =
            new ToString<FishingZone>() {

        @Override
        public String toString(FishingZone o) {
            return o.getDistrictCode();
        }
    };

    public String getFishingZones() {
        return StringUtil.join(row.getFishingZone(),
                FISHING_ZONE_TO_STRING_DISTRICT, ", ", false);
//        String result = "";
//        for (FishingZone zone : row.getFishingZone()) {
//            result += zone.getDistrictCode() + ", ";
//        }
//        return result.substring(0, result.length()-2);

    }

    public Integer getNbTidesReal() {
        Integer result = null;
        SampleMonth sampleMonth = row.getSampleMonth(month);
        if (sampleMonth != null) {
            result = getFilter().getEstimatedTides() ?
                sampleMonth.getEstimatedTidesValue() :
                sampleMonth.getRealTidesValue();
        }
        return result;
    }

    public boolean hasNbTidesReal() {
        Date current = new Date();
        boolean validMonth = month.before(current) || isCurrentMonth();
        return validMonth && getNbTidesReal() != null;
    }

    public boolean canDisplayTidesReal() {
        // Evo #2227 : Guest users can't have access to real tides values
        boolean result = hasNbTidesReal() && !user.isGuest();
        return result;
    }

    public String getTotalPercentage() {
        double result = 0;
        if (row.getTotalTidesExpected() > 0) {
            result = (double)row.getTotalTidesReal() / (double)row.getTotalTidesExpected();
        }
        return NumberFormat.getPercentInstance().format(result);
    }

    public NumberFormat getNumberFormat() {
        return NumberFormat.getNumberInstance();
    }

    private Map<Date, Integer> totalExpectedForMonths;
    private Map<Date, Integer> totalRealForMonths;
    private Integer highTotalExpected;
    private Integer highTotalReal;

    /** Set totalExpectedForMonths, totalRealForMonths, highTotalExpected and
     * highTotalReal values.
     */
    protected void setTotalsForMonths() {

        // to prevent a side-effect, keep month value
        Date keptMonth = month;

        totalExpectedForMonths = new HashMap<Date, Integer>();
        totalRealForMonths = new HashMap<Date, Integer>();

        // high totals is the total... of the totals, it will appear
        // in the bottom right, as the last element of the "totals" line
        highTotalExpected = 0;
        highTotalReal = 0;

        for (Date month : getMonths()) {
            this.month = month;

            Integer totalExpected = 0, totalReal = 0;

            for (SampleRow row : getData()) {
                this.row = row;

                Integer expected = getNbTidesExpected(), real = getNbTidesReal();
                if (logger.isDebugEnabled()) {
                    logger.debug( String.format("for month %s and row %s : expected = %s, real = %s",
                                  getDateFormat().format(month), row.getCode(), expected, real));
                }

                if (expected != null) {
                    totalExpected += expected;
                }
                if (real != null && canDisplayTidesReal()) {
                    totalReal += real;
                }
            }

            totalExpectedForMonths.put(month, totalExpected);
            totalRealForMonths.put(month, totalReal);

            if (logger.isDebugEnabled()) {
                logger.debug( String.format("totals for month %s : expected = %s, actual = %s",
                              getDateFormat().format(month), totalExpected, totalReal));
            }

            highTotalExpected += totalExpected;
            highTotalReal += totalReal;

        }
        
        if (logger.isDebugEnabled()) {
            for (Map.Entry<Date, Integer> expected : totalExpectedForMonths.entrySet()) {
                logger.debug( String.format("totals for month %s : expected = %s, actual = %s",
                        getDateFormat().format(expected.getKey()), expected.getValue(),
                        totalRealForMonths.get(expected.getKey())));
            }
        }

        // to prevent side-effect, restore month value
        month = keptMonth;
    }

    public Integer getTotalExpectedForMonth() {
        if (totalExpectedForMonths == null) {
            setTotalsForMonths();
        }
        Integer total = totalExpectedForMonths.get(month);
        if (logger.isDebugEnabled()) {
            logger.debug(String.format("total for month %s is %s", getDateFormat().format(month), total));
        }
        return total;
    }

    public Integer getTotalRealForMonth() {
        if (totalExpectedForMonths == null) {
            setTotalsForMonths();
        }
        return totalRealForMonths.get(month);
    }

    public String getRatioForMonth() {
        String ratio = "";
        if (getTotalExpectedForMonth() > 0) {
            double percent = ((double) getTotalRealForMonth() / getTotalExpectedForMonth());
            ratio = NumberFormat.getPercentInstance().format(percent);
        }
        return ratio;
    }

    public Integer getHighTotalExpected() {
        if (highTotalExpected == null) {
            setTotalsForMonths();
        }
        return highTotalExpected;
    }

    public Integer getHighTotalReal() {
        if (highTotalReal == null) {
            setTotalsForMonths();
        }
        return highTotalReal;
    }

    public String getHighTotalRatio() {
        String ratio = "";
        if (getHighTotalExpected() > 0) {
            double percent = ((double) getHighTotalReal() / getHighTotalExpected());
            ratio = NumberFormat.getPercentInstance().format(percent);
        }
        return ratio;
    }

    /** ------------------------- HTML & STYLE ----------------------------- **/

    @Property
    private int rowIndex;

    public String getRowComment() {
        String comment = row.getComment();
        // Problem with " in chenilleKit ToolTip component
        comment = comment.replace("\n", "<br />");
        return comment;
    }

    public int getNbColumnsForProfession() {
        // code, program.name, fishingZone.districts, profession.code, profession.libelle
        int fixed = 5;
        if (fullView) {
            // company, program.periodBegin, program.periodEnd, fishingZonesInfos
            // profession.meshSize, profession.size, profession.other, profession.species
            fixed += 8;
        }
        return fixed;
    }

    public int getNbColumnsForMonths() {
        return getMonths().size() + 1;
    }

    public int getNbColumnsForOther() {
        return 4;
    }

    public int getNbColumnsTotal() {
        return getNbColumnsForProfession() +
                getNbColumnsForMonths() + getNbColumnsForOther();
    }

    public String getMainClass() {
        return fullView ? "admin" : "user";
    }

    public String getParityClass() {
        return rowIndex % 2 == 0 ? "even" : "odd";
    }

    public String getActionsClass() {
        return fullView ? "width100" : "width50";
    }

    public String getRealTidesClass() {
        String result = "real-warning";
        if (getNbTidesReal() < getNbTidesExpected()) {
            result += "-inf";
        } else if (getNbTidesReal() > getNbTidesExpected()) {
            result += "-sup";
        }
        return result;
    }

    public String getRealTidesClassForTotal() {
        String result = "real-warning";
        if (getTotalRealForMonth() < getTotalExpectedForMonth()) {
            result += "-inf";
        } else if (getTotalExpectedForMonth() > getTotalRealForMonth()) {
            result += "-sup";
        }
        return result;
    }

    public String getMonthCurrentClass() {
        return isCurrentMonth() ? "selected" : "";
    }

    public DateFormat getDateFormat() {
        return new SimpleDateFormat("MM/yyyy");
    }

    /** ------------------------- ACTIONS ---------------------------------- **/

    /**
     * Can edit the sampleRow. Only for admin with no readOnly rights.
     * 
     * @return true if the sampleRow can be edited.
     */
    public boolean canEditSampleRow() {
        return user.isAdmin() && !user.isReadOnly();
    }

    /**
     * Used to filter period using dates from the program selected in table.
     * @param rowIndex index of the row in the table
     * @throws WaoException for a data problem
     */
    void onActionFromFilterPeriodDates(int rowIndex) throws WaoException {
        row = getData().get(rowIndex);
        getPeriod().setFromDate(row.getPeriodBegin());
        getPeriod().setThruDate(row.getPeriodEnd());
    }

    void onActionFromDeleteSampleRow(int rowIndex) throws WaoException {
        row = getData().get(rowIndex);
        try {
            serviceSampling.deleteSampleRow(row);
        } catch (WaoBusinessException eee) {
            String error = manager.getErrorMessage(eee, messages, logger);
            layout.addError(error);
        }
    }

    public boolean isRowNotFinished() {
        // Test if the row isFinished with a gap of 1 month (today - 1 month)
        return !row.isFinished(-1);
    }

    public boolean isRowModified() {
        // XXX bleny 20101105 this is a dirty hack. Actually, we should test
        // CollectionUtils.isEmpty(row.getSampleRowLog())
        // but it's not initialized by hibernate
        return row.getTopiaVersion() > 1;
    }
        
}
