/*
 * *##% 
 * SuiviObsmer :: Web Interface
 * Copyright (C) 2009 - 2010 Ifremer
 *
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as
 * published by the Free Software Foundation, either version 3 of the
 * License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Lesser Public License for more details.
 *
 * You should have received a copy of the GNU General Public
 * License along with this program.  If not, see
 * <http://www.gnu.org/licenses/gpl-3.0.html>.
 * ##%*
 */

package fr.ifremer.suiviobsmer.ui.pages;

import fr.ifremer.suiviobsmer.SuiviObsmerBusinessException;
import fr.ifremer.suiviobsmer.SuiviObsmerException;
import fr.ifremer.suiviobsmer.entity.Company;
import fr.ifremer.suiviobsmer.entity.CompanyImpl;
import fr.ifremer.suiviobsmer.entity.WaoUser;
import fr.ifremer.suiviobsmer.entity.WaoUserImpl;
import fr.ifremer.suiviobsmer.services.ServiceReferential;
import fr.ifremer.suiviobsmer.services.ServiceUser;
import fr.ifremer.suiviobsmer.ui.base.GenericSelectModel;
import fr.ifremer.suiviobsmer.ui.base.SuiviObsmerPage;
import fr.ifremer.suiviobsmer.ui.components.Layout;
import java.util.List;
import org.apache.commons.lang.StringUtils;
import org.apache.tapestry5.EventContext;
import org.apache.tapestry5.annotations.IncludeStylesheet;
import org.apache.tapestry5.annotations.InjectComponent;
import org.apache.tapestry5.annotations.Log;
import org.apache.tapestry5.annotations.Persist;
import org.apache.tapestry5.annotations.Property;
import org.apache.tapestry5.ioc.annotations.Inject;
import org.apache.tapestry5.ioc.services.PropertyAccess;
import org.apache.tapestry5.ioc.services.TypeCoercer;
import org.apache.tapestry5.upload.services.UploadedFile;
import org.slf4j.Logger;

/**
 * Administration
 *
 * Created: 9 nov. 2009
 *
 * @author fdesbois
 * @version $Revision: 330 $
 *
 * Mise a jour: $Date: 2010-02-05 15:04:29 +0100 (ven. 05 févr. 2010) $
 * par : $Author: fdesbois $
 */
@IncludeStylesheet("context:css/administration.css")
public class Administration implements SuiviObsmerPage {

    @Override
    public boolean isOnlyForAdmin() {
        return true;
    }

    @Inject
    private Logger log;

    @InjectComponent
    private Layout layout;

    @Inject
    private ServiceUser serviceUser;

    @Inject
    private ServiceReferential serviceReferential;

    @Inject
    private TypeCoercer typeCoercer;

    @Property
    private String companyId;

    @Property
    private String userId;

    @Persist
    private List<Company> companies;

    @Inject
    private PropertyAccess propertyAccess;

    private GenericSelectModel<Company> companiesSelectModel;

    private Company company;

    private GenericSelectModel<WaoUser> usersSelectModel;

    /**
     * User is persistant cause of autobuild problem ?!?
     */
    @Persist
    private WaoUser user;

    @Property
    private boolean generatePassword;

    @Property
    private String password;

    private boolean addNewCompanySelected;
    private boolean addNewUserSelected;
    private boolean deleteUserSelected;

    void setupRender() {
        companies = null;
        getCompanies();
        user = null;
    }

    void onActivate(EventContext ec) {
        if (ec.getCount() > 0) {
            companyId = ec.get(String.class, 0);
            if (ec.getCount() > 1) {
                userId = ec.get(String.class, 1);
            }
        }
    }

    Object[] onPassivate() {
        return new String[] {companyId, userId};
    }

   /**************************** IMPORT (ADMIN) *******************************/

    /**
     * CSV File which contains fishing zones
     */
    @Property
    private UploadedFile fishingZoneCsvFile;

    @Log
    void onSuccessFromImportFishingZones() {
        try {
            int result = serviceReferential.importFishingZoneCsv(fishingZoneCsvFile.getStream());
            layout.getFeedBack().addInfo(result + " zones de pêche importés");
        } catch (SuiviObsmerException eee) {
            layout.getFeedBack().addError(eee.getMessage());
        }
    }

    /**************************** FORMS ****************************************/

    public List<Company> getCompanies() {
        if (companies == null) {
            try {
                if (log.isInfoEnabled()) {
                    log.info("BUSINESS REQUEST [getCompanies]");
                }
                companies = serviceUser.getCompanies(false);
            } catch(SuiviObsmerException eee) {
                // errors
            }
        }
        return companies;
    }

    public GenericSelectModel<Company> getCompaniesSelectModel() {
        if (companiesSelectModel == null) {
            companiesSelectModel = new GenericSelectModel<Company>(getCompanies(), Company.class, Company.NAME, "id", propertyAccess);
        }
        return companiesSelectModel;
    }

    public Company getCompany() {
        if (company == null) {
            if (companyId != null) {
                company = getCompaniesSelectModel().findObject(companyId);
            } else {
                company = new CompanyImpl();
            }
        }
        return company;
    }

    public boolean isUsersAvailable() {
        return !StringUtils.isEmpty(getCompany().getId()) && getCompany().getActive();
    }

    void onSelectedFromAddNewCompany() {
        addNewCompanySelected = true;
    }

    void onSuccessFromActionsForm() {
        if (addNewCompanySelected) {
            companyId = null;            
        }
        userId = null;
        user = null;
    }

    void onSuccessFromCompany() throws SuiviObsmerException {
        serviceUser.createUpdateCompany(company);
        companyId = company.getId();
    }

    @Log
    public GenericSelectModel<WaoUser> getUsersSelectModel() {
        if (usersSelectModel == null) {
            List<WaoUser> users = getCompany().getWaoUser();
            if (log.isDebugEnabled()) {
                log.debug("Nb users : " + users.size());
            }
            usersSelectModel = new GenericSelectModel<WaoUser>(users, WaoUser.class, "fullName", "id", propertyAccess);
        }
        return usersSelectModel;
    }

    @Log
    public WaoUser getUser() {
        if (user == null) {
            if (userId != null) {
                if (log.isDebugEnabled()) {
                    log.debug("User exist in selectModel : " + userId);
                }
                user = getUsersSelectModel().findObject(userId);
            } else {
                user = new WaoUserImpl();
                generatePassword = true;
                if (log.isDebugEnabled()) {
                    log.debug("Set company for new user : " + getCompany().getName());
                }
                user.setCompany(getCompany());
            }
        }
        return user;
    }

    void onSelectedFromAddNewUser() {
        addNewUserSelected = true;
    }

    void onSelectedFromDeleteUser() {
        deleteUserSelected = true;
    }

    void onSuccessFromUserActionsForm() throws SuiviObsmerException {
        if (addNewUserSelected) {
            userId = null;
        } else if (deleteUserSelected && userId != null) {
            user = getUsersSelectModel().findObject(userId);
            try {
                serviceUser.deleteUser(user);
                user = null;
                userId = null;
                layout.getFeedBack().addInfo("Utilisateur supprimé avec succès !");
            } catch (SuiviObsmerBusinessException eee) {
                layout.getFeedBack().addError(eee.getMessage());
            }
        }
    }

    @Log
    void onSuccessFromUser() throws SuiviObsmerException {
        if (log.isDebugEnabled()) {
            log.debug("User : " + user);
        }

        if (!StringUtils.isEmpty(password)) {
            user.setPassword(password);
            user.setPasswordChanged(true);
        }
        try {
            // TODO return true if mail ok
            serviceUser.createUpdateUser(user, generatePassword);
            userId = user.getId();
        } catch (SuiviObsmerBusinessException eee) {
            layout.getFeedBack().addError(eee.getMessage());
        }
    }
        
}
