/*
 * *##% 
 * SuiviObsmer :: Web Interface
 * Copyright (C) 2009 - 2010 Ifremer
 *
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as
 * published by the Free Software Foundation, either version 3 of the
 * License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Lesser Public License for more details.
 *
 * You should have received a copy of the GNU General Public
 * License along with this program.  If not, see
 * <http://www.gnu.org/licenses/gpl-3.0.html>.
 * ##%*
 */

package fr.ifremer.suiviobsmer.ui.pages;

import fr.ifremer.suiviobsmer.SuiviObsmerBusinessException;
import fr.ifremer.suiviobsmer.SuiviObsmerContext;
import fr.ifremer.suiviobsmer.SuiviObsmerException;
import fr.ifremer.suiviobsmer.bean.BoatFilter;
import fr.ifremer.suiviobsmer.bean.ContactFilter;
import fr.ifremer.suiviobsmer.bean.ContactFilterImpl;
import fr.ifremer.suiviobsmer.bean.ContactState;
import fr.ifremer.suiviobsmer.bean.ImportResults;
import fr.ifremer.suiviobsmer.entity.Boat;
import fr.ifremer.suiviobsmer.entity.Contact;
import fr.ifremer.suiviobsmer.entity.SampleRow;
import fr.ifremer.suiviobsmer.entity.WaoUser;
import fr.ifremer.suiviobsmer.services.ServiceBoat;
import fr.ifremer.suiviobsmer.services.ServiceContact;
import fr.ifremer.suiviobsmer.services.ServiceSampling;
import fr.ifremer.suiviobsmer.ui.base.AbstractFilteredPage;
import fr.ifremer.suiviobsmer.ui.data.ContactDataSource;
import fr.ifremer.suiviobsmer.ui.data.BusinessUtils;
import fr.ifremer.suiviobsmer.ui.base.GenericSelectModel;
import fr.ifremer.suiviobsmer.ui.base.SuiviObsmerPage;
import fr.ifremer.suiviobsmer.ui.components.Layout;
import fr.ifremer.suiviobsmer.ui.services.ContactModelFactory;
import java.io.IOException;
import java.io.InputStream;
import java.text.DateFormat;
import java.text.SimpleDateFormat;
import java.util.Date;
import java.util.List;
import org.apache.commons.lang.BooleanUtils;
import org.apache.commons.lang.StringUtils;
import org.apache.tapestry5.ComponentResources;
import org.apache.tapestry5.Field;
import org.apache.tapestry5.PersistenceConstants;
import org.apache.tapestry5.StreamResponse;
import org.apache.tapestry5.annotations.IncludeStylesheet;
import org.apache.tapestry5.annotations.InjectComponent;
import org.apache.tapestry5.annotations.Log;
import org.apache.tapestry5.annotations.Persist;
import org.apache.tapestry5.annotations.Property;
import org.apache.tapestry5.annotations.SessionState;
import org.apache.tapestry5.beaneditor.BeanModel;
import org.apache.tapestry5.corelib.components.Form;
import org.apache.tapestry5.corelib.components.Zone;
import org.apache.tapestry5.ioc.annotations.Inject;
import org.apache.tapestry5.ioc.services.PropertyAccess;
import org.apache.tapestry5.services.BeanModelSource;
import org.apache.tapestry5.services.Response;
import org.apache.tapestry5.upload.services.UploadedFile;
import org.nuiton.util.DateUtils;
import org.slf4j.Logger;

/**
 * Contacts
 *
 * Created: 9 nov. 2009
 *
 * @author fdesbois
 * @version $Revision: 294 $
 *
 * Mise a jour: $Date: 2010-01-29 18:21:13 +0100 (ven., 29 janv. 2010) $
 * par : $Author: fdesbois $
 */
@IncludeStylesheet("context:css/contacts.css")
public class Contacts extends AbstractFilteredPage implements SuiviObsmerPage {

    @Override
    public boolean isOnlyForAdmin() {
        return false;
    }

    @Inject
    private Logger log;

    @InjectComponent
    private Layout layout;

    @SessionState
    @Property
    private WaoUser user;
    
    @Inject
    private ServiceContact serviceContact;

    @Inject
    private ServiceSampling serviceSampling;

    @Inject
    private ServiceBoat serviceBoat;

//    @Persist(PersistenceConstants.FLASH)
//    private String contactEditedId;

    @Log
    void setupRender() throws SuiviObsmerException {
//        if (contactEdited == null) {
            if (log.isDebugEnabled()) {
                log.debug("RESET DATA");
            }
            initSelectFilters(true, true, false);
            contacts = null;
            getContacts();
//        }
        contactsForm.clearErrors();
    }

    /**************************** CONTACT FILTERS *****************************/

    @Persist
    private ContactFilter contactFilter;

    @InjectComponent
    private Zone filtersZone;

    @InjectComponent
    private Zone importExportZone;

    private boolean reset;

    @Property
    private ContactState stateFilter;

    public ContactFilter getContactFilter() throws SuiviObsmerException {
        if (contactFilter == null) {
            if (log.isDebugEnabled()) {
                log.debug("Init contactFilter");
            }
            contactFilter = new ContactFilterImpl();
            // Initialized to 12 months before the current day
            Date fromDate = DateUtils.createDateAfterToday(0, -3, 0);
            contactFilter.setFromDate(fromDate);
        }
        return contactFilter;
    }

    public boolean isFiltersVisible() {
        boolean companyFiltered = contactFilter.getCompany() != null && user.getAdmin();
        return (contactFilter.isFiltered() || companyFiltered) && StringUtils.isEmpty(contactFilter.getBoatName()); /*&& !getDefaultFromDate().equals(contactFilter.getFromDate());*/
    }

    @Override
    protected BoatFilter getFilter() throws SuiviObsmerException {
        return getContactFilter();
    }

    @Override
    protected boolean isAvailableDataForFiltersOnly() {
        return true;
    }

    Object onActionFromShowFilters() {
        //contactFilter = null;
        return filtersZone.getBody();
    }

    Object onActionFromShowImportExport() {
        return importExportZone.getBody();
    }

    void onSelectedFromReset() {
        reset = true;
    }

    @Log
    Object onSuccessFromFiltersForm() throws SuiviObsmerException {
        if (isEdited()) {
            return filtersZone.getBody();
        }
        if (reset) {
            contactFilter = null;
        }
//        else {
//            filtersVisible = true;
//        }
        return this;
    }

    /**************************** CONTACT IMPORT/EXPORT ***********************/

    @Property
    private UploadedFile contactsCsvFile;

    @Log
    void onSuccessFromImportContacts() throws SuiviObsmerException {
        //importBoatsForm.clearErrors();
        try {
            ImportResults result = serviceContact.importContactCsv(user, contactsCsvFile.getStream());
            // Suppress persitant list of contacts
            contacts = null;
            layout.getFeedBack().addInfo(result.getNbRowsImported() + " contacts importés,  " +
                    result.getNbRowsRefused() + " refusés.");
            for (String error : result.getErrors()) {
                layout.getFeedBack().addInfo(error);
            }
        } catch (SuiviObsmerBusinessException eee) {
            layout.getFeedBack().addError(eee.getMessage());
        }
        //return importBoatsForm.getHasErrors() ? importBoatsForm : this;
    }

    StreamResponse onActionFromExportShowContacts() {
        return new StreamResponse() {

            @Override
            public String getContentType() {
                return "text/csv;charset=utf-8";
            }

            @Override
            public InputStream getStream() throws IOException {
                InputStream result = null;
                try {
                    result = serviceContact.exportContactCsv(getContactFilter());
                } catch (SuiviObsmerException eee) {
                    throw new IOException(eee);
                }
                return result;
            }

            @Override
            public void prepareResponse(Response response) {
                DateFormat dateFormat = new SimpleDateFormat("dd-MM-yyyy");
                Date current = SuiviObsmerContext.getCurrentDate();
                response.setHeader("Content-Disposition", "attachment; filename=\"wao-contacts-" + dateFormat.format(current) + ".csv\"");
            }
        };
    }

    /**************************** CONTACT LIST ********************************/

    @Inject
    private BeanModelSource beanModelSource;

    @Inject
    private ComponentResources resources;

    @Inject
    private ContactModelFactory contactModelFactory;

    @Persist
    private ContactDataSource contacts;
//    private Map<String, Contact> contacts;

    @Property
    private Contact contact;

    private BeanModel<Contact> contactModel;

    @Inject
    private PropertyAccess propertyAccess;

    private GenericSelectModel<WaoUser> userSelectModel;

    @Property
    @Persist(PersistenceConstants.FLASH)
    private String contactUserId;
    
    private ContactState contactState;

    private boolean even = true;

    public ContactDataSource getContacts() throws SuiviObsmerException {
        if (contacts == null) {
            if (log.isInfoEnabled()) {
                log.info("Create DataSource");
            }
            contacts = new ContactDataSource(getContactFilter(), serviceContact);
        }
        return contacts;
    }

//    public Map<String, Contact> getContacts() throws SuiviObsmerException {
//        if (contacts == null) {
//            if (log.isInfoEnabled()) {
//                log.info("BUSINESS REQUEST [getContactsByFilter]");
//            }
//            contacts = serviceContact.getContactsByFilter(getContactFilter());
//        }
//        return contacts;
//    }

    public BeanModel<Contact> getContactModel() {
        if (contactModel == null) {
            contactModel = user.getAdmin() ?
                contactModelFactory.buildAdminContactModel(beanModelSource, resources) :
                contactModelFactory.buildContactModel(beanModelSource, resources);
        }
        return contactModel;
    }

    public ContactState getContactState() {
        if (contactState == null) {
            // initialization
            this.contactState = ContactState.createContactStateEnum(contactEdited.getState());
        }
        return this.contactState;
    }

    public GenericSelectModel<WaoUser> getUserSelectModel() {
        if (userSelectModel == null) {
            List<WaoUser> users = user.getCompany().getWaoUser();
            if (log.isDebugEnabled()) {
                log.debug("Nb users : " + users.size());
            }
            userSelectModel = new GenericSelectModel<WaoUser>(users, WaoUser.class, "fullName", "id", propertyAccess);
        }
        return userSelectModel;
    }

    public void setContactState(ContactState contactState) {
        this.contactState = contactState;
    }

    public String getCommentDisplayed() {
        if (contact.getComment() != null && contact.getComment().length() > 20) {
            return contact.getComment().substring(0, 20) + "...";
        }
        return contact.getComment();
    }

    public String getCommentTooltip() {
        return contact.getComment().replaceAll("\n", "<br />").replaceAll("\r", "");
    }

    public String getSampleRowDescription() {
        return BusinessUtils.getTooltipSampleRow(contact.getSampleRow());
    }

    public String getBoatDescription() {
        return BusinessUtils.getTooltipBoat(contact.getBoat());
    }

    public String getTooltipExportFrom() throws SuiviObsmerException {
        if (getContactFilter().getFromDate() != null) {
            return "depuis le " + getDateFormat().format(getContactFilter().getFromDate());
        }
        return "";
    }

    public String getGridClass() {
        return user.getAdmin() ? "admin" : "user";
    }

    public String getRowClass() {
        String result = BusinessUtils.getContactStyle(contact, user.getAdmin());
        if (contact.getTopiaId().equals(contactSelectedId)) {
            result = "selected";
        }
        return result;
    }

    public DateFormat getDateFormat() {
        return new SimpleDateFormat("dd/MM/yyyy");
    }

    public boolean isEditionMode() {
        //return contact.getTopiaId().equals(contactEditedId);
        return contactEdited != null && contact.equals(contactEdited);
    }

    public boolean isEmpty(Boolean validation) {
        return validation == null;
    }

    /**************************** CONTACT ROW ACTION **************************/

    /**
     * Flag to know if it's only edition (=true) or save action (=false)
     */
    private boolean edited;

    private boolean deleted;

    @Persist
    @Property
    private Contact contactEdited;

    @Persist
    private String oldComment;

    public void prepareContactEdited(String contactId) throws SuiviObsmerException {
        if (contactEdited == null) {
            contactEdited = getContacts().get(contactId);
        }
        //contact = contactEdited;
    }

    public boolean hasActions() {
        return !user.getAdmin() && contact.getValidationCompany() == null;
    }

    public boolean canValidate() {
        // Can't validate during edition of the contact row
        if (isEditionMode()) {
            return false;
        }
        ContactState state = ContactState.createContactStateEnum(contact.getState());
        boolean boardingDone = state.equals(ContactState.BOARDING_DONE) && contact.getDataInputDate() != null;
        if (!user.getAdmin()) {
            return contact.getValidationCompany() == null && (state.isUnfinishedState() || boardingDone);
        }
        return contact.getValidationProgram() == null && BooleanUtils.isTrue(contact.getValidationCompany());
    }

    public boolean canUnvalidate() {
        if (!user.getAdmin()) {
            return contact.getValidationCompany() != null &&
                    contact.getValidationProgram() == null;
        }
        return contact.getValidationProgram() != null;
    }

    void onSelectedFromAcceptContact(String contactId) throws SuiviObsmerException {
        //contact = getContacts().get(contactId);
        prepareContactEdited(contactId);
        if (user.getAdmin()) {
            contactEdited.setValidationProgram(Boolean.TRUE);
        } else {
            // For company accepted, addRealTideTime
            contactEdited.getSampleRow().addRealTideTime(contactEdited);
            contactEdited.setValidationCompany(Boolean.TRUE);
        }
    }

    void onSelectedFromRefuseContact(String contactId) throws SuiviObsmerException {
        //contact = getContacts().get(contactId);
        prepareContactEdited(contactId);
        if (user.getAdmin()) {
            // For program refused, removeRealTideTime
            contactEdited.getSampleRow().removeRealTideTime(contactEdited);
            contactEdited.setValidationProgram(Boolean.FALSE);
        } else {
            contactEdited.setValidationCompany(Boolean.FALSE);
        }
    }

    void onSelectedFromUnvalidateContact(String contactId) throws SuiviObsmerException {
        //contact = getContacts().get(contactId);
        prepareContactEdited(contactId);
        if (user.getAdmin()) {
            // For program unvalidate from previous refused validation, addRealTideTime
            if (BooleanUtils.isFalse(contactEdited.getValidationProgram())) {
                contactEdited.getSampleRow().addRealTideTime(contactEdited);
            }
            contactEdited.setValidationProgram(null);
        } else {
            // For company unvalidate from previous accepted validation,
            // removeRealTideTime
            if (BooleanUtils.isTrue(contactEdited.getValidationCompany())) {
                contactEdited.getSampleRow().removeRealTideTime(contactEdited);
            }
            contactEdited.setValidationCompany(null);
        }
    }

    void onSelectedFromEditContact(String contactId) throws SuiviObsmerException {
        //contact = getContacts().get(contactId);
        prepareContactEdited(contactId);
        contactUserId = contactEdited.getObserver().getTopiaId();
        //contactEditedId = contactId;
        contactSelectedId = contactId;
        oldComment = contactEdited.getComment();
        edited = true;
    }

    void onSelectedFromDeleteContact(String contactId) throws SuiviObsmerException {
        //contact = getContacts().get(contactId);
        prepareContactEdited(contactId);
        deleted = true;
    }

    void onSelectedFromSaveContact(String contactId) throws SuiviObsmerException {
        //contact = getContacts().get(contactId);
        prepareContactEdited(contactId);
        contactEdited.setState(contactState.toString());
        WaoUser contactUser = getUserSelectModel().findObject(contactUserId);
        contactEdited.setObserver(contactUser);

        if (log.isDebugEnabled()) {
            log.debug("Comment : " + contactEdited.getComment());
        }
    }

    void onSelectedFromCancelEditContact() throws SuiviObsmerException {
        contactEdited = null;
        edited = true;
    }

//    @Log
//    void onSelectedFromSaveComment(String contactId) throws SuiviObsmerException {
//        contact = getContacts().get(contactId);
//        if (log.isInfoEnabled()) {
//            log.info("Comment : " + contact.getComment());
//        }
//        contactEdited = true;
//    }

    /**************************** CONTACT SAVE ********************************/

    @Persist(PersistenceConstants.FLASH)
    private String contactSelectedId;

    @InjectComponent
    private Form contactsForm;

    @InjectComponent
    private Zone gridZone;

    @InjectComponent
    private Field beginDate;

    @InjectComponent
    private Field endDate;

    @InjectComponent
    private Field nbObservants;

    @InjectComponent
    private Field comment;

    @InjectComponent
    private Field inputDate;

    @Log
    void onValidateFormFromContactsForm() {
        contactsForm.clearErrors();
        // Validation for saving contact depends on contactState (only edition
        // form)
        if (!edited && contactState != null) {
            if (log.isInfoEnabled()) {
                log.info("For state : " + contactState);
            }

            Date begin = contactEdited.getTideBeginDate();
            Date end = contactEdited.getTideEndDate();
            Date input = contactEdited.getDataInputDate();
            SampleRow row = contactEdited.getSampleRow();
            //DateFormat dateFormat = new SimpleDateFormat("MM/yyyy");

            if (begin != null && !row.isValid(begin)) {
                contactsForm.recordError(beginDate,
                        "La date de début de la marée doit correspondre à un mois valide (non vide) de la ligne " + row.getCode());
            }

            if (begin != null && end != null && end.before(begin)) {
                contactsForm.recordError(endDate, "La date de fin de la marée ne peut pas être antérieure à celle de début");
            }

            Date current = SuiviObsmerContext.getCurrentDate();

            if (end != null && end.after(current)) {
                contactsForm.recordError(endDate, "La date de fin de la marée ne peut pas être postérieure à la date du jour");
            }

            if (end != null && input != null && end.after(input)) {
                contactsForm.recordError(inputDate, "La date de saisie des données ne peut pas être antérieure à la date de fin de la marée");
            }

            if (input != null && input.after(current)) {
                contactsForm.recordError(inputDate, "La date de saisie des données ne peut pas être postérieure à la date du jour");
            }

            // Non abouti, Refus ou Refus Définitif
            if (contactState.isUnfinishedState()) {
                if (oldComment != null && StringUtils.isNotEmpty(contactEdited.getComment()) && !oldComment.equals(contactEdited.getComment())) {
                    // RAZ des champs
                    contactEdited.setTideBeginDate(null);
                    contactEdited.setTideEndDate(null);
                    contactEdited.setNbObservants(0);
                    contactEdited.setMammalsCapture(false);
                    contactEdited.setMammalsObservation(false);
                    contactEdited.setDataInputDate(null);
                } else {
                    contactsForm.recordError(comment, "Vous devez ajouter un commentaire pour l'état '" + contactState + "'");
                }
            // Embarquement Réalisé
            } else if (contactState.equals(ContactState.BOARDING_DONE)) {
                if (begin == null) {
                    contactsForm.recordError(beginDate, "La date de début de marée est obligatoire pour l'état '" + contactState + "'");
                }
                if (end == null) {
                    contactsForm.recordError(endDate, "La date de fin de marée est obligatoire pour l'état '" + contactState + "'");
                }
                if (contactEdited.getNbObservants() == 0) {
                    contactsForm.recordError(nbObservants, "Il ne peut y avoir aucun observateur pour l'état '" + contactState + "'");
                }
            }
        }
    }

    @Log
    Object onSuccessFromContactsForm() {
        if (!edited) {
            try {
                if (log.isDebugEnabled()) {
                    log.debug("Contact save : " + contactEdited);
                }
                serviceContact.saveContact(contactEdited, deleted);
                contactSelectedId = contactEdited.getTopiaId();
                oldComment = null;
                contactEdited = null;
            } catch (SuiviObsmerException eee) {
                layout.getFeedBack().addError(eee.getMessage());
            }
        }
        return this;
    }

    @Log
    Object onFailureFromContactsForm() {
        if (log.isDebugEnabled()) {
            log.debug("Contact can't be saved with errors");
        }
        // The contact is not saved, the contact must be editable to show form
        // and correct errors
        edited = true;
        //contactEditedId = contact.getTopiaId();
        return contactsForm;
        //return gridZone;
    }

    public void createNewContact(Boat boat, SampleRow sampleRow) throws SuiviObsmerException {
        contact = serviceContact.getNewContact(user, sampleRow, boat);
        serviceContact.saveContact(contact, Boolean.FALSE);
        contactSelectedId = contact.getTopiaId();
    }
}
