package org.nuiton.wikitty.solr;

import java.util.ArrayList;
import java.util.Arrays;
import java.util.List;

import org.apache.solr.client.solrj.SolrQuery;
import org.apache.solr.client.solrj.SolrServer;
import org.apache.solr.client.solrj.SolrServerException;
import org.apache.solr.client.solrj.response.QueryResponse;
import org.apache.solr.common.SolrDocument;
import org.apache.solr.common.SolrDocumentList;
import org.nuiton.wikitty.WikittyException;
import org.nuiton.wikitty.WikittyTransaction;
import org.nuiton.wikitty.search.And;
import org.nuiton.wikitty.search.AssociatedRestriction;
import org.nuiton.wikitty.search.Between;
import org.nuiton.wikitty.search.Contains;
import org.nuiton.wikitty.search.Element;
import org.nuiton.wikitty.search.EndsWith;
import org.nuiton.wikitty.search.Equals;
import org.nuiton.wikitty.search.Greater;
import org.nuiton.wikitty.search.GreaterOrEqual;
import org.nuiton.wikitty.search.In;
import org.nuiton.wikitty.search.Keyword;
import org.nuiton.wikitty.search.Less;
import org.nuiton.wikitty.search.LessOrEqual;
import org.nuiton.wikitty.search.Like;
import org.nuiton.wikitty.search.Not;
import org.nuiton.wikitty.search.NotEquals;
import org.nuiton.wikitty.search.Or;
import org.nuiton.wikitty.search.Restriction;
import org.nuiton.wikitty.search.RestrictionHelper;
import org.nuiton.wikitty.search.StartsWith;
import org.nuiton.wikitty.search.Unlike;
import org.nuiton.wikitty.search.Like.SearchAs;
import org.nuiton.wikitty.search.Null;

/**
 * @author "Nicolas Chapurlat" <nicolas.chapurlat@logica.com>
 * @author "Guillaume Dufrêne" <dufrene@argia.fr>
 * 
 * This class is used to parse Restriction to create lucene request on
 * content. Every operators describe in RestrictionName is handle. Parsing may
 * throw exception when restriction parameters are incorrect.
 */
public class Restriction2Solr {
    
    private static final int MAX_SUBQUERY_RESULT = 100;

    final static protected WikittySearchEnginSolr.FieldModifier dummyFieldModifier = new WikittySearchEnginSolr.FieldModifier() {
        public String convertToSolr(WikittyTransaction transaction, String fieldname) {
            return fieldname;
        }
        public String convertToField(WikittyTransaction transaction, String solrName) {
            return solrName;
        }
    };

    protected WikittySearchEnginSolr.FieldModifier fieldModifer;
    protected WikittyTransaction transaction;

    public Restriction2Solr() {
        this(null, dummyFieldModifier);
    }

    // FIXME 20101201 jru improve manage transaction and fieldModifeir in helper
    public Restriction2Solr(WikittyTransaction transaction, WikittySearchEnginSolr.FieldModifier fieldModifer) {
        this.transaction = transaction;
        this.fieldModifer = fieldModifer;
    }

    public String toSolr(Restriction restriction) {
        return toSolr(restriction, null);
    }
    
    public String toSolr(Restriction restriction, SolrServer solr)
            throws WikittyException {
        // ParameterValidator.checkNullParameter(restriction, "restriction");
        switch (restriction.getName()) {
        case TRUE:
            return true2solr();
        case FALSE:
            return false2solr();
        case NOT:
            Not not = (Not) restriction;
            return not2solr(not);
        case AND:
            And and = (And) restriction;
            return and2solr(and);
        case OR:
            Or or = (Or) restriction;
            return or2solr(or);
        case EQUALS:
            Equals eq = (Equals) restriction;
            return eq2solr(eq);
        case LIKE:
            Like like = (Like) restriction;
            return like2solr(like);
        case UNLIKE:
            Unlike unlike = (Unlike) restriction;
            return unlike2solr(unlike);
        case NOT_EQUALS:
            NotEquals neq = (NotEquals) restriction;
            return neq2solr(neq);
        case LESS:
            Less less = (Less) restriction;
            return less2solr(less);
        case LESS_OR_EQUAL:
            LessOrEqual lessEq = (LessOrEqual) restriction;
            return lessEq2solr(lessEq);
        case GREATER:
            Greater great = (Greater) restriction;
            return great2solr(great);
        case GREATER_OR_EQUAL:
            GreaterOrEqual greatEq = (GreaterOrEqual) restriction;
            return greatEq2solr(greatEq);
        case BETWEEN:
            Between between = (Between) restriction;
            return between2solr(between);
        case CONTAINS:
            Contains contains = (Contains) restriction;
            return contains2solr(contains);
        case IN:
            In in = (In) restriction;
            return in2solr(in);
        case STARTS_WITH:
            StartsWith start = (StartsWith) restriction;
            return start2solr(start);
        case ENDS_WITH:
            EndsWith end = (EndsWith) restriction;
            return end2solr(end);
        case ASSOCIATED:
            AssociatedRestriction associated = (AssociatedRestriction) restriction;
            return associated2solr(associated, solr);
        case KEYWORD:
            Keyword keyword = (Keyword) restriction;
            return keyword2solr(keyword);
        case IS_NULL:
            Null isNull = (Null) restriction;
            return isNull2solr(isNull);
        case IS_NOT_NULL:
            Null isNotNull = (Null) restriction;
            return isNotNull2solr(isNotNull);
        default:
            throw new WikittyException("this kind of restriction is not supported : "
                            + restriction.getName().toString());
        }
    }

    private String in2solr(In in) {
        boolean first = true;
        String result = in.getElement().getName() + ":[";
        for( String value : in.getValue() ) {
            if ( !first ) { result += ", "; first = false; }
            result += value;
        }
        result +="]";
        return result;
    }

    private String associated2solr(AssociatedRestriction associated, SolrServer solr) throws WikittyException {
        String subQuery = toSolr( associated.getRestriction() );
        SolrQuery query = new SolrQuery(WikittySearchEnginSolr.SOLR_QUERY_PARSER + subQuery);
        query.setRows(MAX_SUBQUERY_RESULT);
        QueryResponse resp = null;
        try {
            resp = solr.query(query);
        } catch (SolrServerException e) {
            throw new WikittyException("Unable to execute associative query on " + associated.getElement().getName(), e);
        }
        SolrDocumentList solrResults = resp.getResults();
        
        Restriction generatedRestriction = null;
        long size = solrResults.size();
        if ( size == 0 ) {
            throw new WikittyException("Associated " + associated.getElement().getName() + " do not retrieved any result");
        }
        if ( size == 1 ) {
            generatedRestriction = RestrictionHelper.eq( associated.getElement(), (String) solrResults.get(0).getFieldValue(WikittySearchEnginSolr.SOLR_ID) );
        } else {
            List<String> ids = new ArrayList<String>(solrResults.size());
            for (SolrDocument doc : solrResults) {
                String id = (String) doc.getFieldValue(WikittySearchEnginSolr.SOLR_ID);
                ids.add(id);
            }
            generatedRestriction = new In(associated.getElement(), ids);
        }
        Restriction parent = associated.getParentRestrictionDto();
        And and;
        if ( parent instanceof And ) {
            and = (And) parent;
            and.getRestrictions().add( generatedRestriction );
        } else {
            and = RestrictionHelper.and( Arrays.asList(new Restriction[]{ associated.getParentRestrictionDto(), generatedRestriction }) );
        }
        return toSolr(and);
    }

    private String not2solr(Not not) throws WikittyException {
        if (not.getRestriction() == null) {
            throw new WikittyException( "not.restriction" );
        }
        return "( *:* - " + toSolr(not.getRestriction()) + " )";
    }

    private String and2solr(And and) throws WikittyException {
        if (and.getRestrictions() == null) {
            throw new WikittyException( "and.restrictions is null" );
        }
        if (and.getRestrictions().size() < 2) {
            throw new WikittyException(    "AND is an operator that handle 2 operand at least");
        }
        boolean first = true;
        StringBuffer result = new StringBuffer();
        for (Restriction restriction : and.getRestrictions()) {
            if (first) {
                result.append("( ").append(toSolr(restriction));
                first = false;
            } else {
                result.append(" AND ").append(toSolr(restriction));
            }
        }
        return result.append(" )").toString();
    }

    private String or2solr(Or or) throws WikittyException {
        if (or.getRestrictions() == null) {
            throw new WikittyException("or.restrictions is null");
        }
        if (or.getRestrictions().size() < 2) {
            throw new WikittyException("OR is an operator that handle 2 operand at least");
        }
        boolean first = true;
        StringBuffer result = new StringBuffer();
        for (Restriction restriction : or.getRestrictions()) {
            if (first) {
                result.append("( ");
                first = false;
            } else {
                result.append(" OR ");
            }
            result.append(toSolr(restriction));
        }
        return result.append(" )").toString();
    }

    private String eq2solr(Equals eq) throws WikittyException {
        return element2solr(eq.getElement()) + ":" + value2solr(eq.getValue());
    }

    private String like2solr(Like like) throws WikittyException {
        SearchAs searchAs = like.getSearchAs();
        String element2solr = element2solr(like.getElement());
        if(element2solr.endsWith("_s")) { // is string
            switch(searchAs) {
                case AsText:
                    element2solr += "_t";
                    break;
                case ToLowerCase:
                    element2solr += "_c";
                    break;
            }
        }

        // Warning if you need add searchAs, AsText and ToLowerCase need search
        // at lowercase
        String value2solr = value2solr(like.getValue());
        if(!element2solr.endsWith("_dt")) { // is not date
            value2solr = value2solr.toLowerCase();
        }

        return element2solr + ":" + value2solr;
    }

    private String unlike2solr(Unlike unlike) throws WikittyException {
        SearchAs searchAs = unlike.getSearchAs();
        String element2solr = element2solr(unlike.getElement());
        if(element2solr.endsWith("_s")) { // is string
            switch(searchAs) {
                case AsText:
                    element2solr += "_t";
                    break;
                case ToLowerCase:
                    element2solr += "_c";
                    break;
            }
        }

        // Warning if you need add searchAs, AsText and ToLowerCase need search
        // at lowercase
        String value2solr = value2solr(unlike.getValue());
        if(!element2solr.endsWith("_dt")) { // is not date
            value2solr = value2solr.toLowerCase();
        }

        return "-" + element2solr + ":" + value2solr;
    }

    private String neq2solr(NotEquals neq)
            throws WikittyException {
        return "-" + element2solr(neq.getElement()) + ":"
                + value2solr(neq.getValue());
    }

    private String less2solr(Less less) throws WikittyException {
        return element2solr(less.getElement()) + ":{* TO "
                + value2solr(less.getValue()) + "}";
    }

    private String lessEq2solr(LessOrEqual lessEq)
            throws WikittyException {
        return element2solr(lessEq.getElement()) + ":[* TO "
                + value2solr(lessEq.getValue()) + "]";
    }

    private String great2solr(Greater great)
            throws WikittyException {
        return element2solr(great.getElement()) + ":{"
                + value2solr(great.getValue()) + " TO *}";
    }

    private String greatEq2solr(GreaterOrEqual greatEq)
            throws WikittyException {
        return element2solr(greatEq.getElement()) + ":["
                + value2solr(greatEq.getValue()) + " TO *]";
    }

    private String between2solr(Between between)
            throws WikittyException {
        if (between.getElement() == null) {
            throw new WikittyException("contains.element");
        }
        if (between.getMin() == null) {
            throw new WikittyException("contains.min");
        }
        if (between.getMax() == null) {
            throw new WikittyException("contains.max");
        }
        return element2solr(between.getElement()) + ":["
                + value2solr(between.getMin()) + " TO "
                + value2solr(between.getMax()) + "]";
    }

    private String contains2solr(Contains contains)
            throws WikittyException {
        if (contains.getElement() == null) {
            throw new WikittyException("contains.element");
        }
        if (contains.getValue() == null) {
            throw new WikittyException("contains.values");
        }
        if (contains.getValue().size() < 1) {
            throw new WikittyException("CONTAINS is an operator that handle 1 operand at least");
        }

        String operand = "";
        StringBuffer result = new StringBuffer();
        result.append("(");
        for (String value : contains.getValue()) {
            result.append(operand);
            result.append(element2solr(contains.getElement()))
                    .append(":").append(value2solr(value));
            operand = " OR ";
        }
        result.append(")");
        return result.toString();
    }

    private String start2solr(StartsWith start)
            throws WikittyException {
        return element2solr(start.getElement()) + ":"
                + value2solr(start.getValue()) + "*";
    }

    private String end2solr(EndsWith end) {
        return element2solr(end.getElement()) + ":*"
                + value2solr(end.getValue());
    }

    private String true2solr() {
        return "( *:* )";
    }

    private String false2solr() {
        return "( *:* - *:* )";
    }

    private String keyword2solr(Keyword keyword) {
        return value2solr(keyword.getValue());
    }

    private String isNull2solr(Null isNull) {
        return "( *:* - " + WikittySearchEnginSolr.SOLR_NOT_NULL_FIELDS + ":" + isNull.getFieldName() + ")";
    }

    private String isNotNull2solr(Null isNotNull) {
        return WikittySearchEnginSolr.SOLR_NOT_NULL_FIELDS + ":" + isNotNull.getFieldName();
    }

    private String element2solr(Element element) throws WikittyException {
        String result = element.getName();
        result = fieldModifer.convertToSolr(transaction, result);
        return result;
    }

    private String value2solr(String value) {
        String result;
        if (value != null) {
            result = Restriction2Solr.escapeValue(value);
        } else {
            throw new WikittyException("Parse error, value must be not empty");
        }

        if (result.contains(" ")) {
            result = "\"" + result + "\"";
        }
        return result;
    }

    private static String escapeValue(String value) {
        final String LUCENE_REPLACE_PATTERN = "\\+" + "|-" + "|&&" + "|\\|"
                + "|!" + "|\\(|\\)" + "|\\[|\\]" + "|\\{|\\}" + "|\"" + "|:";
        return value.replaceAll(LUCENE_REPLACE_PATTERN, "\\\\$0");
    }
}

