/*
 * #%L
 * Wikitty :: api
 * 
 * $Id$
 * $HeadURL$
 * %%
 * Copyright (C) 2012 CodeLutin, Benjamin Poussin
 * %%
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Lesser General Public License as 
 * published by the Free Software Foundation, either version 3 of the 
 * License, or (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Lesser Public License for more details.
 * 
 * You should have received a copy of the GNU General Lesser Public 
 * License along with this program.  If not, see
 * <http://www.gnu.org/licenses/lgpl-3.0.html>.
 * #L%
 */
package org.nuiton.wikitty;

import java.io.Serializable;
import java.lang.reflect.Array;
import java.lang.reflect.Field;
import java.lang.reflect.Method;
import java.math.BigDecimal;

import org.apache.commons.lang.StringUtils;
import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.nuiton.util.ApplicationConfig;
import org.nuiton.util.TimeLog;
import org.nuiton.wikitty.entities.BusinessEntity;
import org.nuiton.wikitty.entities.BusinessEntityImpl;
import org.nuiton.wikitty.entities.Wikitty;
import org.nuiton.wikitty.entities.WikittyExtension;
import org.nuiton.wikitty.entities.WikittyGroup;
import org.nuiton.wikitty.entities.WikittyUser;
import org.nuiton.wikitty.services.WikittyEvent;
import org.nuiton.wikitty.services.WikittySecurityUtil;
import org.nuiton.wikitty.services.WikittyServiceEnhanced;

import java.util.ArrayList;
import java.util.Collection;
import java.util.Collections;
import java.util.Date;
import java.util.HashMap;
import java.util.LinkedHashSet;
import java.util.LinkedList;
import java.util.List;
import java.util.Map;
import java.util.Set;
import org.apache.commons.collections.CollectionUtils;
import org.nuiton.wikitty.entities.WikittyField;
import org.nuiton.wikitty.entities.WikittyTokenHelper;
import org.nuiton.wikitty.query.WikittyQuery;
import org.nuiton.wikitty.query.WikittyQueryMaker;
import org.nuiton.wikitty.query.WikittyQueryResult;
import org.nuiton.wikitty.query.WikittyQueryResultTreeNode;
import org.nuiton.wikitty.query.conditions.ElementField;
import org.nuiton.wikitty.query.conditions.Select;
import org.nuiton.wikitty.services.WikittyExtensionMigrationRegistry;

/**
 * Wikitty client is object used in client side to access WikittyService.
 * It is used to transform wikitty object used by {@link WikittyService}
 * into business objects used by applications.
 * 
 * It also manage {@link #securityToken} for {@link org.nuiton.wikitty.services.WikittyServiceSecurity}.
 *
 * All method that need {@link #securityToken} and {@link org.nuiton.wikitty.services.WikittyServiceSecurity}
 * must be in this class and not in {@link WikittyUtil}
 *
 * @author poussin
 * @version $Revision$
 * @since 3.3
 *
 * Last update: $Date$
 * by : $Author$
 */
public class WikittyClient {

    /** to use log facility, just put in your code: log.info(\"...\"); */
    final static private Log log = LogFactory.getLog(WikittyClient.class);
    final static private TimeLog timeLog = new TimeLog(WikittyClient.class);

    protected ApplicationConfig config = null;
    /** Delegated wikitty service. */
    protected WikittyServiceEnhanced wikittyService;

    /**
     * Security token.
     * 
     * @see org.nuiton.wikitty.services.WikittyServiceSecurity#login(String, String)
     */
    protected String securityToken;

    /**
     * Empty constructor (uninitialized wikittyService).
     */
    protected WikittyClient() {
    }

    /**
     * Creation du client, le wikittyService est instancier grace au information
     * trouve dans la configuration.
     *
     * @param config
     */
    public WikittyClient(ApplicationConfig config) {
        this(config, WikittyServiceFactory.buildWikittyService(config));
    }

    /**
     * Creation du client en forcant le wikittyService
     *
     * @param config
     * @param wikittyService
     */
    public WikittyClient(ApplicationConfig config, WikittyService wikittyService) {
        if (config != null) {
            this.config = config;
            long timeToLogInfo = config.getOptionAsInt(WikittyConfigOption.
                    WIKITTY_CLIENT_TIME_TO_LOG_INFO.getKey());
            long timeToLogWarn = config.getOptionAsInt(WikittyConfigOption.
                    WIKITTY_CLIENT_TIME_TO_LOG_WARN.getKey());
            timeLog.setTimeToLogInfo(timeToLogInfo);
            timeLog.setTimeToLogWarn(timeToLogWarn);

        }
        setWikittyService(wikittyService);
    }

    static public TimeLog getTimeTrace() {
        return timeLog;
    }

    static public Map<String, TimeLog.CallStat> getCallCount() {
        return timeLog.getCallCount();
    }

    public WikittyExtensionMigrationRegistry getMigrationRegistry() {
        WikittyExtensionMigrationRegistry result =
                config.getObject(WikittyExtensionMigrationRegistry.class);
        return result;
    }

    public void login(String login, String password) {
        long start = TimeLog.getTime();
        String result = wikittyService.login(login, password);
        setSecurityToken(result);
        
        timeLog.log(start, "login");
    }

    public void logout() {
        long start = TimeLog.getTime();
        wikittyService.logout(securityToken);
        
        timeLog.log(start, "logout");
    }

    public String getSecurityToken() {
        return securityToken;
    }

    public void setSecurityToken(String securityToken) {
        this.securityToken = securityToken;
    }

    /**
     * get current wikittyUser logged or null if no user logged
     * @return null if no user logged
     */
    public WikittyUser getUser() {
        WikittyUser result = getUser(WikittyUser.class);
        return result;
    }

    /**
     * get current logged user wikitty object
     * @param clazz Business class used as User in your application,
     * this extension should be require WikittyUser.
     * @return null if no user logged
     */
    public <E extends BusinessEntity> E getUser(Class<E> clazz) {
        E result = null;
        if (securityToken != null) {
            //Get the token
            Wikitty securityTokenWikitty = restore(securityToken);
            if (securityTokenWikitty != null) {
                //Get the user
                String userId = WikittyTokenHelper.getUser(securityTokenWikitty);
                result = restore(clazz, userId);
            }
        }
        return result;
    }

    public WikittyService getWikittyService() {
        return wikittyService.getDelegate();
    }

    public void setWikittyService(WikittyService wikittyService) {
        this.wikittyService = new WikittyServiceEnhanced(wikittyService);
    }

    public ApplicationConfig getConfig() {
        return config;
    }

    ////////////////////////////////////////////////////////////////////////////
    //
    // STORE
    //
    ////////////////////////////////////////////////////////////////////////////

    public Wikitty store(Wikitty w) {
        long start = TimeLog.getTime();
        WikittyEvent resp = wikittyService.store(securityToken, w);
        // update object
        resp.update(w);

        timeLog.log(start, "store");
        return w;
    }

    public <E extends BusinessEntity> E store(E e) {
        Wikitty w = ((BusinessEntityImpl)e).getWikitty();
        store(w);
        return e;
    }

    public <E extends BusinessEntity> E[] store(E e1, E e2, E... eN) {
        List<E> es = new ArrayList<E>(eN.length + 2);
        Collections.addAll(es, e1, e2);
        Collections.addAll(es, eN);

        List<E> list = store(es);

        E[] result = list.toArray((E[])Array.newInstance(
                eN.getClass().getComponentType(), list.size()));
        return result;
    }

    public Wikitty[] store(Wikitty w1, Wikitty w2, Wikitty... wN) {
        List<Wikitty> ws = new ArrayList<Wikitty>(wN.length + 2);
        Collections.addAll(ws, w1, w2);
        Collections.addAll(ws, wN);

        List<Wikitty> resultList = storeWikitty(ws);
        Wikitty[] result = resultList.toArray(new Wikitty[resultList.size()]);
        return result;
    }

    /**
     * Store to WikittyService objects.
     * 
     * @param <E> object type
     * @param objets list
     * @return updated objects list
     */
    public <E extends BusinessEntity> List<E> store(List<E> objets) {
        long start = TimeLog.getTime();
        // prepare data to send to service
        List<Wikitty> wikitties = new ArrayList<Wikitty>(objets.size());
        for (E e : objets) {
            if (e == null) {
                wikitties.add(null);
            } else {
                Wikitty w = ((BusinessEntityImpl)e).getWikitty();
                wikitties.add(w);
            }
        }

        // call the service with Wikitty
        WikittyEvent resp = wikittyService.store(securityToken, wikitties);

        // update object
        for (Wikitty w : wikitties) {
            resp.update(w);
        }

        timeLog.log(start, "store<list>");
        return objets;
    }

    public  List<Wikitty> storeWikitty(List<Wikitty> wikitties) {
        long start = TimeLog.getTime();

        // call the service with Wikitty
        WikittyEvent resp = wikittyService.store(securityToken, wikitties);

        // update object
        for (Wikitty w : wikitties) {
            resp.update(w);
        }

        timeLog.log(start, "storeWikitty<list>");
        return wikitties;
    }

    ////////////////////////////////////////////////////////////////////////////
    //
    // RESTORE
    //
    ////////////////////////////////////////////////////////////////////////////

    /**
     * Restore wikitty entity with specified id or {@code null} if entity can't be be found.
     *
     * @param id entity ids if null return is empty list
     * @return wikitty entity with specified id or {@code null} if entity can't be found
     */
    public List<Wikitty> restore(List<String> id) {
        long start = TimeLog.getTime();

        List<Wikitty> result;
        if (id == null) {
            result = new ArrayList<Wikitty>();
        } else {
            result = wikittyService.restore(securityToken, id);
        }

        timeLog.log(start, "restore<list>");
        return result;
    }

    /**
     * Restore wikitty entity with specified id or {@code null} if entity can't be found.
     *
     * @param id entity id
     * @return wikitty entity with specified id or {@code null} if entity can't be found
     */
    public Wikitty restore(String id) {
        long start = TimeLog.getTime();
        Wikitty result = null;
        if (id != null) {
            result = restore(Collections.singletonList(id)).get(0);
        }

        timeLog.log(start, "restore");
        return result;
    }

    /**
     * Restore wikitty entity with specified id or {@code null} if entity
     * can't be be found, or checkExtension is true and wikitty don't match
     * extension wanted.
     * 
     * @param <E> object type
     * @param clazz entity class
     * @param id entity ids if null return is empty list
     * @param checkExtension if true check that Wikitty result has all extension
     * @return wikitty entity with specified id or {@code null} if entity
     * can't be found or if one wikitty don't have extension wanted by E type
     */
    public <E extends BusinessEntity> List<E> restore(
            Class<E> clazz, List<String> id, boolean checkExtension) {
        long start = TimeLog.getTime();
        List<E> result = new ArrayList<E>();

        if (id != null) {
            List<Wikitty> wikitties = wikittyService.restore(securityToken, id);

            Collection<String> businessExtension = null;
            if (checkExtension) {
                // Recuperation de la liste des extensions du BusinessEntity resultats
                BusinessEntityImpl sample =
                        (BusinessEntityImpl) WikittyUtil.newInstance(clazz);
                businessExtension = sample.getExtensionNames();
            }

            for (Wikitty w : wikitties) {
                E dto = null;
                if (!checkExtension ||
                        // on ne check pas les extensions ou ...
                        CollectionUtils.subtract(businessExtension, w.getExtensionNames()).isEmpty()) {
                        // ... on a retrouve toutes les extensions du wikitty dans l'objet
                        // on le prend dans les resultats
                        dto = WikittyUtil.newInstance(clazz, w);
                } // sinon on ne prend pas l'objet
                
                // add entity to result after checkExtension
                result.add(dto);
            }
        }
        timeLog.log(start, "restore<list>");
        return result;
    }

    /**
     * Restore wikitty entity with specified id or {@code null} if entity can't be found.
     *
     * @param <E> object type
     * @param clazz entity class
     * @param id entity id
     * @return wikitty entity with specified id or {@code null} if entity can't be found
     */
    public <E extends BusinessEntity> E restore(Class<E> clazz, String id) {
        E result = restore(clazz, id, false);
        return result;
    }

    /**
     * Restore wikitty entity with specified id or {@code null} if entity can't be found.
     *
     * @param <E> object type
     * @param clazz entity class
     * @param id entity id
     * @param checkExtension if true check that Wikitty result has all extension
     * declared in clazz
     * @return wikitty entity with specified id or {@code null} if entity can't be found
     */
    public <E extends BusinessEntity> E restore(Class<E> clazz, String id, boolean checkExtension) {
        long start = TimeLog.getTime();
        E result = null;
        if (id != null) {
            result = restore(clazz, Collections.singletonList(id), checkExtension).get(0);
        }

        timeLog.log(start, "restore<Business>");
        return result;
    }

    public <E extends BusinessEntity> List<E> restore(Class<E> clazz, List<String> id) {
        List<E> result = restore(clazz, id, false);
        return result;
    }

    public Set<Wikitty> restore(Set<String> id) {
        ArrayList<String> list = null;
        if (id != null) {
            list = new ArrayList<String>(id);
        }
        List<Wikitty> resultList = restore(list);
        Set<Wikitty> result = new LinkedHashSet<Wikitty>(resultList);
        return result;
    }

    public <E extends BusinessEntity> Set<E> restore(Class<E> clazz, Set<String> id) {
        Set<E> result = restore(clazz, id, false);
        return result;
    }

    public <E extends BusinessEntity> Set<E> restore(Class<E> clazz, Set<String> id, boolean checkExtension) {
        ArrayList<String> list = null;
        if (id != null) {
            list = new ArrayList<String>(id);
        }
        List<E> resultList = restore(clazz, list, checkExtension);
        Set<E> result = new LinkedHashSet<E>(resultList);
        return result;
    }

    ////////////////////////////////////////////////////////////////////////////
    //
    // DETELE
    //
    ////////////////////////////////////////////////////////////////////////////

    public void delete(String id) {
        long start = TimeLog.getTime();
        wikittyService.delete(securityToken, id);

        timeLog.log(start, "delete");
    }

    public <E extends BusinessEntity> void delete(E object) {
        long start = TimeLog.getTime();
        if (object != null) {
            String id = object.getWikittyId();
            wikittyService.delete(securityToken, id);
        }
        timeLog.log(start, "delete(BusinessEntity)");
    }

    public void delete(Collection<String> ids) {
        long start = TimeLog.getTime();
        wikittyService.delete(securityToken, ids);
        
        timeLog.log(start, "delete<list>");
    }

    public <E extends BusinessEntity> void delete(List<E> objets) {
        long start = TimeLog.getTime();
        
        // prepare data to send to service
        List<String> ids = new ArrayList<String>(objets.size());
        for (E e : objets) {
            if (e != null) {
                String id = e.getWikittyId();
                ids.add(id);
            }
        }

        // call the service with Wikitty
        wikittyService.delete(securityToken, ids);

        timeLog.log(start, "delete<list<BusinessEntity>>");
    }

    ///////////////////////////////////////////////////////////////////////////
    //
    // FIND (ALL) BY EXEMPLE <E>
    //
    ///////////////////////////////////////////////////////////////////////////

    /**
     * Null field are not used in search request.
     *
     * @param e sample wikitty
     * @param firstIndex
     * @param endIndex
     * @param fieldFacet
     * @return
     */
    public <E extends BusinessEntityImpl> WikittyQueryResult<E> findAllByExample(E e,
            int first, int limit, ElementField ... fieldFacet ) {
        long start = TimeLog.getTime();

        WikittyQuery query = new WikittyQueryMaker().wikitty(e).end()
                .setFirst(first).setLimit(limit)
                .setFacetField(fieldFacet);

        WikittyQueryResult<String> queryResult = findAllByQuery(query);
        WikittyQueryResult<E> result = (WikittyQueryResult<E>)castTo(
                e.getClass(), queryResult);
        
        timeLog.log(start, "findAllByExample<limit>");
        return result;
    }

    /**
     * Null field are not used in search request.
     * 
     * @param e sample wikitty
     * @return
     */
    public <E extends BusinessEntityImpl> E findByExample(E e) {
        long start = TimeLog.getTime();
        WikittyQuery query = new WikittyQueryMaker().wikitty(e).end();

        String id = findByQuery(query);
        E result = (E)restore(e.getClass(), id);
        
        timeLog.log(start, "findByExample");
        return result;
    }

    ///////////////////////////////////////////////////////////////////////////
    //
    // FIND ALL BY QUERY <E>
    //
    ///////////////////////////////////////////////////////////////////////////

    /**
     * Cette method doit etre l'unique methode finalement utilise par toutes
     * les methodes find avec un cast vers clazz
     *
     * Search object that correspond to criteria and that have all extension
     * needed by BusinessEntity (clazz), if clazz is BusinessEntity class.
     * If one criteria is empty, find all extensions
     * for this criteria else if criteria is null return null.
     *
     * @param <E> object type
     * Can be Wikitty, BusinessEntity, String, Date, Number (returned is BigDecimal), Boolean, byte[]
     * @param clazz entity class
     * @param queries criterias
     * @param limitToFirst if true limit result to first result (first = 0, limit = 1)
     * @return paged result
     */
    protected <E> List<WikittyQueryResult<E>> findAllByQuery(
            Class<E> clazz, List<WikittyQuery> queries, boolean limitToFirst) {
        long start = TimeLog.getTime();
        List<WikittyQueryResult<E>> result = null;
        List<WikittyQuery> serviceQueries;
        if (queries != null) {
            if (clazz.isAssignableFrom(BusinessEntity.class)) {
                // on demande un business entity donc on modifie
                // pas les criteres pour ajouter les contraintes sur les
                // extensions

                // newInstance only return BusinessEntityWikittyImpl
                BusinessEntityImpl sample =
                        (BusinessEntityImpl) WikittyUtil.newInstance((Class<BusinessEntity>)clazz);

                Wikitty wikitty = sample.getWikitty();
                Collection<String> extensions = wikitty.getExtensionNames();

                serviceQueries = new ArrayList<WikittyQuery>(queries.size());
                for (WikittyQuery criteria : queries) {

                    // on ajoute la condition sur les extensions dans le critere
                    // du coup, pour ne pas modifier le critere qui vient en parametre
                    // il faut creer un nouveau critere ...
                    WikittyQuery serviceQuery = null;
                    if (criteria != null) {
                        serviceQuery = criteria.copy();
                        WikittyQueryMaker queryMaker;
                        if (serviceQuery.getCondition() instanceof Select) {
                            // si la condition commence par un select
                            // alors il faut modifier la condition du select
                            Select select = (Select)serviceQuery.getCondition();
                            queryMaker = new WikittyQueryMaker()
                                    .select(select.getElement())
                                    .and().condition(select.getSubCondition())
                                    .extContainsAll(extensions);

                        } else {
                            // sinon on modifie directement la condition
                            queryMaker = new WikittyQueryMaker()
                                    .and().condition(serviceQuery.getCondition())
                                    .extContainsAll(extensions);
                        }
                        // utilisation de cette nouvelle contrainte sur le nouvel objet
                        serviceQuery.setCondition(queryMaker.getCondition());
                    }

                    // ajout de ce criteria dans la liste de tous les criteres
                    serviceQueries.add(serviceQuery);
                }
            } else {
                // on ne demande pas un business entity donc on ne modifie
                // pas les criteres
                serviceQueries = queries;
            }

            if (limitToFirst) {
                for (WikittyQuery query : serviceQueries) {
                    query.setFirst(0);
                    query.setLimit(1);
                    // lorsqu'on limit au premier, c'est qu'on utilise la methode
                    // pour retourne seulement un objet et pas un WikittyQueryResult
                    // on n'a donc pas a calculer les facets
                    query.setFacetExtension(false);
                    query.setFacetField();
                    query.setFacetQuery();
                }
            }

            List<WikittyQueryResult<String>> pagedResult = wikittyService.findAllByQuery(
                    securityToken, serviceQueries);

            result = new ArrayList<WikittyQueryResult<E>>(pagedResult.size());
            for (WikittyQueryResult<String> p : pagedResult) {
                result.add((WikittyQueryResult<E>)castTo(clazz, p));
            }
        }
        timeLog.log(start, "findAllByQuery<E>(List, limitToFirst)");
        return result;
    }

    /**
     * Search object that correspond to criteria and that have all extension
     * needed by BusinessEntity (clazz), if clazz is BusinessEntity class.
     * If one criteria is empty, find all extensions
     * for this criteria else if criteria is null return null.
     *
     * @param <E> object type
     * Can be Wikitty, BusinessEntity, String, Date, Number (returned is BigDecimal), Boolean, byte[]
     * @param clazz entity class
     * @param criterias criterias
     * @return paged result
     */
    public <E> List<WikittyQueryResult<E>> findAllByQuery(
            Class<E> clazz, List<WikittyQuery> criterias) {
        long start = TimeLog.getTime();
        List<WikittyQueryResult<E>> result = findAllByQuery(clazz, criterias, false);
        timeLog.log(start, "findAllByQuery<E>(List)");
        return result;
    }

    /**
     * Search object that correspond to criteria and that have all extension
     * needed by BusinessEntity (clazz), if clazz is BusinessEntity class.
     * If one criteria is empty, find all extensions
     * for this criteria else if criteria is null return null.
     *
     * @param <E> object type
     * Can be Wikitty, BusinessEntity, String, Date, Number (returned is BigDecimal), Boolean, byte[]
     * @param clazz entity class
     * @param criteria criteria
     * @return paged result
     */
    public <E> WikittyQueryResult<E> findAllByQuery(
            Class<E> clazz, WikittyQuery criteria) {
        long start = TimeLog.getTime();
        WikittyQueryResult<E> result = findAllByQuery(clazz,
                Collections.singletonList(criteria), false).get(0);
        timeLog.log(start, "findAllByQuery<E>(One)");
        return result;
    }

    /**
    /**
     * Search object that correspond to criteria and that have all extension
     * needed by BusinessEntity (clazz), if clazz is BusinessEntity class.
     * If one criteria is empty, find all extensions
     * for this criteria else if criteria is null return null.
     *
     * @param <E> object type
     * Can be Wikitty, BusinessEntity, String, Date, Number (returned is BigDecimal), Boolean, byte[]
     * @param clazz entity class
     * @param c1 criteria 1
     * @param c2 criteria 2
     * @param otherCriteria otherCriteria
     * @return paged result
     */
    public <E> WikittyQueryResult<E>[] findAllByQuery(
            Class<E> clazz, WikittyQuery c1, WikittyQuery c2, WikittyQuery... otherCriteria) {
        long start = TimeLog.getTime();
        List<WikittyQuery> criterias = new ArrayList<WikittyQuery>(otherCriteria.length + 2);
        Collections.addAll(criterias, c1, c2);
        Collections.addAll(criterias, otherCriteria);

        List<WikittyQueryResult<E>> resultList = findAllByQuery(clazz, criterias, false);
        WikittyQueryResult<E>[] result = resultList.toArray(new WikittyQueryResult[criterias.size()]);
        timeLog.log(start, "findAllByCriteria<Business>(Varargs)");
        return result;
    }

    ///////////////////////////////////////////////////////////////////////////
    //
    // FIND BY CRITERIA <E>
    //
    ///////////////////////////////////////////////////////////////////////////

    public <E> List<E> findByQuery(Class<E> clazz, List<WikittyQuery> criterias) {
        long start = TimeLog.getTime();

        List<WikittyQueryResult<E>> queryResult =
                findAllByQuery(clazz, criterias, true);
        List<E> result = new ArrayList<E>(queryResult.size());
        for (WikittyQueryResult<E> r : queryResult) {
            if (r.size() > 0) {
                result.add(r.peek());
            } else {
                result.add(null);
            }
        }

        timeLog.log(start, "multiFindByCriteria<E>(List>");
        return result;
    }

    public <E> E findByQuery(Class<E> clazz, WikittyQuery criteria) {
        long start = TimeLog.getTime();
        E result = null;
        if (criteria != null) {
            result = findByQuery(clazz, Collections.singletonList(criteria)).get(0);
        }
        timeLog.log(start, "findByQuery<E>(One)");
        return result;
    }

    public <E> E[] findByQuery(
            Class<E> clazz, WikittyQuery c1, WikittyQuery c2, WikittyQuery... otherCriteria) {
        long start = TimeLog.getTime();

        List<WikittyQuery> criterias = new ArrayList<WikittyQuery>(otherCriteria.length + 2);
        Collections.addAll(criterias, c1, c2);
        Collections.addAll(criterias, otherCriteria);

        List<E> resultList = findByQuery(clazz, criterias);
        E[] result = resultList.toArray((E[])Array.newInstance(clazz, resultList.size()));

        timeLog.log(start, "findByCriteria<E>(Varargs)");
        return result;
    }

    ///////////////////////////////////////////////////////////////////////////
    //
    // FIND ALL FIELD OR ID BY CRITERIA <String>
    //
    ///////////////////////////////////////////////////////////////////////////

    public List<WikittyQueryResult<String>> findAllByQuery(List<WikittyQuery> criteria) {
        long start = TimeLog.getTime();
        List<WikittyQueryResult<String>> result = null;
        if (criteria != null) {
            result = wikittyService.findAllByQuery(securityToken, criteria);
        }
        timeLog.log(start, "findAllByCriteria(List)");
        return result;
    }

    public WikittyQueryResult<String> findAllByQuery(WikittyQuery criteria) {
        long start = TimeLog.getTime();
        WikittyQueryResult<String> result = null;
        if (criteria != null) {
            result = findAllByQuery(
                    Collections.singletonList(criteria)).get(0);
        }
        timeLog.log(start, "findAllByCriteria(One)");
    	return result;
    }

    public WikittyQueryResult<String>[] findAllByQuery(
            WikittyQuery c1, WikittyQuery c2, WikittyQuery ... otherCriteria) {
        long start = TimeLog.getTime();

        List<WikittyQuery> criterias = new ArrayList<WikittyQuery>(otherCriteria.length + 2);
        Collections.addAll(criterias, c1, c2);
        Collections.addAll(criterias, otherCriteria);

        List<WikittyQueryResult<String>> resultList = findAllByQuery(criterias);
        WikittyQueryResult<String>[] result = resultList.toArray(new WikittyQueryResult[criterias.size()]);

        timeLog.log(start, "findAllByCriteria(Varargs)");
    	return result;
    }

    ///////////////////////////////////////////////////////////////////////////
    //
    // FIND ID BY CRITERIA <String>
    //
    ///////////////////////////////////////////////////////////////////////////

    public List<String> findByQuery(List<WikittyQuery> criteria) {
        long start = TimeLog.getTime();
        List<String> result = null;
        if (criteria != null) {
            result = wikittyService.findByQuery(securityToken, criteria);
        }
        timeLog.log(start, "findByCriteria(List)");
    	return result;
    }

    public String findByQuery(WikittyQuery criteria) {
        long start = TimeLog.getTime();
        String result = null;
        if (criteria != null) {
            result = findByQuery(Collections.singletonList(criteria)).get(0);
        }
        timeLog.log(start, "findByCriteria(One)");
    	return result;
    }

    public String[] findByQuery(
            WikittyQuery c1, WikittyQuery c2, WikittyQuery... otherCriteria) {
        long start = TimeLog.getTime();

        List<WikittyQuery> criterias = new ArrayList<WikittyQuery>(otherCriteria.length + 2);
        Collections.addAll(criterias, c1, c2);
        Collections.addAll(criterias, otherCriteria);

        List<String> resultList = findByQuery(criterias);
        String[] result = resultList.toArray(new String[criterias.size()]);

        timeLog.log(start, "findByCriteria(Varargs)");
        return result;
    }

    ///////////////////////////////////////////////////////////////////////////
    //
    // FIND BY TREE NODE
    //
    ///////////////////////////////////////////////////////////////////////////

    /**
     * Recupere une portion d'arbre a partir de l'id passer en parametre. L'id
     * doit etre celui d'un WikittyTreeNode. Ce WikittyTreeNode est alors le
     * root de l'arbre retourne.
     *
     * Return Wikitty in result, those Wikitties have WikittyTreeNode extension
     *
     * @param wikittyId root
     * @param depth profondeur de noeud a recuperer
     * @param count vrai si l'on veut le nombre de piece attaches sur le noeud
     * (piece des enfants compris)
     * @param filter filter pour compter les pieces attachees
     * @return treeNodeResult of wikitty
     *
     * @since 3.1
     */
    public WikittyQueryResultTreeNode<Wikitty> findTreeNode(
            String wikittyId, int depth, boolean count, WikittyQuery filter) {
        long start = TimeLog.getTime();

        WikittyQueryResultTreeNode<String> resultId = wikittyService.findTreeNode(
                securityToken, wikittyId, depth, count, filter);

        RetrieveIdVisitor retrieveIdVisitor = new RetrieveIdVisitor();
        resultId.acceptVisitor(retrieveIdVisitor);

        List<String> ids = retrieveIdVisitor.getIds();
        List<Wikitty> wikitties = restore(ids);
        
        IdToObjectConverter<Wikitty> converter =
                new IdToObjectConverter<Wikitty>(ids, wikitties);

        ConvertTreeVisitor<Wikitty> convertVisitor =
                new ConvertTreeVisitor<Wikitty>(converter);

        resultId.acceptVisitor(convertVisitor);

        WikittyQueryResultTreeNode<Wikitty> result = convertVisitor.getTree();
        timeLog.log(start, "findTreeNode<Wikitty>");
        return result;
    }

    /**
     * Recupere une portion d'arbre a partir de l'id passer en parametre. L'id
     * doit etre celui d'un WikittyTreeNode. Ce WikittyTreeNode est alors le
     * root de l'arbre retourne.
     *
     * Return E in result
     *
     * @param clazz business class wanted to replace id in TreeNodeResult
     * @param wikittyId root
     * @param depth profondeur de noeud a recuperer
     * @param count vrai si l'on veut le nombre de piece attaches sur le noeud (piece des enfants compris)
     * @param filter filter pour compter les pieces attachees
     * @return
     *
     * @since 3.1
     */
    public <E extends BusinessEntity> WikittyQueryResultTreeNode<E> findTreeNode(
            Class<E> clazz, String wikittyId, int depth,
            boolean count, WikittyQuery filter) {
        long start = TimeLog.getTime();

        WikittyQueryResultTreeNode<String> resultId = wikittyService.findTreeNode(
                securityToken, wikittyId, depth, count, filter);

        RetrieveIdVisitor retrieveIdVisitor = new RetrieveIdVisitor();
        resultId.acceptVisitor(retrieveIdVisitor);

        List<String> ids = retrieveIdVisitor.getIds();
        List<E> wikitties = restore(clazz, ids);

        IdToObjectConverter<E> converter =
                new IdToObjectConverter<E>(ids, wikitties);

        ConvertTreeVisitor<E> convertVisitor =
                new ConvertTreeVisitor<E>(converter);

        resultId.acceptVisitor(convertVisitor);

        WikittyQueryResultTreeNode<E> result = convertVisitor.getTree();
        timeLog.log(start, "findTreeNode");
        return result;
    }

    /**
     * Used to collect all node id
     * @since 3.1
     */
    static private class RetrieveIdVisitor implements WikittyQueryResultTreeNode.Visitor<String> {

        protected List<String> ids = new ArrayList<String>();

        public List<String> getIds() {
            return ids;
        }

        @Override
        public boolean visitEnter(WikittyQueryResultTreeNode<String> node) {
            String id = node.getObject();
            ids.add(id);
            return true;
        }

        @Override
        public boolean visitLeave(WikittyQueryResultTreeNode<String> node) {
            return true;
        }
    }

    /**
     * Converti un id en son object WikittyTreeNode
     * @since 3.1
     */
    static private class IdToObjectConverter<T> implements ConvertTreeVisitor.Converter<String, T> {
        protected Map<String, T> objects = new HashMap<String, T>();
        protected String securityToken;
        protected WikittyService wikittyService;
        public IdToObjectConverter(List<String> ids, List<T> objectList) {

            for (int i = 0; i < ids.size(); i++) {
                this.objects.put(ids.get(i), objectList.get(i));
            }
        }

        @Override
        public T convert(String id) {
            T result = objects.get(id);
            return result;
        }
    }

    /**
     * Parcours un TreeNodeResult et en fait une copie en modifiant le type
     * d'objet stocker dans le noeud grace a un converter, si le converter
     * est null une exception est levee
     *
     * @param <TARGET> le type d'objet pour le nouvel arbre
     * @since 3.1
     */
    static private class ConvertTreeVisitor<TARGET extends Serializable>
            implements WikittyQueryResultTreeNode.Visitor<String> {

        static private interface Converter<SOURCE, TARGET> {
            public TARGET convert(SOURCE o);
        }
        protected Converter<String, TARGET> converter;
        protected WikittyQueryResultTreeNode<TARGET> tree = null;
        protected LinkedList<WikittyQueryResultTreeNode<TARGET>> stack =
                new LinkedList<WikittyQueryResultTreeNode<TARGET>>();

        public ConvertTreeVisitor(Converter<String, TARGET> converter) {
            this.converter = converter;
            if (converter == null) {
                throw new IllegalArgumentException("Converter can't be null");
            }
        }

        public WikittyQueryResultTreeNode<TARGET> getTree() {
            return tree;
        }

        @Override
        public boolean visitEnter(WikittyQueryResultTreeNode<String> node) {
            String id = node.getObject();
            int count = node.getAttCount();
            
            TARGET object = converter.convert(id);
            WikittyQueryResultTreeNode<TARGET> newNode = new WikittyQueryResultTreeNode<TARGET>(
                    object, count);

            WikittyQueryResultTreeNode<TARGET> parent = stack.peekLast();
            if (parent == null) {
                // le premier noeud, donc le root a retourner plus tard
                tree = newNode;
            } else {
                parent.add(newNode);
            }

            stack.offerLast(newNode);

            return true;
        }

        @Override
        public boolean visitLeave(WikittyQueryResultTreeNode<String> node) {
            stack.pollLast();
            return true;
        }
    }

    /**
     * Recupere une portion d'arbre a partir de l'id passer en parametre. L'id
     * doit etre celui d'un WikittyTreeNode. Ce WikittyTreeNode est alors le
     * root de l'arbre retourne.
     *
     * Return just wikitty Id in result
     *
     * @param wikittyId
     * @param depth
     * @param count
     * @param filter
     * @return
     * @since 3.1
     */
    public WikittyQueryResultTreeNode<String> findAllIdTreeNode(
            String wikittyId, int depth, boolean count, WikittyQuery filter) {
        long start = TimeLog.getTime();
        WikittyQueryResultTreeNode<String> result = wikittyService.findTreeNode(
                securityToken, wikittyId, depth, count, filter);

        timeLog.log(start, "findAllIdTreeNode");
    	return result;
    }

    /**
     * Delete specified tree node and all sub nodes.
     * 
     * @param treeNodeId tree node id to delete
     * @return {@true} if at least one node has been deleted
     */
    public WikittyEvent deleteTree(String treeNodeId) {
        long start = TimeLog.getTime();
        WikittyEvent result = wikittyService.deleteTree(securityToken,treeNodeId);
        
        timeLog.log(start, "deleteTree");
        return result;
    }

    public Wikitty restoreVersion(String wikittyId, String version) {
        long start = TimeLog.getTime();
        Wikitty result = wikittyService.restoreVersion(
                securityToken, wikittyId, version);
        
        timeLog.log(start, "restoreVersion");
        return result;
    }

    /**
     * Manage Update and creation.
     *
     * @param ext extension to be persisted
     * @return update response
     */
    public WikittyEvent storeExtension(WikittyExtension ext) {
        long start = TimeLog.getTime();
        WikittyEvent response =
                wikittyService.storeExtension(securityToken, ext);
        
        timeLog.log(start, "storeExtension");
        return response;
    }

    /**
     * Manage Update and creation.
     *
     * @param exts list of wikitty extension to be persisted
     * @return update response
     */
    public WikittyEvent storeExtension(Collection<WikittyExtension> exts) {
        long start = TimeLog.getTime();
        WikittyEvent response =
                wikittyService.storeExtension(securityToken, exts);
        
        timeLog.log(start, "storeExtension<list>");
        return response;
    }

    /**
     * Load extension from id. Id is 'name[version]'.
     * 
     * @param extensionId extension id to restore
     * @return the corresponding object, exception if no such object found.
     */
    public WikittyExtension restoreExtension(String extensionId) {
        long start = TimeLog.getTime();
        WikittyExtension extension = wikittyService.restoreExtension(securityToken, extensionId);
        
        timeLog.log(start, "restoreExtension");
        return extension;
    }

    /**
     * Search extension with name in last version.
     * 
     * @param extensionName extension name
     * @return the corresponding object, exception if no such object found.
     */
    public WikittyExtension restoreExtensionLastVersion(String extensionName) {
        long start = TimeLog.getTime();
        WikittyExtension extension = wikittyService.restoreExtensionLastVersion(securityToken, extensionName);
        
        timeLog.log(start, "restoreExtensionLastVersion");
        return extension;
    }

    /**
     * Search extension with name in last version.
     *
     * @param extensionNames extension name
     * @return extension wanted with dependencies extensions at head of list
     */
    public List<WikittyExtension> restoreExtensionAndDependenciesLastVesion(Collection<String> extensionNames) {
        long start = TimeLog.getTime();
        List<WikittyExtension> result =
                wikittyService.restoreExtensionAndDependenciesLastVesion(
                securityToken, extensionNames);

        timeLog.log(start, "restoreExtensionAndDependenciesLastVesion");
        return result;

    }

    public void deleteExtension(String extName) {
        long start = TimeLog.getTime();
        wikittyService.deleteExtension(securityToken, extName);

        timeLog.log(start, "deleteExtension");
    }

    public void deleteExtension(Collection<String> extNames) {
        long start = TimeLog.getTime();
        wikittyService.deleteExtension(securityToken, extNames);

        timeLog.log(start, "deleteExtension<list>");
    }

    /**
     * Return all extension id (ex: "extName[version])").
     * 
     * @return extension id list
     */
    public List<String> getAllExtensionIds() {
        long start = TimeLog.getTime();
        List<String> result = wikittyService.getAllExtensionIds(securityToken);
        
        timeLog.log(start, "getAllExtensionIds");
        return result;
    }
    
    /**
     * Return all extension id (ex: "extName[version])") where
     * {@code extensionName} is required.
     * 
     * @param extensionName extension name
     * @return extensions
     */
    public List<String> getAllExtensionsRequires(String extensionName) {
        long start = TimeLog.getTime();
        List<String> result = wikittyService.getAllExtensionsRequires(securityToken, extensionName);
        
        timeLog.log(start, "getAllExtensionsRequires");
        return result;
    }

    /**
     * Use with caution : It will delete ALL indexes from search engine !
     * This operation should be disabled in production environment.
     */
    public WikittyEvent clear() {
        long start = TimeLog.getTime();
        WikittyEvent result = wikittyService.clear(securityToken);
        
        timeLog.log(start, "clear");
        return result;
    }

    /**
     * Synchronize search engine with wikitty storage engine, i.e. clear and
     * reindex all object.
     */
    public void syncSearchEngine() {
        long start = TimeLog.getTime();
        wikittyService.syncSearchEngine(securityToken);
        
        timeLog.log(start, "syncSearchEngine");
    }

    /**
     * Method to get the Wikitty encapsulated into a BusinessEntity
     *
     * This method can go to serveur side, if BusinessEntity is not
     * BusinessEntityImpl, this append when use GWT
     *
     * @param entity the BusinessEntity encapsulating the Wikitty
     * @return the wikitty encapsulated
     */
    public Wikitty getWikitty(BusinessEntity entity){
        long start = TimeLog.getTime();
        Wikitty result;

        if (entity instanceof BusinessEntityImpl) {
            result = ((BusinessEntityImpl) entity).getWikitty();
        } else {
            String id = entity.getWikittyId();

            result = restore(id);

            //try settings all fields except version
            try {
                //get all fields
                Class entityClass = entity.getClass();
                Field[] fields = entityClass.getDeclaredFields();

                for(Field field:fields){
                    //for each field that got WikittyField annotation
                    if (field.isAnnotationPresent(WikittyField.class)){

                        //get the attribute's wikitty fqn
                        WikittyField annotation = field.getAnnotation(WikittyField.class);
                        String fieldFQN = annotation.fqn();

                        //set the value
                        Method m = entityClass.getMethod("get" + StringUtils.capitalize(field.getName()));
                        Object value = m.invoke(entity);

                        result.setFqField(fieldFQN, value);
                    }
                }
            } catch (Exception eee) {
                throw new WikittyException("Could not transform entity to Wikitty", eee);
            }

            //manually set version
            result.setVersion(entity.getWikittyVersion());
        }
        
        timeLog.log(start, "getWikitty");
        return result;
    }

    /**
     * Check that the logged in user is in a group. A #SecurityException might
     * be thrown at runtime if the #WikittyUser session timed out.
     * @param groupName the name of the group to check
     * @return true is the logged in user is in the group
     */
    public boolean isMember(String groupName) {
        long start = TimeLog.getTime();
        boolean result = false;

        WikittyUser user = getLoggedInUser();

        //Find the group from its name
        WikittyQuery criteria = new WikittyQueryMaker().and()
                .exteq(WikittyGroup.EXT_WIKITTYGROUP)
                .eq(WikittyGroup.FQ_FIELD_WIKITTYGROUP_NAME, groupName)
                .end();

        Wikitty group = findByQuery(Wikitty.class, criteria);

        if (group != null && user != null) {
            result = WikittySecurityUtil.isMember(wikittyService, securityToken,
                    user.getWikittyId(), group.getId());
        }

        timeLog.log(start, "isMember");
        return result;
    }

    /**
     * Get the #WikittyUser that is logged in. A #SecurityException might be
     * thrown at runtime if the #WikittyUser session timed out.
     * @return the logged in #WikittyUser
     */
    public WikittyUser getLoggedInUser() {
        long start = TimeLog.getTime();

        String userId = WikittySecurityUtil.getUserForToken(wikittyService,
                securityToken);

        WikittyUser user = restore(WikittyUser.class, userId);

        timeLog.log(start, "getLoggedInUser");
        return user;
    }
    
    /**
     * Convert all result to the wanted type and return new WikittyQueryResult
     * with this new result list. For business object transformation, if some
     * result don't have the right extension (clazz) this extension is
     * automatically added.
     * 
     * @param queryResult result to convert
     * @param target to cast into.
     * Can be Wikitty, BusinessEntity, String, Date, Number (returned is BigDecimal), Boolean, byte[]
     * @return new WikittyQueryResult with element in right class or Exception
     * if conversion is impossible
     */
    public <E> WikittyQueryResult<E> castTo(Class<E> target,
            WikittyQueryResult queryResult) {
        List<E> castedResult;

        if (queryResult.size() == 0) {
            // on ne fait rien on met juste une nouvelle liste vide
            castedResult = new ArrayList<E>();
        } else if (target.isAssignableFrom(Wikitty.class)) {
            // On veut des Wikitties en sortie

            if (queryResult.peek() instanceof Wikitty) {
                // W, rien a faire
                castedResult = queryResult.getAll();
            } else if (queryResult.peek() instanceof BusinessEntityImpl) {
                // BusinessEntityImpl, il faut recuperer les wikitty
                castedResult = new ArrayList<E>(queryResult.size());
                for (BusinessEntityImpl e : (WikittyQueryResult<BusinessEntityImpl>)queryResult) {
                    castedResult.add((E)e.getWikitty());
                }
            } else if (queryResult.peek() instanceof String) {
                // String, il faut faire un restore
                // le queryResult courant contient des Ids
                // Si ce n'est pas le cas, ca veut dire que le developpeur utilisant
                // ce queryResult ne sait pas ce qu'il fait :)
                List<String> ids = (List<String>) queryResult.getAll();
                castedResult = (List<E>)getWikittyService().restore(securityToken, ids);
            } else {
                throw new ClassCastException("WikittyQueryResult don't contains"
                        + " object convertible to Wikitty"
                        + " (accepted Wikitty, BusinessEntityImpl, String id) but "
                        + queryResult.peek().getClass());
            }
        } else if (target.isAssignableFrom(BusinessEntityImpl.class)) {
            // on commence par tout mettre en Wikitty, en utilisant le if du dessus
            WikittyQueryResult<Wikitty> resultTmp = castTo(Wikitty.class, queryResult);
            castedResult = (List<E>)WikittyUtil.newInstance((Class<BusinessEntity>)target, resultTmp.getAll());
        } else if (target.isAssignableFrom(Number.class)) {
            castedResult = new ArrayList<E>(queryResult.size());
            for (Object o : queryResult) {
                BigDecimal v = WikittyUtil.toBigDecimal(o);
                castedResult.add((E)v);
            }
        } else if (target.isAssignableFrom(Date.class)) {
            castedResult = new ArrayList<E>(queryResult.size());
            for (Object o : queryResult) {
                Date v = WikittyUtil.toDate(o);
                castedResult.add((E)v);
            }
        } else if (target.isAssignableFrom(Boolean.class)) {
            castedResult = new ArrayList<E>(queryResult.size());
            for (Object o : queryResult) {
                Boolean v = WikittyUtil.toBoolean(o);
                castedResult.add((E)v);
            }
        } else if (target.isAssignableFrom(byte[].class)) {
            castedResult = new ArrayList<E>(queryResult.size());
            for (Object o : queryResult) {
                byte[] v = WikittyUtil.toBinary(o);
                castedResult.add((E)v);
            }
        } else if (target.isAssignableFrom(String.class)) {
            castedResult = new ArrayList<E>(queryResult.size());
            for (Object o : queryResult) {
                String v = WikittyUtil.toString(o);
                castedResult.add((E)v);
            }
        }else {
            throw new ClassCastException(String.format(
                    "WikittyQueryResult don't contains"
                    + " object convertible to %s"
                    + " (accepted Wikitty, BusinessEntityImpl, String id, Date,"
                    + " BigDecimal, Boolean, byte[]) but '%s'",
                    target.getName(), queryResult.peek().getClass()));
        }

        WikittyQueryResult<E> result = new WikittyQueryResult<E>(
                queryResult.getQueryName(),
                queryResult.getFirst(), queryResult.getTotalResult(),
                queryResult.getQueryString(), queryResult.getFacets(),
                castedResult);
        return result;
    }

}
