/* *##%
 * Copyright (c) 2009 Sharengo, Guillaume Dufrene, Benjamin POUSSIN.
 *
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 *##%*/

package org.nuiton.wikitty;

import java.beans.PropertyChangeListener;
import java.beans.PropertyChangeSupport;
import java.io.Serializable;
import java.math.BigDecimal;
import java.util.AbstractList;
import java.util.AbstractSet;
import java.util.ArrayList;
import java.util.Collection;
import java.util.Date;
import java.util.HashMap;
import java.util.HashSet;
import java.util.Iterator;
import java.util.LinkedHashMap;
import java.util.List;
import java.util.Map;
import java.util.Set;

/**
 *
 * @author poussin
 * @version $Revision: 193 $
 *
 * Last update: $Date: 2010-07-21 11:44:11 +0200 (mer., 21 juil. 2010) $
 * by : $Author: echatellier $
 */
public class Wikitty implements Serializable {

    /** serialVersionUID. */
    private static final long serialVersionUID = 4910886672760691052L;

    /** Technical id for this wikitty object. id must be never null. */
    protected String id;

    /** Current version of this wikitty object. */
    protected String version = WikittyUtil.DEFAULT_VERSION;

    /** If not null, date of deletion, if date this object is marked as deleted. */
    protected Date deleteDate = null;

    /**
     * Used to add property change support to wikitty object.
     * 
     * Warning, this field can be null after deserialization.
     */
    private transient PropertyChangeSupport propertyChange;

    /**
     * key: field name prefixed by extension name (dot separator)
     * value: value of field
     */
    protected HashMap<String, Object> fieldValue = new HashMap<String, Object>();

    /**
     * all field name currently modified (field name = extension . fieldname)
     */
    protected Set<String> fieldDirty = new HashSet<String>();

    /**
     * Map is LinkedHashMap to maintains order like user want
     * key: extension name
     * value: extension definition
     */
    protected Map<String, WikittyExtension> extensions =
            new LinkedHashMap<String, WikittyExtension>();
      

    public Wikitty() {
        this(null);
    }

    public Wikitty(String id) {
        if(id == null) {
            this.id = WikittyUtil.genUID();
        } else {
            this.id = id;
        }
    }

    /**
     * Always call this method because field is transient.
     * 
     * @return
     */
    protected PropertyChangeSupport getPropertyChangeSupport() {
        if (propertyChange == null) {
            propertyChange = new PropertyChangeSupport(this);
        }
        return propertyChange;
    }

    public synchronized void addPropertyChangeListener(
            PropertyChangeListener listener) {
        getPropertyChangeSupport().addPropertyChangeListener(listener);
    }


    public synchronized void removePropertyChangeListener(
            PropertyChangeListener listener) {
        getPropertyChangeSupport().removePropertyChangeListener(listener);
    }


    public synchronized void addPropertyChangeListener(String propertyName,
            PropertyChangeListener listener) {
        getPropertyChangeSupport().addPropertyChangeListener(propertyName, listener);
    }


    public synchronized void removePropertyChangeListener(String propertyName,
            PropertyChangeListener listener) {
        getPropertyChangeSupport().removePropertyChangeListener(propertyName, listener);
    }

    public String getId() {
        return id;
    }

    public boolean isDeleted() {
        boolean result = deleteDate != null;
        return result;
    }

    public Date getDeleteDate() {
        return deleteDate;
    }

    /**
     * Server only used
     * @param delete
     */
    public void setDeleteDate(Date delete) {
        this.deleteDate = delete;
    }

    /**
     * mark field as dirty
     * @param ext
     * @param fieldName
     */
    protected void setFieldDirty(String ext, String fieldName,
            Object oldValue, Object newValue) {
        String key = ext + "." + fieldName;
        fieldDirty.add(key);
        version = WikittyUtil.incrementMinorRevision(version);
        getPropertyChangeSupport().firePropertyChange(key, oldValue, newValue);
    }

    public void addExtension(WikittyExtension ext) {
        String required = ext.getRequires();
        if (required != null && !required.isEmpty() &&
                !extensions.containsKey(required)) {
            throw new WikittyException(String.format(
                    "You try to add extension '%s' that" +
                    " required not available extension '%s' in this wikitty",
                    ext.getName(), required));
        }
        extensions.put(ext.name, ext);
    }

    public void addExtension(List<WikittyExtension> exts) {
        for (WikittyExtension ext : exts) {
            extensions.put(ext.name, ext);
        }
    }

    public boolean hasExtension(String extName) {
        return extensions.containsKey(extName);
    }

    public boolean hasField(String extName, String fieldName) {
        boolean result = false;
        WikittyExtension ext = extensions.get(extName);
        if (ext != null) {
            result = ext.getFieldType(fieldName) != null;
        }
        return result;
    }

    public WikittyExtension getExtension(String ext) {
        WikittyExtension result = extensions.get(ext);
        return result;
    }

    public Collection<String> getExtensionNames() {
        Collection<String> result = extensions.keySet();
        return result;
    }

    public Collection<WikittyExtension> getExtensions() {
        Collection<WikittyExtension> result = extensions.values();
        return result;
    }

    public Collection<WikittyExtension> getExtensionDependencies(String ext, boolean recursively) {
        Collection<WikittyExtension> result = new HashSet<WikittyExtension>();
        Collection<WikittyExtension> all = extensions.values();
        for (WikittyExtension dependency : all) {
            String requires = dependency.getRequires();
            if(requires != null && !requires.isEmpty() && requires.equals(ext)) {
                result.add(dependency);
                if(recursively) {
                    String dependencyName = dependency.getName();
                    Collection<WikittyExtension> dependencies = getExtensionDependencies(dependencyName, recursively);
                    result.addAll(dependencies);
                }
            }
        }
        return result;
    }

    /**
     * return field type for the given fieldName.
     * @param fqfieldName fully qualified fieldName extension.fieldname
     * @return field type
     */
    public FieldType getFieldType(String fqfieldName) {
        try {
            String[] field = fqfieldName.split("\\.");
            WikittyExtension ext = getExtension(field[0]);
            if (ext == null) {
                throw new WikittyException(String.format(
                    "extension '%s' doesn't exists", field[0]));
            } else {
                String fieldName = field[1];
                int crochet = fieldName.indexOf("[");
                if (crochet != -1) {
                    fieldName = fieldName.substring(0, crochet);
                }
                FieldType result = ext.getFieldType(fieldName);
                if (result == null) {
                    throw new WikittyException(String.format(
                            "field '%s' doesn't exists on extension '%s'", fieldName, field[0]));

                }
                return result;
            }
        } catch (Exception eee) {
            throw new WikittyException(
                    String.format("Field %s is not a fully qualified field name", fqfieldName), 
                    eee
            );
        }
    }

    public void setField(String ext, String fieldName, Object value) {
        if (! hasField(ext, fieldName)) {
            String def = "";
            for ( WikittyExtension extension : extensions.values() ) {
                def += extension.toDefinition() + "\n";
            }
            throw new WikittyException(String.format(
                    "field '%s' is not valid, extensions definition : %s", ext + "." + fieldName, def));
        }
        String key = ext + "." + fieldName;

        // take old value if needed
        Object oldValue = null;
        if (getPropertyChangeSupport().hasListeners(key)) {
            oldValue = fieldValue.get(key);
        }

        // put new value
        FieldType fieldType = getExtension(ext).getFieldType(fieldName);
        Object validValue = fieldType.getValidValue(value);
        fieldValue.put(key, validValue);

        // mark field dirty and call listener
        setFieldDirty(ext, fieldName, oldValue, validValue);
    }

    public Object getFieldAsObject(String ext, String fieldName) {
        if (!hasField(ext, fieldName)) {
            throw new WikittyException(String.format(
                    "field '%s' is not a valid field",
                    ext + "." + fieldName));
        }
        String key = ext + "." + fieldName;
        Object result = fieldValue.get(key);
        return result;
    }

    public boolean getFieldAsBoolean(String ext, String fieldName) {
        Object value = getFieldAsObject(ext, fieldName);
        try {
            boolean result = WikittyUtil.toBoolean(value);
            return result;
        } catch (WikittyException eee) {
            throw new WikittyException(String.format(
                    "field '%s' is not a valid boolean",
                    ext + "." + fieldName), eee);
        }
    }

    public BigDecimal getFieldAsBigDecimal(String ext, String fieldName) {
        Object value = getFieldAsObject(ext, fieldName);
        try {
            BigDecimal result = WikittyUtil.toBigDecimal(value);
            return result;
        } catch (WikittyException eee) {
            throw new WikittyException(String.format(
                    "field '%s' is not a valid numeric",
                    ext + "." + fieldName), eee);
        }
    }

    public int getFieldAsInt(String ext, String fieldName) {
        try {
            BigDecimal value = getFieldAsBigDecimal(ext, fieldName);
            int result = value.intValue();
            return result;
        } catch (WikittyException eee) {
            throw new WikittyException(String.format(
                    "field '%s' is not a valid int",
                    ext + "." + fieldName), eee);
        }
    }

    public long getFieldAsLong(String ext, String fieldName) {
        try {
            BigDecimal value = getFieldAsBigDecimal(ext, fieldName);
            long result = value.longValue();
            return result;
        } catch (WikittyException eee) {
            throw new WikittyException(String.format(
                    "field '%s' is not a valid int",
                    ext + "." + fieldName), eee);
        }
    }

    public float getFieldAsFloat(String ext, String fieldName) {
        try {
            BigDecimal value = getFieldAsBigDecimal(ext, fieldName);
            float result = value.floatValue();
            return result;
        } catch (WikittyException eee) {
            throw new WikittyException(String.format(
                    "field '%s' is not a valid float",
                    ext + "." + fieldName), eee);
        }
    }

    public double getFieldAsDouble(String ext, String fieldName) {
        try {
            BigDecimal value = getFieldAsBigDecimal(ext, fieldName);
            double result = value.doubleValue();
            return result;
        } catch (WikittyException eee) {
            throw new WikittyException(String.format(
                    "field '%s' is not a valid float",
                    ext + "." + fieldName), eee);
        }
    }

    public String getFieldAsString(String ext, String fieldName) {
        Object value = getFieldAsObject(ext, fieldName);
        try {
            String result = WikittyUtil.toString(value);
            return result;
        } catch (WikittyException eee) {
            throw new WikittyException(String.format(
                    "field '%s' is not a valid String",
                    ext + "." + fieldName), eee);
        }
    }

    public Date getFieldAsDate(String ext, String fieldName) {
        Object value = getFieldAsObject(ext, fieldName);
        try {
            Date result = WikittyUtil.toDate(value);
            return result;
        } catch (WikittyException eee) {
            throw new WikittyException(String.format(
                    "field '%s' is not a valid Date",
                    ext + "." + fieldName), eee);
        }
    }

    /**
     * return wikitty id and not wikitty objet because this method can be call
     * on server or client side and it's better to keep conversion between id
     * and objet to the caller
     * @param ext extension name where this field must to be
     * @param fieldName the field name
     * @return id of wikitty object or null
     * @throws org.nuiton.wikitty.WikittyException
     */
    public String getFieldAsWikitty(String ext, String fieldName) {
        Object value = getFieldAsObject(ext, fieldName);
        String result = WikittyUtil.toWikitty(value);
        return result;
    }

    /**
     * If object is a set, it is automatically transform to list.
     * @param <E>
     * @param clazz
     * @return unmodifiable list
     */
    public <E> List<E> getFieldAsList(String ext, String fieldName, final Class<E> clazz) {
        try {
            final Collection<E> collection = (Collection<E>) getFieldAsObject(ext, fieldName);
            if (collection != null) {
                // return unmodiable collection that check type of element
                return new AbstractList<E>() {
                    List<E> contained = new ArrayList<E>(collection);
                    @Override public E get(int index) {
                        return WikittyUtil.cast( contained.get(index), clazz );
                    }
                    @Override public int size() {
                        return contained.size();
                    }
                };
            }
            return null;
        } catch (Exception eee) {
            throw new WikittyException(String.format(
                    "Can't add value to field '%s'",
                    ext + "." + fieldName), eee);
        }
    }

    /**
     *
     * @param <E>
     * @param clazz
     * @return unmodifiable list
     */
    public <E> Set<E> getFieldAsSet(String ext, String fieldName, final Class<E> clazz) {
        try {
            final Set<E> result = (Set<E>) getFieldAsObject(ext, fieldName);
            if (result != null) {
                // return unmodifable Set
                return new AbstractSet<E>() {
                    Set<E> contained = result;
                    @Override public int size() {
                        return contained.size();
                    }
                    @Override
                    public Iterator<E> iterator() {
                        return new Iterator<E>() {
                            Iterator containedIterator = contained.iterator();
                            public boolean hasNext() {
                                return containedIterator.hasNext();
                            }

                            public E next() {
                                Object o = containedIterator.next();
                                return WikittyUtil.cast(o, clazz);
                            }

                            public void remove() {
                                throw new UnsupportedOperationException("Not supported operation");
                            }
                        };

                    }
                };
            }
            return result;
        } catch (Exception eee) {
            throw new WikittyException(String.format(
                    "Can't add value to field '%s'",
                    ext + "." + fieldName), eee);
        }
    }

    public void addToField(String ext, String fieldName, Object value) {
        try {
            FieldType fieldType = getExtension(ext).getFieldType(fieldName);
            Collection col = (Collection) getFieldAsObject(ext, fieldName);
            if (col == null) {
                if (fieldType.isUnique()) {
                    col = new HashSet();
                } else {
                    col = new ArrayList();
                }
                col.add(value);
                setField(ext, fieldName, col);
                // no call dirty, because already done in setField
            } else {
                // check upper bound only if col exists,
                // because ask upper bound == 0 is ridiculous

                if (fieldType.isUnique()) {
                    if (!col.contains(value)) {
                        // only add if not already in collection (unique)
                        if (col.size() + 1 > fieldType.getUpperBound()) {
                            // if upper bound reached, throw an exception
                            throw new WikittyException(String.format(
                                    "Can't add value for field '%s', upper bound is reached",
                                    ext + "." + fieldName));
                        }
                        col.add(value);
                        setFieldDirty(ext, fieldName, null, col);
                    }
                } else {
                    if (col.size() + 1 > fieldType.getUpperBound()) {
                        throw new WikittyException(String.format(
                                "Can't add value for field '%s', upper bound is reached",
                                ext + "." + fieldName));
                    }
                    col.add(value);
                    setFieldDirty(ext, fieldName, null, col);
                }
            }
        } catch (Exception eee) {
            throw new WikittyException(String.format(
                    "Can't add value to field '%s'",
                    ext + "." + fieldName), eee);
        }
    }

    public void removeFromField(String ext, String fieldName, Object value) {
        try {
            Collection col = (Collection) getFieldAsObject(ext, fieldName);
            if (col != null) {
                FieldType type = getExtension(ext).getFieldType(fieldName);
                if (col.contains(value)) {
                    if (col.size() - 1 < type.getLowerBound()) {
                        throw new WikittyException(String.format(
                                "Can't remove value for field '%s', lower bound is reached",
                                ext + "." + fieldName));
                    } else {
                        if (col.remove(value)) {
                            // field is dirty only if remove is done
                            setFieldDirty(ext, fieldName, null, col);
                        }
                    }
                }
            }
        } catch (Exception eee) {
            throw new WikittyException(String.format(
                    "Can't remove value for field '%s'",
                    ext + "." + fieldName), eee);
        }
    }

    public void clearField(String ext, String fieldName) {
        FieldType type = getExtension(ext).getFieldType(fieldName);
        if (type.getLowerBound() > 0) {
            throw new WikittyException(String.format(
                    "Can't clear values for field '%s', lower bound is > 0",
                    ext + "." + fieldName));
        }
        try {
            Collection col = (Collection) getFieldAsObject(ext, fieldName);
            if (col != null) {
                col.clear();
                setFieldDirty(ext, fieldName, null, col);
            }
        } catch (Exception eee) {
            throw new WikittyException(String.format(
                    "Can't clear value for field '%s'",
                    ext + "." + fieldName), eee);
        }
    }

    @Override
    public boolean equals(Object obj) {
        boolean result = false;
        if (obj instanceof Wikitty) {
            Wikitty other = (Wikitty) obj;
            result = id.equals(other.id);
        }
        return result;
    }

    @Override
    public int hashCode() {
        if (id == null) {
            return super.hashCode();
        } else {
            return id.hashCode();
        }
    }

    public Set<String> fieldNames() {
        return fieldValue.keySet();
    }

    public Object getFqField(String fqFieldName) {
        return fieldValue.get(fqFieldName);
    }

    public String getVersion() {
        return version;
    }

    /**
     * Server only used
     * @param version
     */
    public void setVersion(String version) {
        this.version = version;
    }

    /**
     * Server only used
     * @param version
     */
    public void clearDirty() {
        fieldDirty.clear();
    }

    /**
     * Server only used
     * @param fieldName fqn (ex: extensionName.fieldName)
     * @param value new value
     */
    public void setFqField(String fieldName, Object value) {
        FieldType fieldType = getFieldType(fieldName);
        Object validValue = fieldType.getValidValue(value);
        fieldValue.put(fieldName, validValue);
    }

    public boolean isEmpty() {
        return fieldValue.isEmpty();
    }
    
    @Override
    public String toString() {
        boolean cr = true;
        String str = "[" + getId() + ":" + getVersion() + "] {";
        for ( String extName : getExtensionNames() ) {
            WikittyExtension ext = getExtension(extName);
            str += (cr ? "\n" : "") + "\t<" + extName + ">\n";
            cr = false;
            for ( String fieldName : ext.getFieldNames() ) {
                str += "\t\t" + fieldName + " = " + getFieldAsString(extName, fieldName) + "\n"; 
            }
        }
        str += "}";
        return str;
    }

}
