/* *##% 
 * ToPIA :: Service Migration
 * Copyright (C) 2004 - 2009 CodeLutin
 *
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Lesser General Public License as
 * published by the Free Software Foundation, either version 3 of the
 * License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Lesser Public License for more details.
 *
 * You should have received a copy of the GNU General Lesser Public
 * License along with this program.  If not, see
 * <http://www.gnu.org/licenses/lgpl-3.0.html>.
 * ##%*/
package org.nuiton.topia.migration.kernel;

import java.io.Serializable;
import java.util.Collection;
import java.util.HashMap;
import java.util.HashSet;
import java.util.Map;
import java.util.Set;
import java.util.SortedMap;

import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.nuiton.topia.migration.common.MapAdapter;
import org.nuiton.topia.migration.common.MapAdapterAdmin;
import org.nuiton.topia.migration.common.ProxyClass;
import org.nuiton.topia.migration.common.ProxyClassMapped;
import org.nuiton.topia.migration.common.SimpleProxyClass;
import org.nuiton.topia.migration.common.SimpleProxyClassMapped;
import org.nuiton.topia.migration.transformation.FinderMigration;
import org.nuiton.topia.migration.transformation.MapHelper;
import org.nuiton.topia.migration.transformation.Migration;
import org.nuiton.topia.migration.transformation.MigrationClass;
import org.nuiton.topia.migration.transformation.MigrationNull;
import org.nuiton.util.Version;

/**
 * Transformer.java
 * 
 * @author Chatellier Eric
 * @author Chevallereau Benjamin
 * @author Eon Sébastien
 * @author Trève Vincent
 * @version $Revision: 1459 $
 * 
 * Last update : $Date: 2009-05-16 09:56:47 +0200 (Sat, 16 May 2009) $
 */
public class Transformer implements MapHelper {

    private FinderMigration classFinder;
    private ConfigurationAdapter baseConf;
    private ConfigurationAdapter nextConf;
    private transient Log log;
    private SortedMap<Version, ConfigurationAdapter> confAdapters;
    private Map<ProxyClassMapped, ProxyClassMapped> linkedClass;
    private Map<ProxyClass, Collection<MapAdapterAdmin>> loadedClass;
    private Collection<ProxyClass> alreadyMigrated;

    /**
     * Transformer
     *
     * @param smConfigurationAdapterForVersion
     *            Map "ordonnees suivant les versions
     */
    public Transformer(
            SortedMap<Version, ConfigurationAdapter> smConfigurationAdapterForVersion) {

        this.confAdapters = smConfigurationAdapterForVersion;
        this.loadedClass = new HashMap<ProxyClass, Collection<MapAdapterAdmin>>();
        this.baseConf = smConfigurationAdapterForVersion.get(smConfigurationAdapterForVersion.firstKey());

        alreadyMigrated = new HashSet<ProxyClass>();
        this.linkedClass = new HashMap<ProxyClassMapped, ProxyClassMapped>();
        this.classFinder = new FinderMigration();
        log = LogFactory.getLog(this.getClass());
    }

    /**
     * transforme la base
     */
    public void execute() {
        // on l'enleve de la liste des version a obtenir
        this.confAdapters.remove(this.baseConf.getVersion());
        // tant qu'on en est pas la derniere
        while (!this.confAdapters.isEmpty()) {
            // la suivante :
            this.nextConf = confAdapters.get(confAdapters.firstKey());
            this.migrateCurrentToNext();
        }

    }

    private void migrateCurrentToNext() {
        log.info("Processing step: version " + this.baseConf.getVersion().getVersion() + " to version " + this.nextConf.getVersion().getVersion());
        // les classes de la version de destination :
        Collection<ProxyClass> classes = this.nextConf.getClasses();
        Map<ProxyClass, MigrationClass> migrations = findCurrentMigrations(classes);
        log.debug("Migrations list : " + migrations);
        this.performMigrations(migrations);
        this.baseConf = nextConf;
        alreadyMigrated.clear();
        this.nextConf = null;
        this.linkedClass.clear();
        this.loadedClass.clear();
        confAdapters.remove(confAdapters.firstKey());

    }

    private void performMigrations(Map<ProxyClass, MigrationClass> migrations) {
        for (ProxyClass pc : migrations.keySet()) {
            log.debug("Migration of '" + pc.getCanonicalName() + "' to " + this.nextConf.getVersion());

            migrateAClassRec(pc, migrations, null);
        }
    }

    private void migrateAClassRec(ProxyClass pc,
            Map<ProxyClass, MigrationClass> migrations, ProxyClass grandparent) {
        if (alreadyMigrated.contains(pc)) {
            return;
        }
        //log.debug("migration of " + pc);
        Set<ProxyClass> deps = nextConf.getDependenciesHelper().getDependencies(pc);
        for (ProxyClass dep : deps) {
            if (dep.equals(grandparent)) {
                continue;
            }
            log.debug("Migration of '" + dep.getCanonicalName() + "' dependencies");
            migrateAClassRec(dep, migrations, pc);
        }

        Collection<MapAdapterAdmin> data = null;
        MigrationClass mc = migrations.get(pc);

        data = baseConf.getData(mc.getParentClassName());

        try {
            // EC-20090714 fix class loader probleme
            //Migration migration = (Migration) ClassLoader.getSystemClassLoader().loadClass(
            //        mc.getNameMigrationClass()).newInstance();
            Migration migration = (Migration) Transformer.class.getClassLoader().loadClass(
                            mc.getNameMigrationClass()).newInstance();
            ProxyClass parent = mc.getParentClassName();

            log.debug(pc.getCanonicalName() + " (data provided by class '" + parent.getCanonicalName() + "')");

            ProxyClassMapped pcmBase = new SimpleProxyClassMapped(parent, this.baseConf.getNameIdAttribute(parent));
            ProxyClassMapped pcmNext = new SimpleProxyClassMapped(
                    pc, this.nextConf.getNameIdAttribute(pc), migration);

            linkedClass.put(pcmBase, pcmNext);
            for (MapAdapterAdmin m : data) {
                migration.migrate(m, this);
                m.setOuterClass(parent);
                //m.setOuterVersion(this.nextConf.getVersion());
            }
        } catch (ClassNotFoundException e) {
            log.error("Cannot found migration class " + mc.getNameMigrationClass());
        } catch (InstantiationException e) {
            log.error("Cannot initalize migration class " + mc.getNameMigrationClass());
        } catch (IllegalAccessException e) {
            log.error("IllegalAccessException while loading migration class " + mc.getNameMigrationClass());
        }

        alreadyMigrated.add(pc);
        this.switchMaps(data);
        log.debug("Saving '" + pc.getCanonicalName() + "' in version " + this.nextConf.getVersion());
        save(data);
        clearLoadedClasses();

    }

    private void switchMaps(Collection<MapAdapterAdmin> data) {
        for (MapAdapterAdmin m : data) {
            m.switchVersion();
        }
    }

    private void clearLoadedClasses() {
        this.loadedClass.clear();
    }

    private void save(Collection<MapAdapterAdmin> data) {
        nextConf.saveMaps(data, linkedClass);
    }

    private Map<ProxyClass, MigrationClass> findCurrentMigrations(
            Collection<ProxyClass> classes) {

        Map<ProxyClass, MigrationClass> result = new HashMap<ProxyClass, MigrationClass>();
        for (ProxyClass klass : classes) {
            MigrationClass mig = this.classFinder.getMigrationClass(klass,
                    this.baseConf.getVersion(), this.nextConf.getVersion());
            if (mig == null) {
                mig = new NullMigrationClass(MigrationNull.class.getCanonicalName(), klass, this.baseConf.getVersion(),
                        this.nextConf.getVersion());
            }
            result.put(klass, mig);
        }
        return result;
    }

    @Override
    public MapAdapter getNewMap(ProxyClass clazz, Serializable idVal) throws ObjectNotFound {
        return getMap(this.nextConf, clazz, idVal);
    }

    @Override
    public MapAdapter getNewMap(String clazz, Serializable idVal) throws ObjectNotFound {
        return getMap(this.nextConf, new SimpleProxyClass(clazz), idVal);
    }

    @Override
    public MapAdapter getOldMap(ProxyClass clazz, Serializable idVal) throws ObjectNotFound {
        return getMap(this.baseConf, clazz, idVal);
    }

    @Override
    public MapAdapter getOldMap(String clazz, Serializable idVal) throws ObjectNotFound {
        return getMap(this.baseConf, new SimpleProxyClass(clazz), idVal);
    }

    @Override
    public Collection<MapAdapter> getNewMaps(String clazz) {
        return getMaps(this.nextConf, new SimpleProxyClass(clazz));
    }

    @Override
    public Collection<MapAdapter> getNewMaps(ProxyClass clazz) {
        return getMaps(this.nextConf, clazz);
    }

    @Override
    public Collection<MapAdapter> getOldMaps(String clazz) {
        return getMaps(this.baseConf, new SimpleProxyClass(clazz));
    }

    @Override
    public Collection<MapAdapter> getOldMaps(ProxyClass clazz) {
        return getMaps(this.baseConf, clazz);
    }

    private MapAdapter getMap(ConfigurationAdapter cfg, ProxyClass pc,
            Serializable idVal) throws ObjectNotFound {
        MapAdapter result = cfg.getData(pc, idVal);
        if (result == null) {
            throw new ObjectNotFound();
        }
        return result;
    }

    private Collection<MapAdapter> getMaps(ConfigurationAdapter cfg, ProxyClass clazz) {
        Collection<MapAdapterAdmin> tmp = cfg.getData(clazz);
        Collection<MapAdapter> result = new HashSet<MapAdapter>();
        for (MapAdapterAdmin m : tmp) {
            result.add(m);
        }
        return result;
    }
}
