/*
 * *##% 
 * ToPIA :: Persistence
 * Copyright (C) 2004 - 2009 CodeLutin
 *
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Lesser General Public License as
 * published by the Free Software Foundation, either version 3 of the
 * License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Lesser Public License for more details.
 *
 * You should have received a copy of the GNU General Lesser Public
 * License along with this program.  If not, see
 * <http://www.gnu.org/licenses/lgpl-3.0.html>.
 * ##%*
 */
package org.nuiton.topia.generator;

import org.apache.commons.lang.StringUtils;
import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.nuiton.eugene.GeneratorUtil;
import org.nuiton.eugene.Template;
import org.nuiton.eugene.java.ObjectModelTransformerToJava;
import org.nuiton.eugene.models.object.*;
import org.nuiton.topia.TopiaException;
import org.nuiton.topia.framework.TopiaContextImplementor;
import org.nuiton.topia.persistence.TopiaDAO;
import org.nuiton.topia.persistence.TopiaDAOImpl;
import org.nuiton.topia.persistence.TopiaEntity;
import org.nuiton.util.StringUtil;

import java.util.*;
import java.util.regex.Matcher;
import java.util.regex.Pattern;


/*{generator option: parentheses = false}*/

/*{generator option: writeString = +}*/

/**
 * Created: 13 déc. 2009
 *
 * @author Tony Chemit <chemit@codelutin.com> Copyright Code Lutin
 * @version $Revision: 1845 $
 *          <p/>
 *          Mise a jour: $Date: 2010-03-17 14:06:52 +0100 (mer. 17 mars 2010) $ par :
 *          $Author: tchemit $
 * @since 2.3.0
 * @plexus.component role="org.nuiton.eugene.Template" role-hint="org.nuiton.topia.generator.DAOAbstractTransformer"
 */
public class DAOAbstractTransformer extends ObjectModelTransformerToJava {

    /**
     * Logger
     */
    private static final Log log = LogFactory.getLog(
            DAOAbstractTransformer.class);

    Map<ObjectModelClass, Set<ObjectModelClass>> usages;

    @Override
    public void transformFromModel(ObjectModel model) {
        
        usages = TopiaGeneratorUtil.searchDirectUsages(model);
    }

    @Override
    public void transformFromClass(ObjectModelClass clazz) {
        if (!clazz.hasStereotype(TopiaGeneratorUtil.STEREOTYPE_ENTITY)) {
            return;
        }

        String clazzName = clazz.getName();

        ObjectModelClass result = createAbstractClass(
                clazzName + "DAOAbstract<E extends " + clazzName + '>',
                clazz.getPackageName());

        // super class

        String extendClass = "";
        for (ObjectModelClass parent : clazz.getSuperclasses()) {
            extendClass = parent.getQualifiedName();
            if (parent.hasStereotype(TopiaGeneratorUtil.STEREOTYPE_ENTITY)) {
                extendClass += "DAOImpl<E>";
                // in java no multi-inheritance
                break;
            }
        }
        if (extendClass.length() == 0) {
            extendClass = TopiaDAOImpl.class.getName() + "<E>";

        }
        if (log.isDebugEnabled()) {
            log.debug("super class = "+extendClass);
        }
        setSuperClass(result, extendClass);

        addInterface(result, TopiaDAO.class.getName() + "<E>");

        // imports

        Collection<ObjectModelOperation> DAOoperations = getDAOOperations(clazz);
        if (isCollectionNeeded(DAOoperations)) {
            addImport(result, Collection.class);
        }
        if (isSetNeeded(DAOoperations)) {
            addImport(result, Set.class);
        }
        addImport(result, List.class);
        addImport(result, Arrays.class);
        addImport(result, TopiaException.class);
        addImport(result, TopiaContextImplementor.class);

        boolean enableSecurity = (
                clazz.hasTagValue(TopiaGeneratorUtil.TAG_SECURITY_CREATE) ||
                clazz.hasTagValue(TopiaGeneratorUtil.TAG_SECURITY_LOAD) ||
                clazz.hasTagValue(TopiaGeneratorUtil.TAG_SECURITY_UPDATE) ||
                clazz.hasTagValue(TopiaGeneratorUtil.TAG_SECURITY_DELETE)
        );

        if (enableSecurity) {
            addImport(result, ArrayList.class);
            addImport(result, java.security.Permission.class);
            addImport(result, "org.nuiton.topia.taas.entities.TaasAuthorizationImpl");
            addImport(result, "org.nuiton.topia.taas.jaas.TaasPermission");
            addImport(result, "org.nuiton.topia.taas.TaasUtil");
            addImport(result, TopiaDAO.class);

            //FIXME : how to do static imports ?
//import static org.nuiton.topia.taas.TaasUtil.CREATE;
//import static org.nuiton.topia.taas.TaasUtil.DELETE;
//import static org.nuiton.topia.taas.TaasUtil.LOAD;
//import static org.nuiton.topia.taas.TaasUtil.UPDATE;
        }

        ObjectModelOperation op;

        // getEntityClass

        op = addOperation(result,
                "getEntityClass",
                "Class<E>",
                ObjectModelModifier.PUBLIC);
        setOperationBody(op,""
/*{
        return (Class<E>)<%=clazzName%>.class;
    }*/
        );


        generateDAOOperations(result, DAOoperations);

        generateDelete(clazz,result);

        generateNaturalId(result, clazz);

        for (ObjectModelAttribute attr : clazz.getAttributes()) {
            if (!attr.isNavigable()) {
                continue;
            }

            if (!GeneratorUtil.isNMultiplicity(attr)) {
                generateNoNMultiplicity(clazzName, result, attr, false);
            } else {
                generateNMultiplicity(clazzName, result, attr);
            }
        }

        if (clazz instanceof ObjectModelAssociationClass) {
            ObjectModelAssociationClass assocClass =
                    (ObjectModelAssociationClass)clazz;
            for (ObjectModelAttribute attr : assocClass.getParticipantsAttributes()) {
                if (attr != null) {
                    if (!GeneratorUtil.isNMultiplicity(attr)) {
                        generateNoNMultiplicity(clazzName, result, attr, true);
                    } else {
                        generateNMultiplicity(clazzName, result, attr);
                    }
                }
            }
        }

        if(enableSecurity) {

            // getRequestPermission

            op = addOperation(result,
                    "getRequestPermission",
                    "List<Permission>",
                    ObjectModelModifier.PUBLIC);
            setDocumentation(op,"Retourne les permissions a verifier pour " +
                    "l'acces a l'entite pour le service Taas");
            addException(op, TopiaException.class);
            addParameter(op,String.class,"topiaId");
            addParameter(op,int.class,"actions");
            StringBuilder buffer = new StringBuilder();
            buffer.append(""
/*{
        List<Permission> resultPermissions = new ArrayList<Permission>();
        if ((actions & TaasUtil.CREATE) == TaasUtil.CREATE) {
}*/
                    );
            buffer.append(generateSecurity(result, clazz,
                    TopiaGeneratorUtil.TAG_SECURITY_CREATE));
            buffer.append(""
/*{
        }
        if ((actions & TaasUtil.LOAD) == TaasUtil.LOAD) {
}*/
                    );
            buffer.append(generateSecurity(result, clazz,
                    TopiaGeneratorUtil.TAG_SECURITY_LOAD));
            buffer.append(""
/*{
        }
        if ((actions & TaasUtil.UPDATE) == TaasUtil.UPDATE) {
}*/
                    );
            buffer.append(generateSecurity(result, clazz,
                    TopiaGeneratorUtil.TAG_SECURITY_UPDATE));
            buffer.append(""
/*{
        }
        if ((actions & TaasUtil.DELETE) == TaasUtil.DELETE) {
}*/
                    );
            buffer.append(generateSecurity(result, clazz,
                    TopiaGeneratorUtil.TAG_SECURITY_DELETE));
            buffer.append(""
/*{
        }
        return resultPermissions;
    }*/
                    );
            
            setOperationBody(op,buffer.toString());
            
            // THIMEL : Le code suivant doit pouvoir être déplacé dans DAODelegator ?

            // getRequestPermission


            op = addOperation(result,
                    "getRequestPermission",
                    "List<Permission>",
                    ObjectModelModifier.PROTECTED);
            addParameter(op,String.class,"topiaId");
            addParameter(op,int.class,"actions");
            addParameter(op,String.class,"query");
            addParameter(op,Class.class,"daoClass");
            addException(op,TopiaException.class);
            setDocumentation(op,"Retourne les permissions a verifier pour " +
                    "l'acces a l'entite pour le service Taas");
            setOperationBody(op,""
/*{    TopiaContextImplementor context = getContext();
    List<String> result = context.find(query, "id", topiaId);

    List<Permission> resultPermissions = new ArrayList<Permission>();
    for (String topiaIdPermission : result) {
        TopiaDAO dao = context.getDAO(daoClass);
        List<Permission> permissions = dao.getRequestPermission(topiaIdPermission, actions);
        if(permissions != null) {
            resultPermissions.addAll(permissions);
        } else {
            TaasPermission permission = new TaasPermission(topiaIdPermission, actions);
            resultPermissions.add(permission);
        }
    }
    return resultPermissions;
    }*/
            );
        }

        Set<ObjectModelClass> usagesForclass = usages.get(clazz);
        generateFindUsages(clazz,result,usagesForclass);
    }

    private void generateDelete(ObjectModelClass clazz,
                                ObjectModelClass result) {
        ObjectModelOperation op;
        op = addOperation(result, "delete", "void", ObjectModelModifier.PUBLIC);
        addException(op, TopiaException.class);
        addParameter(op, "E", "entity");
        StringBuilder body = new StringBuilder();
        String modelName = StringUtils.capitalize(model.getName());
        String providerFQN = getOutputProperties().getProperty(
                Template.PROP_DEFAULT_PACKAGE) + '.' + modelName +
                "DAOHelper.getImplementationClass";
        for (ObjectModelAttribute attr : clazz.getAttributes()) {
        	String attrType = attr.getType();
        	String reverseAttrName = attr.getReverseAttributeName();
            ObjectModelAttribute reverse = attr.getReverseAttribute();
            if (!attr.hasAssociationClass() && reverse != null && reverse.isNavigable()
                    && GeneratorUtil.isNMultiplicity(attr) && GeneratorUtil.isNMultiplicity(reverse)) {
                // On doit absolument supprimer pour les relations many-to-many
                // le this de la collection de l'autre cote

            	String attrDBName = TopiaGeneratorUtil.getDBName(attr);
            	String attrClassifierDBName = TopiaGeneratorUtil.getDBName(attr.getClassifier());
            	String attrJoinTableName = TopiaGeneratorUtil.getManyToManyTableName(attr);
            	String attrReverseDBName = TopiaGeneratorUtil.getReverseDBName(attr);
                body.append(""
/*{
        {
            List<<%=attrType%>> list = getContext().getHibernate().createSQLQuery(
                    "SELECT main.topiaid " +
                    "from <%=attrClassifierDBName%> main, <%=attrJoinTableName%> secondary " +
                    "where main.topiaid=secondary.<%=attrDBName%>" +
                    " and secondary.<%=attrReverseDBName%>='" + entity.getTopiaId() + "'")
                    .addEntity("main", <%=providerFQN%>(<%=attrType%>.class)).list();
            for (<%=attrType%> item : list) {
                item.remove<%=StringUtils.capitalize(reverseAttrName)%>(entity);
            }
        }
}*/
                );
            } else  if (!attr.hasAssociationClass() && reverse != null
                    && reverse.isNavigable()
                    && !GeneratorUtil.isNMultiplicity(reverse)) {
                // On doit mettre a null les attributs qui ont cet objet sur les
                // autres entites en one-to-*
                // TODO peut-etre qu'hibernate est capable de faire ca tout seul ?
            	// THIMEL: J'ai remplacé reverse.getName() par reverseAttrName sans certitude
                builder.addImport(result, attrType);
                String attrSimpleType = TopiaGeneratorUtil.getClassNameFromQualifiedName(attrType);

                body.append(""
                /*{
                {
                List<<%=attrSimpleType%>> list = getContext()
                            .getDAO(<%=attrSimpleType%>.class)
                            .findAllByProperties(<%=attrSimpleType%>.<%=getConstantName(reverseAttrName)%>, entity);
//                            .findAllByProperties("<%=reverseAttrName%>", entity);
                    for (<%=attrSimpleType%> item : list) {
                        item.set<%=StringUtils.capitalize(reverseAttrName)%>(null);
}*/
                );
            	if(attr.isAggregate()){
                    body.append(""
/*{
            			item.delete();
}*/
                            );
            	}
                body.append(""
/*{
                    }
                }
}*/
                );

            }
        }
        body.append(""
/*{
        super.delete(entity);
    }*/
        );

        setOperationBody(op,body.toString());
    }
    private void generateFindUsages(ObjectModelClass clazz,
                                    ObjectModelClass result,
                                    Set<ObjectModelClass> usagesForclass) {

        builder.addImport(result, ArrayList.class.getName());
        builder.addImport(result, Map.class.getName());
        builder.addImport(result, HashMap.class.getName());
        builder.addImport(result, TopiaEntity.class.getName());
        
        if (clazz instanceof ObjectModelAssociationClass || usagesForclass.isEmpty()) {
            // not for an association class
            // just let a null method
            ObjectModelOperation operation;
            operation = addOperation(result,
                    "findUsages",
                    "<U extends TopiaEntity> List<U>",
                    ObjectModelModifier.PUBLIC);

            addParameter(operation, "Class<U>", "type");
            addParameter(operation, "E", "entity");
            addException(operation, TopiaException.class);
            addAnnotation(result, operation,"Override");
            setOperationBody(operation,""
/*{
        return new ArrayList<U>();
    }*/
            );

            operation = addOperation(result,
                    "findAllUsages",
                    "Map<Class<? extends TopiaEntity>, List<? extends TopiaEntity>>",
                    ObjectModelModifier.PUBLIC);

            addParameter(operation, "E", "entity");
            addException(operation, TopiaException.class);
            addAnnotation(result, operation,"Override");
            setOperationBody(operation, ""
/*{
        return new HashMap<Class<? extends TopiaEntity>, List<? extends TopiaEntity>>();
    }*/
            );

            return;
        }
        List<ObjectModelClass> allEntities;
        Map<String, ObjectModelClass> allEntitiesByFQN;

        allEntities = TopiaGeneratorUtil.getEntityClasses(model, true);
        allEntitiesByFQN = new TreeMap<String, ObjectModelClass>();

        // prepare usages map and fill allEntitiesByFQN map
        for (ObjectModelClass klass : allEntities) {
            allEntitiesByFQN.put(klass.getQualifiedName(), klass);
        }

        ObjectModelOperation operation;
        operation = addOperation(result,
                    "findUsages",
                    "<U extends TopiaEntity> List<U>",
                    ObjectModelModifier.PUBLIC);

        addParameter(operation,"Class<U>","type");
        addParameter(operation,"E","entity");
        addException(operation,TopiaException.class);
        addAnnotation(result, operation,"Override");
        StringBuilder buffer = new StringBuilder(300);
        buffer.append(""
/*{
        List<?> result = new ArrayList();
        List tmp;
}*/
        );

        for (ObjectModelClass usageClass : usagesForclass) {
            String usageType = usageClass.getQualifiedName();
            builder.addImport(result, usageType);
            String usageSimpleType =
                    TopiaGeneratorUtil.getClassNameFromQualifiedName(usageType);

            for (ObjectModelAttribute attr : usageClass.getAttributes()) {
                if (!attr.isNavigable()) {
                    // skip this case
                    continue;
                }
                String type;
                String attrName = attr.getName();
                if (attr.hasAssociationClass()) {
                    //FIXME-TC20100224 dont known how to do this ?
                    continue;
//                    type = attr.getAssociationClass().getQualifiedName();
//                    //FIXME-TC20100224 : this is crazy ??? must find the good name
//                    // Perhaps need to make different cases?
//                    attrName = attrName + "_" + TopiaGeneratorUtil.toLowerCaseFirstLetter(attr.getAssociationClass().getName());
                } else {
                    type = attr.getType();
                }
                if (!allEntitiesByFQN.containsKey(type)) {
                    // not a entity, can skip for this attribute
                    continue;
                }
                ObjectModelClass targetEntity = allEntitiesByFQN.get(type);
                //if (!type.equals(clazz.getQualifiedName())) {
                if (!targetEntity.equals(clazz)) {
                    // not a good attribute reference
                    continue;
                }
                // found something to seek

                String methodName;
                if (!GeneratorUtil.isNMultiplicity(attr)) {
                    methodName = "findAllBy"+ StringUtils.capitalize(attrName);
                } else {
                    methodName = "findAllContains"+ StringUtils.capitalize(attrName);
                }
                String daoName = StringUtils.capitalize(usageSimpleType) + "DAO";

                builder.addImport(result, usageClass.getPackageName() + '.' + daoName);

                buffer.append(""
/*{
        if (type == <%=usageSimpleType%>.class) {
            <%=daoName%> dao = (<%=daoName%>)
                getContext().getDAO(<%=usageSimpleType%>.class);
            tmp = dao.<%=methodName%>(entity);
//            tmp = dao.findAllByProperties(<%=usageSimpleType%>.<%=getConstantName(attrName)%>, entity);
            result.addAll(tmp);
        }
}*/
                );
            }
        }

        buffer.append(""
/*{
        return (List<U>) result;
    }*/
        );
        setOperationBody(operation, buffer.toString());

        operation = addOperation(result,
                "findAllUsages",
                "Map<Class<? extends TopiaEntity>, List<? extends TopiaEntity>>",
                ObjectModelModifier.PUBLIC);

        addParameter(operation, "E", "entity");
        addException(operation, TopiaException.class);
        addAnnotation(result, operation,"Override");

        buffer = new StringBuilder(300);
        buffer.append(""
/*{
        Map<Class<? extends TopiaEntity>,List<? extends TopiaEntity>> result;
        result = new HashMap<Class<? extends TopiaEntity>, List<? extends TopiaEntity>>(<%=usagesForclass.size()%>);
        
        List<? extends TopiaEntity> list;
}*/
        );
        for (ObjectModelClass usageClass : usagesForclass) {

            String fqn = usageClass.getName();
            buffer.append(""
/*{
        list = findUsages(<%=fqn%>.class, entity);
        if (!list.isEmpty()) {
            result.put(<%=fqn%>.class, list);
        }
}*/
            );

        }
        buffer.append(""
/*{
        return result;
    }*/
        );

        setOperationBody(operation, buffer.toString());
    }

    /**
     * Generation of DAO operations signatures from class.
     * These operations are abstract and identified by <<dao>> stereotype in the model.
     * The developper must defined these methods in the DAOImpl associated to this DAOAbstract.
     * @param result clazz where to add operations
     * @param operations operations to generate
     */
    private void generateDAOOperations(ObjectModelClass result,
                                       Collection<ObjectModelOperation>
                                               operations) {
        for (ObjectModelOperation op : operations) {

            //TODO: add to transformer cloneOperation
            
            ObjectModelOperation op2;
            op2 = addOperation(result,
                    op.getName(),
                    op.getReturnType(),
                    ObjectModelModifier.ABSTRACT,
                    ObjectModelModifier.toValue(op.getVisibility()));
            setDocumentation(op2, op.getDocumentation());

            // parameters
            
            for (ObjectModelParameter param : op.getParameters()) {
                ObjectModelParameter param2 = addParameter(op2,
                        param.getType(), param.getName());
                setDocumentation(param2, param.getDocumentation());
            }

            // exceptions 
            Set<String> exceptions = op.getExceptions();
            exceptions.add(TopiaException.class.getName());
            for (String exception : exceptions) {
                addException(op2,exception);
            }
        }
    }


    private String generateSecurity(ObjectModelClass result,
                                    ObjectModelClass clazz,
                                    String securityTagName) {
        StringBuilder buffer = new StringBuilder();
        
    	if (clazz.hasTagValue(securityTagName)) {
    		String security = clazz.getTagValue(securityTagName);
            Pattern propertiesPattern = Pattern
                .compile("((?:[_a-zA-Z0-9]+\\.)+(?:_?[A-Z][_a-zA-Z0-9]*\\.)+)attribute\\.(?:([_a-z0-9][_a-zA-Z0-9]*))#(?:(create|load|update|delete))");
            String[] valuesSecurity = security.split(":");

            for (String valueSecurity : valuesSecurity) {
                Matcher matcher = propertiesPattern.matcher(valueSecurity);
                matcher.find();
                // className is fully qualified name of class
                String className = matcher.group(1);
                className = StringUtil.substring(className, 0, -1); // remove ended
                // .
                // target is class, attribute or operation
                String attributeName = matcher.group(2);
                String actions = matcher.group(3).toUpperCase();

                String query = "";
                String daoClass = "";
                if(className.equals(clazz.getQualifiedName())) {
                    query = "select " + attributeName + ".topiaId from " + clazz.getQualifiedName() + " where topiaId = :id";
                    daoClass = clazz.getAttribute(attributeName).getClassifier().getQualifiedName();
                } else {
                    query = "select at.topiaId from " + className + " at inner join at." + attributeName + " cl where cl.topiaId = :id";
                    daoClass = className;
                }
                buffer.append(""
/*{
              resultPermissions.addAll(getRequestPermission(topiaId,
                                                            <%=actions%>,
                                                            "<%=query%>",
                                                            <%=daoClass%>.class));
}*/
                );
            }
        } else {
            buffer.append(""
/*{            return null;
    }*/
            );
        }
        return buffer.toString();
    }

    protected void generateNoNMultiplicity(String clazzName,
                                           ObjectModelClass result,
                                           ObjectModelAttribute attr,
                                           boolean isAssoc)  {
    	String attrName = attr.getName();
    	String attrType = attr.getType();
        String propertyName = attrName;
        if (!isAssoc && attr.hasAssociationClass()) {
            propertyName = TopiaGeneratorUtil.toLowerCaseFirstLetter(
                    attr.getAssociationClass().getName()) + '.' + propertyName;
        }
        ObjectModelOperation op;
        op = addOperation(result,
                "findBy" + StringUtils.capitalize(attrName),
                "E",
                ObjectModelModifier.PUBLIC);
        addException(op, TopiaException.class);
        addParameter(op, attrType, "v");
        setDocumentation(op, "Retourne le premier élément trouvé ayant comme valeur pour l'attribut "+ attrName + " le paramètre.");
        setOperationBody(op,""
/*{
        E result = findByProperty(<%=clazzName + "." + getConstantName(propertyName)%>, v);
        return result;
    }*/
        );

        op = addOperation(result,
                "findAllBy" + StringUtils.capitalize(attrName),
                "List<E>",
                ObjectModelModifier.PUBLIC);
        addException(op, TopiaException.class);
        addParameter(op, attrType, "v");
        setDocumentation(op, "Retourne les éléments ayant comme valeur pour " +
                "l'attribut " + attrName + " le paramètre.");
        setOperationBody(op,""
/*{
        List<E> result = findAllByProperty(<%=clazzName + "." + getConstantName(propertyName)%>, v);
        return result;
    }*/
        );

        if (attr.hasAssociationClass()) {
            String assocClassName = attr.getAssociationClass().getName();
            String assocClassFQN = attr.getAssociationClass().getQualifiedName();
            op = addOperation(result,
                    "findBy" + StringUtils.capitalize(assocClassName),
                    "E",
                    ObjectModelModifier.PUBLIC);
            addException(op, TopiaException.class);
            addParameter(op, assocClassFQN, "value");
            setDocumentation(op, "Retourne le premier élément trouvé ayant " +
                    "comme valeur pour l'attribut " +
                    TopiaGeneratorUtil.toLowerCaseFirstLetter(assocClassName) +
                    " le paramètre.");
            setOperationBody(op, ""
/*{
        E result = findByProperty(<%=clazzName + "." + getConstantName(TopiaGeneratorUtil.toLowerCaseFirstLetter(assocClassName))%>, value);
        return result;
    }*/
            );

            op = addOperation(result,
                    "findAllBy" + StringUtils.capitalize(assocClassName),
                    "List<E>",
                    ObjectModelModifier.PUBLIC);
            addException(op, TopiaException.class);
            addParameter(op, assocClassFQN, "value");
            setDocumentation(op, "Retourne les éléments ayant comme valeur pour" +
                    " l'attribut " +
                    TopiaGeneratorUtil.toLowerCaseFirstLetter(assocClassName) +
                    " le paramètre.");
            setOperationBody(op,""
/*{
        List<E> result = findAllByProperty(<%=clazzName + "." + getConstantName(TopiaGeneratorUtil.toLowerCaseFirstLetter(assocClassName))%>, value);
        return result;
    }*/
            );
        }
    }

    protected void generateNMultiplicity(String clazzName, ObjectModelClass result, ObjectModelAttribute attr) {
    	String attrName = attr.getName();
    	String attrType = attr.getType();
        ObjectModelOperation op;

        op = addOperation(result,
                "findContains" + StringUtils.capitalize(attrName),
                "E",
                ObjectModelModifier.PUBLIC);
        addException(op, TopiaException.class);
        addParameter(op, attrType + "...", "v");
        setDocumentation(op, "Retourne le premier élément trouvé dont " +
                "l'attribut " +attrName + " contient le paramètre." );
        setOperationBody(op,""
/*{
        //FIXME TC-20100129 : really strange behaviour : contains on sub-sub properties instead of sub properties ?
        //E  result = findContainsProperties(<%=clazzName + "." + getConstantName(TopiaGeneratorUtil.toLowerCaseFirstLetter(attrName))%>, Arrays.asList(v));
        E result = findContainsProperties("<%=TopiaGeneratorUtil.toLowerCaseFirstLetter(attrName)%>", Arrays.asList(v));
        return result;
    }*/
        );

        op = addOperation(result,
                "findAllContains" + StringUtils.capitalize(attrName),
                "List<E>",
                ObjectModelModifier.PUBLIC);
        addException(op, TopiaException.class);
        addParameter(op, attrType + "...", "v");
        setDocumentation(op, "Retourne les éléments trouvé dont l'attribut " +
                attrName + " contient le paramètre.");
        setOperationBody(op, ""
/*{
        //FIXME TC-20100129 : really strange behaviour : contains on sub-sub properties instead of sub properties ?
        //List<E> results = findAllContainsProperties(<%=clazzName + "." + getConstantName(TopiaGeneratorUtil.toLowerCaseFirstLetter(attrName))%>, Arrays.asList(v));
        List<E> results = findAllContainsProperties("<%=TopiaGeneratorUtil.toLowerCaseFirstLetter(attrName)%>", Arrays.asList(v));
        return results;
    }*/
        );
    }

    private boolean isCollectionNeeded(
            Collection<ObjectModelOperation> operations) {
        return isImportNeeded(operations, "Collection");
    }

    private boolean isSetNeeded(Collection<ObjectModelOperation> operations) {
        return isImportNeeded(operations, "Set");
    }

    private boolean isImportNeeded(Collection<ObjectModelOperation> operations,
                                   String importName) {
        for (ObjectModelOperation op : operations) {
            if (op.getReturnType().contains(importName)) {
                return true;
            }
            for (ObjectModelParameter param : op.getParameters()) {
                if (param.getType().contains(importName)) {
                    return true;
                }
            }
        }
        return false;
    }

    public static Collection<ObjectModelOperation> getDAOOperations(
            ObjectModelClass clazz) {
        Collection<ObjectModelOperation> results =
                new ArrayList<ObjectModelOperation>();
        for (ObjectModelOperation op : clazz.getOperations()) {
            if (op.hasStereotype(TopiaGeneratorUtil.STEREOTYPE_DAO)) {
                results.add(op);
            }
        }
        return results;
    }

    private void generateNaturalId(ObjectModelClass result,
            ObjectModelClass clazz) {
        List<ObjectModelAttribute> props =
                TopiaGeneratorUtil.getNaturalIdAttributes(clazz);

        if (!props.isEmpty()) {

            if (log.isDebugEnabled()) {
                log.debug("generateNaturalId for " + props);
            }
            ObjectModelOperation findByNaturalId = addOperation(result,
                    "findByNaturalId", "E", ObjectModelModifier.PUBLIC);
            addException(findByNaturalId, TopiaException.class);

            ObjectModelOperation existNaturalId = addOperation(result,
                    "existNaturalId", "boolean", ObjectModelModifier.PUBLIC);
            addException(existNaturalId, TopiaException.class);

            ObjectModelOperation create = addOperation(result,
                    "create", "E", ObjectModelModifier.PUBLIC);
            addException(create, TopiaException.class);

            // used for calling findByProperties in findByNaturalId
            String searchProperties = "";
            // used for calling findByNaturalId in existNaturalId
            String params = "";
            String clazzName = clazz.getName();
            for (ObjectModelAttribute attr : props) {
                String propName = attr.getName();
                // add property as param in both methods
                addParameter(findByNaturalId, attr.getType(), propName);
                addParameter(existNaturalId, attr.getType(), propName);
                addParameter(create, attr.getType(), propName);

                searchProperties += 
                        ", " + clazzName + '.' + getConstantName(propName) +
                        ", " + propName;
                params += ", " + propName;
            }
            searchProperties = searchProperties.substring(2);
            params = params.substring(2);

            setOperationBody(findByNaturalId, ""
/*{
        return findByProperties(<%=searchProperties%>);
    }*/
            );
            
            setOperationBody(existNaturalId, ""
/*{
        return findByNaturalId(<%=params%>) != null;
    }*/
            );

            setOperationBody(create, ""
/*{
        return create(<%=searchProperties%>);
    }*/
            );
        }


    }
}
