/*
 * #%L
 * Nuiton Utils
 * 
 * $Id: ReflectUtil.java 1948 2010-11-17 21:31:00Z sletellier $
 * $HeadURL: http://svn.nuiton.org/svn/nuiton-utils/tags/nuiton-utils-1.5.1/src/main/java/org/nuiton/util/ReflectUtil.java $
 * %%
 * Copyright (C) 2004 - 2010 CodeLutin
 * %%
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Lesser General Public License as 
 * published by the Free Software Foundation, either version 3 of the 
 * License, or (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Lesser Public License for more details.
 * 
 * You should have received a copy of the GNU General Lesser Public 
 * License along with this program.  If not, see
 * <http://www.gnu.org/licenses/lgpl-3.0.html>.
 * #L%
 */

package org.nuiton.util;

import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;

import java.lang.reflect.Field;
import java.lang.reflect.Method;
import java.lang.reflect.Modifier;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.HashSet;
import java.util.List;
import java.util.Set;

import static org.nuiton.i18n.I18n._;

/**
 * Des méthodes utiles d'introspection
 *
 * @author tchemit <chemit@codelutin.com>
 */
public class ReflectUtil {


    /** Logger */
    static private Log log = LogFactory.getLog(ReflectUtil.class);

    /**
     * Pour déterminer si un champ d'une classe est une constante
     * (modifiers sont static, final et public)
     *
     * @param field le champs à tester
     * @return <code>true</code> si les modifiers sont final, static et public
     */
    public static boolean isConstantField(Field field) {
        int modifiers = field.getModifiers();
        return Modifier.isPublic(modifiers) &&
               Modifier.isStatic(modifiers) &&
               Modifier.isFinal(modifiers);
    }

    /**
     * Recherche dans une classe donnée <code>klazz</code>, les constantes d'un
     * certain type <code>searchingClass</code> et les retourne.
     * <p/>
     * L'algorithme parcourt aussi les superclasses.
     *
     * @param <T>            enumeration's type
     * @param klass          la classe contenant les constantes
     * @param searchingClass le type des champs constants à récupérer
     * @return la liste des champs du type requis dans
     * @throws RuntimeException si problème lors de la récupération
     */
    @SuppressWarnings({"unchecked"})
    public static <T> List<T> getConstants(Class<?> klass,
                                           Class<T> searchingClass) {
        List<T> result = new ArrayList<T>();
        for (Field field : klass.getDeclaredFields()) {
            if (!field.isAccessible()) {
                field.setAccessible(true);
            }
            if (searchingClass.isAssignableFrom(field.getType()) &&
                isConstantField(field)) {
                try {
                    result.add((T) field.get(null));
                } catch (IllegalAccessException e) {
                    throw new RuntimeException(e);
                }
            }
        }
        Class<?> superClass = klass.getSuperclass();
        if (superClass != null) {
            result.addAll(getConstants(superClass, searchingClass));
        }
        return result;
    }

    /**
     * @param <T>       enumeration's type
     * @param klass     the required class
     * @param fieldName the required constant name
     * @return the constant value
     */
    @SuppressWarnings({"unchecked"})
    public static <T> T getConstant(Class<?> klass, String fieldName) {
        try {
            T result = null;
            Field f = klass.getDeclaredField(fieldName);
            if (isConstantField(f)) {
                f.setAccessible(true);
                result = (T) f.get(null);
            }
            return result;
        } catch (NoSuchFieldException e) {
            throw new RuntimeException(e);
        } catch (IllegalAccessException e) {
            throw new RuntimeException(e);
        }
    }

    /**
     * Convertit une classe non typée, en une classe d'enum
     *
     * @param <T>  enumeration's type
     * @param type la classe a typer
     * @return la classe typee
     * @throws IllegalArgumentException si le type est null ou non une extension
     *                                  de la classe Enum.
     */
    @SuppressWarnings({"unchecked"})
    public static <T extends Enum<T>> Class<T> getEnumClass(Class<?> type) throws IllegalArgumentException {
        if (type == null) {
            throw new IllegalArgumentException(_("nuitonutil.error.null.parameter", "type"));
        }
        if (!type.isEnum()) {
            throw new IllegalArgumentException(_("nuitonutil.error.not.an.enum", type));
        }
        return (Class<T>) type;
    }

    /**
     * Cherche une methode selon son nom et ses paramètres d'invocation.
     *
     * @param klass      la classe dans laquelle rechercher la méthode
     * @param methodName le nom de la méthode recherchée
     * @param strict     un drapeau pour déclancher une exception si la méthode
     *                   n'est pas trouvée
     * @param arguments  les arguments d'invocation de la méthode
     * @return la méthode trouvée
     * @throws IllegalArgumentException si la méthode n'est pas trouvée et que
     *                                  le drapeau {@code strict} est à {@code true}
     * @since 1.3.1
     */
    public static Method getDeclaredMethod(Class<?> klass,
                                           String methodName,
                                           boolean strict,
                                           Object... arguments) throws IllegalArgumentException {
        HashSet<Class<?>> classes = new HashSet<Class<?>>();
        Method method;
        try {
            method = getDeclaredMethod(klass, methodName, classes, arguments);
        } finally {
            if (log.isDebugEnabled()) {
                log.debug("Inspected classes : " + classes);
            }
            classes.clear();
        }
        if (method == null && strict) {
            throw new IllegalArgumentException(
                    "could not find method " + methodName + " on type " +
                    klass.getName());
        }

        return method;
    }

    /**
     * Obtain the boxed type of any incoming type.
     * <p/>
     * If incoming type is not a primitive type, then just returns himself.
     *
     * @param type the type to box
     * @return the boxed type
     * @see Class#isPrimitive()
     * @since 1.3.1
     */
    public static Class<?> boxType(Class<?> type) {
        if (!type.isPrimitive()) {
            return type;
        }
        if (boolean.class.equals(type)) {
            return Boolean.class;
        }
        if (char.class.equals(type)) {
            return Character.class;
        }
        if (byte.class.equals(type)) {
            return Byte.class;
        }
        if (short.class.equals(type)) {
            return Short.class;
        }
        if (int.class.equals(type)) {
            return Integer.class;
        }
        if (long.class.equals(type)) {
            return Long.class;
        }
        if (float.class.equals(type)) {
            return Float.class;
        }
        if (double.class.equals(type)) {
            return Double.class;
        }
        if (void.class.equals(type)) {
            return Void.class;
        }
        // should never come here...
        return type;

    }

    /**
     * Obtain the unboxed type of any incoming type.
     * <p/>
     * If incoming type is a primitive type, then just returns himself.
     *
     * @param type the type to unbox
     * @return the unboxed type
     * @see Class#isPrimitive()
     * @since 1.3.1
     */
    public static Class<?> unboxType(Class<?> type) {
        if (type.isPrimitive()) {
            return type;
        }
        if (Boolean.class.equals(type)) {
            return boolean.class;
        }
        if (Character.class.equals(type)) {
            return char.class;
        }
        if (Byte.class.equals(type)) {
            return byte.class;
        }
        if (Short.class.equals(type)) {
            return short.class;
        }
        if (Integer.class.equals(type)) {
            return int.class;
        }
        if (Long.class.equals(type)) {
            return long.class;
        }
        if (Float.class.equals(type)) {
            return float.class;
        }
        if (Double.class.equals(type)) {
            return double.class;
        }
        if (Void.class.equals(type)) {
            return void.class;
        }

        // not a primitive type
        return type;

    }

    protected static Method getDeclaredMethod(Class<?> klass,
                                              String methodName,
                                              Set<Class<?>> visitedClasses,
                                              Object... arguments) {
        if (visitedClasses.contains(klass)) {

            // this means class was already unsucessfull visited
            return null;
        }
        visitedClasses.add(klass);
        Method method = null;
        for (Method m : klass.getDeclaredMethods()) {
            if (!methodName.equals(m.getName())) {
                continue;
            }

            // same method name

            Class<?>[] types = m.getParameterTypes();
            if (arguments.length != types.length) {
                continue;
            }

            // same number arguments

            Class<?>[] prototype = m.getParameterTypes();
            if (log.isDebugEnabled()) {
                log.debug("Found a method with same parameters size : " +
                          m.getName() + " : " + Arrays.toString(prototype));
            }
            int index = 0;
            boolean parametersMatches = true;
            for (Object argument : arguments) {
                Class<?> type = prototype[index++];
                if (argument == null) {

                    // can not say anything, let says it is ok...
                    continue;
                }
                Class<?> runtimeType = argument.getClass();
                if (log.isDebugEnabled()) {
                    log.debug("Test parameter [" + (index - 1) + "] : " +
                              type + " vs  " + runtimeType);
                }

                type = boxType(type);
                runtimeType = boxType(runtimeType);

                if (!type.equals(runtimeType) &&
                    !type.isAssignableFrom(runtimeType)) {

                    // not same type
                    parametersMatches = false;
                    if (log.isDebugEnabled()) {
                        log.debug("Types are not matching.");
                    }
                    break;
                }
            }
            if (parametersMatches) {

                // same parameters types, this is a match
                method = m;
            }
            break;
        }
        if (method == null) {

            // try on super class
            if (klass.getSuperclass() != null) {
                method = getDeclaredMethod(klass.getSuperclass(),
                                           methodName,
                                           visitedClasses,
                                           arguments
                );
            }
        }

        if (method == null) {

            // try on interfaces
            Class<?>[] interfaces = klass.getInterfaces();
            for (Class<?> anInterface : interfaces) {
                method = getDeclaredMethod(anInterface,
                                           methodName,
                                           visitedClasses,
                                           arguments
                );
                if (method != null) {
                    break;
                }
            }
        }
        return method;

    }
}
