/*
 * #%L
 * Nuiton Utils
 * 
 * $Id: Resource.java 1860 2010-05-11 14:42:58Z echatellier $
 * $HeadURL: http://svn.nuiton.org/svn/nuiton-utils/tags/nuiton-utils-1.3.1/src/main/java/org/nuiton/util/Resource.java $
 * %%
 * Copyright (C) 2004 - 2010 CodeLutin
 * %%
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Lesser General Public License as 
 * published by the Free Software Foundation, either version 3 of the 
 * License, or (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Lesser Public License for more details.
 * 
 * You should have received a copy of the GNU General Lesser Public 
 * License along with this program.  If not, see
 * <http://www.gnu.org/licenses/lgpl-3.0.html>.
 * #L%
 */

/**
 * Resource.java
 *
 * Created: Sun Apr 14 2002
 *
 * @author POUSSIN Benjamin <bpoussin@free.fr>
 * Copyright Code Lutin
 * @version $Revision: 1860 $
 *
 * Mise a jour: $Date: 2010-05-11 16:42:58 +0200 (mar., 11 mai 2010) $
 * par : */

package org.nuiton.util;

import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import static org.nuiton.i18n.I18n._;

import javax.swing.ImageIcon;
import java.io.File;
import java.io.FileInputStream;
import java.io.IOException;
import java.io.InputStream;
import java.lang.reflect.Method;
import java.net.MalformedURLException;
import java.net.URISyntaxException;
import java.net.URL;
import java.net.URLClassLoader;
import java.util.ArrayList;
import java.util.List;
import java.util.Properties;
import java.util.jar.Attributes;
import java.util.jar.JarFile;
import java.util.jar.Manifest;
import java.util.zip.ZipEntry;
import java.util.zip.ZipFile;
import java.util.zip.ZipInputStream;

/**
 * Cette class permet de rechercher un fichier en indiquant son nom avec son
 * chemin. Cette librairie ira ensuite chercher ce fichier sur le système de
 * fichier, et s'il n'est pas trouvé dans le classpath. Le fichier peut donc
 * être dans un fichier .jar ou .zip.
 */
public class Resource { // Resource

    /** to use log facility, just put in your code: log.info(\"...\"); */
    private static final Log log = LogFactory.getLog(Resource.class);

    protected Resource() {

    }

    /**
     * Permet d'ajouter dans le classloader par defaut une nouvelle URL dans
     * lequel il faut rechercher les fichiers.
     *
     * @param url l'url a ajouter
     */
    static public void addDefaultClassLoader(URL url) {
        ClassLoader classLoader = ClassLoader.getSystemClassLoader();
        addClassLoader(classLoader, url);
    }

    /**
     * Permet d'ajouter dans un classloader une nouvelle URL dans
     * lequel il faut rechercher les fichiers.
     *
     * @param classLoader le classloader a modifier
     * @param url         l'url a ajouter
     */
    static public void addClassLoader(ClassLoader classLoader, URL url) {
        try {
            Method method = URLClassLoader.class.getDeclaredMethod("addURL",
                    new Class[]{URL.class});
            method.setAccessible(true);
            method.invoke(classLoader, url);
        } catch (Exception eee) {
            throw new RuntimeException(_("nuitonutil.error.add.url.in.classloader", classLoader, eee));
            //throw new RuntimeException("Can't add url in classloader " + classLoader,eee);
        }
    }

    /**
     * Recherche la ressource nom.
     *
     * @param name nom de la ressource
     * @return l'url de la ressource
     * @throws ResourceNotFoundException si la resource n'a pas ete trouvee
     */
    static public URL getURL(String name) {
        URL url = getURLOrNull(name);
        if (url != null) {
            return url;
        }

        throw new ResourceNotFoundException(_("nuitonutil.error.resource.not.found", name));
    }

    /**
     * Recherche la ressource nom.
     *
     * @param name le nom de la ressource
     * @return l'url de la ressource ou null
     */
    static public URL getURLOrNull(String name) {
        // on recherche d'abord sur le filesystem
        File file = new File(name);
        if (file.exists()) {
            try {
                return file.toURI().toURL();
            } catch (MalformedURLException eee) {
                log.warn(_("nuitonutil.error.convert.file.to.url", file, eee.getMessage()));
            }
        }

        // on ne l'a pas trouve on recherche dans le classpath

        // on supprime le / devant le nom de la ressource, sinon elle
        // n'est pas trouve (pas de recherche dans les differents
        // element du classpath.
        if (name.length() > 1 && name.startsWith("/")) {
            name = name.substring(1);
        }
        URL url = ClassLoader.getSystemClassLoader().getResource(name);
        if (url != null) {
            return url;
        }

        ClassLoader cl = Resource.class.getClassLoader();
        url = cl.getResource(name);
        return url;
    }

    /**
     * Retourne l'icone demandee.
     *
     * @param name le nom de l'icone
     * @return Retourne l'icon demande ou null s'il n'est pas trouvé
     */
    static public ImageIcon getIcon(String name) {
        try {
            return new ImageIcon(getURL(name));
        } catch (Exception eee) {
            log.warn("Can't find icon: " + name, eee);
            return null;
        }
    }

    /**
     * Recherche et retourne l'objet properties de configuration à utiliser.
     * <p/>
     * Les différents fichiers trouvé sont chainé. L'ordre de recherche est le
     * classpath, le fichier dans /etc et enfin le chemin sur le filesystem.
     *
     * @param filename le nom du fichier à rechercher
     * @return l'objet Properties de configuration
     * @throws IOException si une erreur est survenue
     * @deprecated since 1.1.2, use {@link org.nuiton.util.ApplicationConfig} class instead
     * to {@link org.nuiton.util.ApplicationConfig#setConfigFileName(java.lang.String) }
     * and search properties file from more folders with
     * {@link org.nuiton.util.ApplicationConfig#parse(java.lang.String[]) } method
     */
    @Deprecated
    static public Properties getConfigProperties(String filename)
            throws IOException {
        Properties result;
        result = getConfigProperties(filename, null);
        return result;
    }

    /**
     * Recherche et retourne l'objet properties de configuration à utiliser.
     * <p/>
     * Les différents fichiers trouvé sont chainé. L'ordre de recherche est le
     * classpath, le fichier dans /etc et enfin le chemin sur le filesystem.
     *
     * @param filename le nom du fichier à rechercher
     * @param parent   les proprietes parent a surcharger
     * @return l'objet Properties de configuration
     * @throws IOException si une erreur est survenue
     * @deprecated since 1.1.2, use {@link org.nuiton.util.ApplicationConfig} class instead
     * to {@link org.nuiton.util.ApplicationConfig#setConfigFileName(java.lang.String) }
     * and search properties file from more folders with
     * {@link org.nuiton.util.ApplicationConfig#parse(java.lang.String[]) } method
     */
    @Deprecated
    static public Properties getConfigProperties(String filename,
                                                 Properties parent) throws IOException {
        Properties result;
        if (parent != null) {
            result = new Properties(parent);
        } else {
            result = new Properties();
        }

        URL inClasspath = ClassLoader.getSystemClassLoader().getResource(
                filename);
        if (inClasspath == null) {
            inClasspath = Resource.class.getResource(filename);
        }
        if (inClasspath == null) {
            inClasspath = getURLOrNull(filename);
        }
        if (inClasspath != null) {
            log.info("Chargement du fichier de config: " + inClasspath);
            result.load(inClasspath.openStream());
            result = new Properties(result);
        }

        File etcConfig = new File("/etc/" + filename);
        if (etcConfig.exists()) {
            log.info("Chargement du fichier de config: " + etcConfig);
            result.load(etcConfig.toURI().toURL().openStream());
            result = new Properties(result);
        }

        File config = new File(filename);
        if (config.exists()) {
            log.info("Chargement du fichier de config: " + config);
            result.load(config.toURI().toURL().openStream());
            result = new Properties(result);
        }

        return result;
    }

    /**
     * Retourner la liste des fichiers du classLoader. Ces fichiers doivent
     * correspondre au pattern donne.
     *
     * @param pattern le nom du fichier a extraire du fichier compressé ou
     *                du repertoire doit correspondre au pattern (repertoire + nom
     *                compris).
     * @return la liste des urls correspondant au pattern
     */
    static public List<URL> getURLs(String pattern) {
        return getURLs(pattern, (URLClassLoader) null);
    }

    /**
     * Recupere la liste des urls d'un {@link java.net.URLClassLoader}.
     * <p/>
     * Note : Un cas particulier est positionné pour JBoss qui utilise la method getAllURLs.
     *
     * @param classLoader le class loader a scanner
     * @return les urls du classloade.
     * @deprecated should use now {@link org.nuiton.util.ClassLoaderUtil#getURLs(java.net.URLClassLoader)}
     */
    static public URL[] getURLs(URLClassLoader classLoader) {
        return ClassLoaderUtil.getURLs(classLoader);
    }

    /**
     * Retourner la liste des fichiers du classLoader. Ces fichiers doivent
     * correspondre au pattern donne.
     *
     * @param classLoader le classLoader
     * @param pattern     le nom du fichier a extraire du fichier compressé ou
     *                    du repertoire doit correspondre au pattern (repertoire + nom
     *                    compris).
     * @return la liste des urls correspondant au pattern
     */
    static public List<URL> getURLs(String pattern, URLClassLoader classLoader) {
        if (classLoader == null) {
            classLoader = (URLClassLoader) ClassLoader.getSystemClassLoader();
        }
        URL[] arrayURL = ClassLoaderUtil.getURLs(classLoader);
        return getURLs(pattern, arrayURL);
    }

    /**
     * Retourner la liste des fichiers du classLoader. Ces fichiers doivent
     * correspondre au pattern donne.
     *
     * @param arrayURL les urls ou chercher
     * @param pattern  le nom du fichier a extraire du fichier compressé ou
     *                 durepertoire doit correspondre au pattern (repertoire + nom
     *                 compris).
     * @return la liste des urls correspondant au pattern
     */
    static public List<URL> getURLs(String pattern, URL... arrayURL) {
        long t0 = System.nanoTime();

        HashList<URL> urlList = new HashList<URL>();

        if (arrayURL.length == 1) {
            URL jarURL = arrayURL[0];
            if (isJar(jarURL.toString())) {
                // jar invocation
                try {
                    arrayURL = getClassPathURLsFromJarManifest(jarURL);
                } catch (Exception e) {
                    log.warn(e);
                    arrayURL = new URL[]{jarURL};
                }
            }
        }
        if (log.isDebugEnabled()) {
            for (URL url : arrayURL) {
                log.debug("found url " + url);
            }
        }

        for (URL urlFile : arrayURL) {
            // EC-20100510 this cause wrong accent encoding
            //String fileName = urlFile.getFile();
            String fileName = null;
            try {
                fileName = urlFile.toURI().getPath();
            }
            catch (Exception e) {
                if (log.isWarnEnabled()) {
                    log.warn(e);
                }
                // warning, this can cause wrong encoding !!!
                fileName = urlFile.getFile();
            }

            // TODO deal with encoding in windows, this is very durty, but it
            // works...
            File file = new File(fileName.replaceAll("%20", " "));
            if (!file.exists()) {
                // this case should not appear
                if (log.isDebugEnabled()) {
                    log.debug("Can't find file " + file + " (" + fileName + ")");
                }
                continue;
            }
            if (isJar(fileName)) {
                // cas ou le ichier du classLoader est un fichier jar
                if (log.isDebugEnabled()) {
                    log.debug("jar to search " + file);
                }
                urlList.addAll(Resource.getURLsFromJar(file, pattern));
                continue;
            }
            if (file.isDirectory()) {
                // cas ou le ichier du classLoader est un repertoire
                if (log.isDebugEnabled()) {
                    log.debug("directory to search " + file);
                }
                // on traite le cas ou il peut y avoir des repertoire dans ce
                // repertoire
                urlList.addAll(Resource.getURLsFromDirectory(file, pattern));
                continue;
            }

            if (isZip(fileName)) {
                // cas ou le ichier du classLoader est un fichier zip
                if (log.isDebugEnabled()) {
                    log.debug("zip to search " + file);
                }
                urlList.addAll(Resource.getURLsFromZip(file, pattern));
            }

        }
        if (log.isInfoEnabled()) {
            log.info("search URLs pattern: " + pattern + " in "
                    + arrayURL.length + " urls in "
                    + StringUtil.convertTime(System.nanoTime() - t0));
        }
        return urlList;
    }

    static public URL[] getClassPathURLsFromJarManifest(URL jarURL)
            throws IOException, URISyntaxException {
        JarFile jar = null;
        URL[] result;
        try {
            String jarPath = jarURL.toURI().getPath();
            File jarFile = new File(jarPath);
            if (log.isDebugEnabled()) {
                log.debug("class-path jar to scan " + jarPath);
            }
            jar = new JarFile(jarFile);
            File container = jarFile.getParentFile();
            Manifest mf = jar.getManifest();
            String classPath = null;
            if (mf != null && mf.getMainAttributes() != null) {
                classPath = mf.getMainAttributes().getValue(Attributes.Name.CLASS_PATH);
            }
            String[] paths;
            if (classPath != null) {
                paths = classPath.split(" ");
            } else {
                paths = new String[0];
            }
            result = new URL[paths.length + 1];
            result[0] = jarURL;
            File path;
            for (int i = 0; i < paths.length; i++) {
                String s = paths[i];
                // test de l'existence d'un protocole dans le path (genre file:...)
                if (s.indexOf(':') != -1) {
                    result[i + 1] = new URL(s);
                    continue;
                }

                if (s.startsWith(".") || !s.startsWith("/")) {
                    // relative url
                    path = new File(container, s);
                } else {
                    path = new File(s);
                }
                if (log.isDebugEnabled()) {
                    log.debug(path);
                }
                result[i + 1] = path.toURI().toURL();
            }
            jar.close();
        } finally {
            if (jar != null) {
                jar.close();
            }
        }
        return result;
    }

    static public List<URL> getURLsFromZip(File zipFile, String pattern) {
        try {
            if (log.isTraceEnabled()) {
                log.trace("search '" + pattern + "' in " + zipFile);
            }

            ArrayList<URL> result = new ArrayList<URL>();
            InputStream in = new FileInputStream(zipFile);
            ZipInputStream zis = new ZipInputStream(in);
            while (zis.available() != 0) {
                ZipEntry entry = zis.getNextEntry();

                if (entry == null) {
                    break;
                }

                String name = entry.getName();
                if (log.isTraceEnabled()) {
                    log.trace("zipFile: " + zipFile + " name: " + name);
                }
                if (pattern == null || name.matches(pattern)) {
                    // on recupere le fichier correspondant au pattern dans le
                    // classloader
                    URL url = Resource.getURL(name);
                    // on ajoute le fichier correspondant au pattern dans la
                    // liste
                    if (log.isTraceEnabled()) {
                        log.trace("zipFile: " + zipFile + " url: " + url);
                    }
                    result.add(url);
                }
            }
            if (log.isTraceEnabled()) {
                log.trace("found with pattern '" + pattern + "' : " + result);
            }
            return result;
        } catch (IOException eee) {
            throw new ResourceException(_("nuitonutil.error.get.url.from.zip", zipFile.getAbsolutePath(), eee.getMessage()));
        }
    }

    static public List<URL> getURLsFromJar(File jarfile, String pattern) {
        try {
            if (log.isTraceEnabled()) {
                log.trace("search '" + pattern + "' in " + jarfile);
            }

            ArrayList<URL> result = new ArrayList<URL>();
            InputStream in = new FileInputStream(jarfile);
            ZipInputStream zis = new ZipInputStream(in);
            while (zis.available() != 0) {
                ZipEntry entry = zis.getNextEntry();

                if (entry == null) {
                    break;
                }

                String name = entry.getName();
                if (log.isTraceEnabled()) {
                    log.trace("jarfile: " + jarfile + " name: " + name);
                }
                if (pattern == null || name.matches(pattern)) {
                    // on recupere le fichier correspondant au pattern dans le
                    // classloader
                    URL url = Resource.getURL(name);
                    // on ajoute le fichier correspondant au pattern dans la
                    // liste
                    if (log.isTraceEnabled()) {
                        log.trace("jarfile: " + jarfile + " url: " + url);
                    }
                    result.add(url);
                }
            }
            if (log.isTraceEnabled()) {
                log.trace("found with pattern '" + pattern + "' : " + result);
            }
            return result;
        } catch (IOException eee) {
            throw new ResourceException(_("nuitonutil.error.get.url.from.zip", jarfile.getAbsolutePath(), eee.getMessage()));
        }
    }

    /**
     * Retourne la liste des fichiers correspondant au pattern donne, aucun
     * ordre ne doit être supposé sur les fichiers.
     *
     * @param repository repertoire dans lequel on recherche les fichiers
     * @param pattern    le nom du fichier a extraire du fichier du repertoire doit
     *                   correspondre au pattern (repertoire + nom compris). si le
     *                   pattern est null, tous les fichiers trouvé sont retourné.
     * @return la liste des urls correspondant au pattern
     */
    static public List<URL> getURLsFromDirectory(File repository, String pattern) {
        try {
            if (log.isTraceEnabled()) {
                log.trace("search '" + pattern + "' in " + repository);
            }

            HashList<URL> urlList = new HashList<URL>();
            File[] filesList = repository.listFiles();

            if (filesList != null) {

                for (File file : filesList) {

                    String name = file.getAbsolutePath();

                    if (log.isTraceEnabled()) {
                        log.trace("directory: " + repository + " name: "
                                        + name);
                    }

                    // cas de recursivite : repertoire dans un repertoire
                    if (file.exists() && file.isDirectory()) {
                        urlList.addAll(Resource.getURLsFromDirectory(file,
                                pattern));
                        // si le fichier du repertoire n'est pas un repertoire
                        // on verifie s'il correspond au pattern
                    } else if (pattern == null || name.matches(pattern)) {
                        URL url = file.toURI().toURL();
                        if (log.isTraceEnabled()) {
                            log.trace("directory: " + repository + " url: "
                                    + url);
                        }
                        urlList.add(url);
                    }
                }
            }
            if (log.isTraceEnabled()) {
                log.trace("found with pattern '" + pattern + "' : " + urlList);
            }
            return urlList;
        } catch (MalformedURLException eee) {
            throw new ResourceException(_("nuitonutil.error.convert.file.to.url", repository + " (pattern " + pattern + ") ", eee.getMessage()));
            //throw new ResourceException("Le fichier n'a pu être converti en URL", eee);
        }
    }

    /**
     * Verifie si le fichier est un fichier jar.
     *
     * @param name nom du fichier a tester
     * @return vrai si le fichier se termine par .jar faux sinon
     */
    static public boolean isJar(String name) {
        if (name != null && name.length() > 4) {
            String ext = name.substring(name.length() - 4, name.length());
            return ".jar".equalsIgnoreCase(ext);
        }
        return false;
    }

    /**
     * Verifie si le fichier est un fichier zip
     *
     * @param name nom du fichier a tester
     * @return vrai si le fichier se termine par .zip faux sinon
     */
    static public boolean isZip(String name) {
        if (name != null && name.length() > 4) {
            String ext = name.substring(name.length() - 4, name.length());
            return ".zip".equalsIgnoreCase(ext);
        }
        return false;
    }

    /**
     * Verifie si la classe est de type primitif.
     *
     * @param clazz nom de la classe a tester
     * @return vrai si le classe est de type primitif faux sinon
     */
    static public boolean isPrimitive(Class clazz) {
        return clazz.isPrimitive() || clazz == Boolean.class
                || clazz == Byte.class || clazz == Character.class
                || clazz == Short.class || clazz == Integer.class
                || clazz == Long.class || clazz == Float.class
                || clazz == Double.class;
    }

    /**
     * Retourne la classe du type primitf associé avec la classe de de l'objet
     * passé en parametre. Par exemple si la classe passée en paramètre est
     * Integer alors le resultat sera la classe de int.
     *
     * @param clazz la classe dont on souhaite le type primitf
     * @return le type primitif associé a la classe en paramètre, ou null s'il
     *         n'y a pas de type primitif associé.
     */
    static public Class getPrimitiveClass(Class clazz) {
        if (clazz == Boolean.class) {
            return Boolean.TYPE;
        }
        if (clazz == Byte.class) {
            return Byte.TYPE;
        }
        if (clazz == Character.class) {
            return Character.TYPE;
        }
        if (clazz == Short.class) {
            return Short.TYPE;
        }
        if (clazz == Integer.class) {
            return Integer.TYPE;
        }
        if (clazz == Long.class) {
            return Long.TYPE;
        }
        if (clazz == Float.class) {
            return Float.TYPE;
        }
        if (clazz == Double.class) {
            return Double.TYPE;
        }
        if (clazz == Void.class) {
            return Void.TYPE;
        }
        return null;
    }


    /**
     * Test if an url contains the given directory with no recurse seeking.
     *
     * @param url       the url to seek
     * @param directory the directory to find
     * @return <code>true</code> if directory was found, <code>false</code> otherwise.
     * @throws java.io.IOException if any io pb
     */
    public static boolean containsDirectDirectory(URL url, String directory) throws IOException {
        String fileName = url.getFile();
        // TODO deal with encoding in windows, this is very durty, but it works...
        File file = new File(fileName.replaceAll("%20", " "));
        if (!file.exists()) {
            return false;
        }
        if (isJar(fileName) || isZip(fileName)) {
            // cas ou le fichier du classLoader est un fichier jar ou zip
            if (log.isTraceEnabled()) {
                log.trace("zip to search " + file);
            }
            return new ZipFile(file).getEntry(directory + '/') != null;
        }
        if (file.isDirectory()) {
            // cas ou le ichier du classLoader est un repertoire
            if (log.isTraceEnabled()) {
                log.trace("directory to search " + file);
            }
            return new File(file, directory).exists();
        }

        if (log.isWarnEnabled()) {
            log.warn(_("nuitonutil.error.unknown.url.type", url));
        }
        return false;
    }
} // Resource
