/*
 * *##% 
 * JRedmine maven plugin
 * Copyright (C) 2009 CodeLutin
 *
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Lesser General Public License as
 * published by the Free Software Foundation, either version 3 of the
 * License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Lesser Public License for more details.
 *
 * You should have received a copy of the GNU General Lesser Public
 * License along with this program.  If not, see
 * <http://www.gnu.org/licenses/lgpl-3.0.html>.
 * ##%*
 */
package org.nuiton.jredmine.plugin.announcement;

import org.apache.maven.plugin.MojoExecutionException;
import org.apache.maven.plugin.logging.Log;
import org.apache.maven.plugins.changes.model.Action;
import org.apache.maven.plugins.changes.model.Release;
import org.apache.velocity.Template;
import org.apache.velocity.VelocityContext;
import org.apache.velocity.app.VelocityEngine;
import org.apache.velocity.context.Context;
import org.apache.velocity.exception.ResourceNotFoundException;
import org.codehaus.plexus.util.StringUtils;
import org.nuiton.jredmine.model.Attachment;
import org.nuiton.plugin.PluginHelper;

import java.io.File;
import java.io.FileOutputStream;
import java.io.OutputStreamWriter;
import java.io.Writer;
import java.util.Collections;
import java.util.HashMap;
import java.util.List;
import java.util.Map;

/**
 * @author chemit
 * @since 1.0.0
 */
public class AnnouncementGenerator {

    /**
     * The token any of urls denoting the base URL for the given url.
     */
    private static final String URL_TOKEN = "%URL%";
    /**
     * The token in {@link #getAttachmentLinkTemplate()} denoting the attachment ID.
     */
    private static final String ATTACHMENT_TOKEN = "%FILE%";

    private final AnnouncementGeneratorConfiguration configuration;

    public AnnouncementGenerator(AnnouncementGeneratorConfiguration configuration) {
        this.configuration = configuration;
    }

    public String getAttachmentLinkTemplate() {
        return configuration.getAttachmentLinkTemplate();
    }

    public Log getLog() {
        return configuration.getLog();
    }

    public String getUrl() {
        return configuration.getUrl();
    }

    public Map<Attachment, String> getAttachmentsUrls(Attachment[] attachments) {

        // transform attachments urls

        boolean hasAttachmentLinks = canGenerateAttachmentLinks();

        Map<Attachment, String> urls = null;

        if (hasAttachmentLinks) {

            urls = new HashMap<Attachment, String>();
            for (Attachment a : attachments) {
                String u = parseAttachmentLink(a.getId() + "");
                urls.put(a, u);
            }
        } else {
            getLog().warn("can not render attachments urls");
        }

        return urls;
    }

    public Context createVelocityContext(List<?> releases) throws MojoExecutionException {

        // prepare velocity context
        Context context = new VelocityContext();
        context.put("releases", releases);
        context.put("groupId", configuration.getGroupId());
        context.put("artifactId", configuration.getArtifactId());
        context.put("version", configuration.getVersionId());
        context.put("packaging", configuration.getPackaging());
        context.put("url", configuration.getUrl());
        context.put("projectUrl", configuration.getProjectUrl());

        Release release = getLatestRelease(releases, configuration.getVersionId());

        context.put("release", release);
        context.put("introduction", configuration.getIntroduction());
        context.put("developmentTeam", configuration.getDevelopmentTeam());
        context.put("finalName", configuration.getFinalName());
        context.put("urlDownload", configuration.getUrlDownload());
        context.put("mavenRepoUrl", configuration.getDeploymentUrl());
        context.put("project", configuration.getProject());

        // add artifacts in context
        Map<File, String> artifactUrls = configuration.getArtifactUrls();
        if (artifactUrls == null || artifactUrls.isEmpty()) {
            context.put("withArtifacts", false);

        } else {
            context.put("withArtifacts", true);
            context.put("artifactUrls", artifactUrls);
        }
        // add attachments in context

        Map<Attachment, String> attachmentUrls = configuration.getAttachmentUrls();
        if (attachmentUrls == null || attachmentUrls.isEmpty()) {
            context.put("withAttachments", false);

        } else {
            context.put("withAttachments", true);
            context.put("attachmentUrls", attachmentUrls);
        }

        Map<String, Object> announceParameters = configuration.getAnnounceParameters();
        if (announceParameters == null) {
            // empty Map to prevent NPE in velocity execution
            context.put("announceParameters", Collections.EMPTY_MAP);
        } else {
            context.put("announceParameters", announceParameters);
        }
        return context;
    }

    public void doGenerate(VelocityEngine engine, Context context, File out, String templatePath, String templateEncoding)
            throws MojoExecutionException {

        try {

            PluginHelper.createDirectoryIfNecessary(out.getParentFile());

            Writer writer = new OutputStreamWriter(new FileOutputStream(out), templateEncoding);

            Template velocityTemplate = engine.getTemplate(templatePath, templateEncoding);

            velocityTemplate.merge(context, writer);

            writer.flush();

            writer.close();

        } catch (ResourceNotFoundException rnfe) {
            throw new MojoExecutionException("Resource not found.", rnfe);
        } catch (Exception e) {
            throw new MojoExecutionException(e.toString(), e);
        }
    }

    /**
     * Checks whether links to the issues can be generated.
     *
     * @return <code>true</code> if issue links can be generated, <code>false</code> otherwise.
     */
    public boolean canGenerateAttachmentLinks() {
        String attachmentLinkTemplate = getAttachmentLinkTemplate();
        String url = getUrl();
        return !StringUtils.isBlank(attachmentLinkTemplate) &&
                attachmentLinkTemplate.indexOf(ATTACHMENT_TOKEN) > 0 &&
                (attachmentLinkTemplate.indexOf(URL_TOKEN) < 0 || !StringUtils.isBlank(url));
    }

    protected String parseAttachmentLink(String id) {
        String parseLink;
        String issueLink = getAttachmentLinkTemplate();
        parseLink = issueLink.replaceFirst(ATTACHMENT_TOKEN, id);

        if (parseLink.indexOf(URL_TOKEN) >= 0) {
//            String u = this.url.substring(0, this.url.lastIndexOf("/"));
//            parseLink = parseLink.replaceFirst(URL_TOKEN, u);
            parseLink = parseLink.replaceFirst(URL_TOKEN, getUrl());
        }

        return parseLink;
    }

    /**
     * Get the latest release by matching the supplied releases
     * with the version from the pom.
     *
     * @param releases       list of releases
     * @param releaseVersion the release version
     * @return A <code>Release</code> that matches the next release of the current project
     * @throws MojoExecutionException if any pb
     */
    public Release getLatestRelease(List<?> releases, String releaseVersion)
            throws MojoExecutionException {
        boolean isFound;

        Release release;

        String pomVersion = releaseVersion;

        getLog().debug("Found " + releases.size() + " releases.");

        for (Object release1 : releases) {
            release = (Release) release1;
            if (getLog().isDebugEnabled()) {
                getLog().debug("The release: " + release.getVersion() + " has " + release.getActions().size() + " actions.");
            }

            if (release.getVersion() != null && release.getVersion().equals(pomVersion)) {
                isFound = true;
                if (getLog().isDebugEnabled()) {
                    getLog().debug("Found the correct release: " + release.getVersion());
                    logRelease(release);
                }
                return release;
            }
        }

        release = getRelease(releases, pomVersion);
        isFound = (release != null);

        if (!isFound) {
            throw new MojoExecutionException("Couldn't find the release '" + pomVersion + "' among the supplied releases.");
        } else {
        }
        return release;
    }

    protected Release getRelease(List<?> releases, String version) {
        Release release;
        for (Object release1 : releases) {
            release = (Release) release1;
            if (getLog().isDebugEnabled()) {
                getLog().debug("The release: " + release.getVersion() + " has " + release.getActions().size() + " actions.");
            }

            if (release.getVersion() != null && release.getVersion().equals(version)) {
                if (getLog().isDebugEnabled()) {
                    getLog().debug("Found the correct release: " + release.getVersion());
                    logRelease(release);
                }
                return release;
            }
        }
        return null;
    }

    private void logRelease(Release release) {
        Action action;
        for (Object o : release.getActions()) {
            action = (Action) o;
            getLog().debug("o " + action.getType());
            getLog().debug("issue : " + action.getIssue());
            getLog().debug("action : " + action.getAction());
            getLog().debug("dueTo : " + action.getDueTo());
            getLog().debug("dev : " + action.getDev());
        }
    }

}
