/*
 * #%L
 * JRedmine :: Maven plugin
 * 
 * $Id: DisplayDataMojo.java 313 2012-07-15 20:50:20Z tchemit $
 * $HeadURL: http://svn.nuiton.org/svn/jredmine/tags/jredmine-1.4/jredmine-maven-plugin/src/main/java/org/nuiton/jredmine/plugin/DisplayDataMojo.java $
 * %%
 * Copyright (C) 2009 - 2012 Tony Chemit, CodeLutin
 * %%
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Lesser General Public License as 
 * published by the Free Software Foundation, either version 3 of the 
 * License, or (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Lesser Public License for more details.
 * 
 * You should have received a copy of the GNU General Lesser Public 
 * License along with this program.  If not, see
 * <http://www.gnu.org/licenses/lgpl-3.0.html>.
 * #L%
 */
package org.nuiton.jredmine.plugin;

import com.google.common.collect.Lists;
import com.google.common.collect.Maps;
import org.apache.maven.plugin.MojoFailureException;
import org.apache.maven.plugins.annotations.Mojo;
import org.apache.maven.plugins.annotations.Parameter;
import org.nuiton.io.rest.RequestFactory;
import org.nuiton.io.rest.RestRequestBuilder;
import org.nuiton.jredmine.RedmineServiceImplementor;
import org.nuiton.jredmine.model.I18nAble;
import org.nuiton.jredmine.model.IdAble;
import org.nuiton.jredmine.model.Issue;
import org.nuiton.jredmine.model.IssueCategory;
import org.nuiton.jredmine.model.IssuePriority;
import org.nuiton.jredmine.model.IssueStatus;
import org.nuiton.jredmine.model.ModelHelper;
import org.nuiton.jredmine.model.News;
import org.nuiton.jredmine.model.Project;
import org.nuiton.jredmine.model.Tracker;
import org.nuiton.jredmine.model.User;
import org.nuiton.jredmine.model.Version;
import org.nuiton.jredmine.rest.RedmineRequestFactory;
import org.nuiton.plugin.PluginHelper;

import java.util.Arrays;
import java.util.Collections;
import java.util.List;
import java.util.Locale;
import java.util.Map;

/**
 * Display in console some data from redmine's server.
 * <p/>
 * This goals requires no authentication to the server.
 *
 * @author tchemit <chemit@codelutin.com>
 * @since 1.0.0
 */
@Mojo(name = "display-data", requiresOnline = true, requiresProject = true)
public class DisplayDataMojo extends AbstractRedmineMojo implements RedmineProjectAware, RedmineVersionAware {

    /**
     * Redmine project name.
     *
     * @since 1.0.0
     */
    @Parameter(property = "redmine.projectId", defaultValue = "${project.artifactId}", required = true)
    protected String projectId;

    /**
     * redmine version name.
     *
     * @since 1.0.0
     */
    @Parameter(property = "redmine.versionId", defaultValue = "${project.version}")
    protected String versionId;

    /**
     * The data types to display : {@code issueSatus}, {@code issuePriority},
     * {@code issueCategory}, {@code tracker} and {@code user}.
     * <p/>
     * {@code issueStatus} and {@code issuePriority} are common for all
     * projects, so for those datas, no extrat configuration is required.
     * <p/>
     * {@code isseCategoriy}, {@code tracker} and {@code user} are project
     * specific, so you have to fill also the {@code projectId} parameter to
     * obtain data for those types.
     * <p/>
     * Note : you can specifiy several types separated by comma
     *
     * @since 1.0.0
     */
    @Parameter(property = "types", required = true)
    protected String types;

    /**
     * A flag to sort of not retrived data by id.
     *
     * @since 1.0.0
     */
    @Parameter(property = "sortById", defaultValue = "true", required = true)
    protected boolean sortById;

    /**
     * Flag to know if anonymous connexion to redmine server is required.
     * <p/>
     * For this goal, the default value is {@code true}
     * <p/>
     * <b>Note:</b> If set to {@code false}, you should fill {@link #username}
     * and {@link #password} properties.
     *
     * @since 1.1.3
     */
    @Parameter(property = "anonymous", defaultValue = "true")
    protected boolean anonymous;

    /** list of actions to perform */
    protected Map<String, Class<?>> actions;

    private final Map<Class<?>, String> typesToRequestName;

    private final Map<String, Class<?>> namesToType;

    public DisplayDataMojo() {
        typesToRequestName = Maps.newHashMap();
        typesToRequestName.put(Project.class, ModelHelper.GET_ALL_PROJECT_REQUEST_NAME);
        typesToRequestName.put(Version.class, ModelHelper.GET_ALL_VERSION_REQUEST_NAME);
        typesToRequestName.put(Issue.class, ModelHelper.GET_ALL_ISSUES_REQUEST_NAME);
        typesToRequestName.put(News.class, ModelHelper.GET_ALL_NEWS_REQUEST_NAME);
        typesToRequestName.put(IssueStatus.class, ModelHelper.GET_ALL_ISSUE_STATUS_REQUEST_NAME);
        typesToRequestName.put(IssuePriority.class, ModelHelper.GET_ALL_ISSUE_PRIORITY_REQUEST_NAME);
        typesToRequestName.put(IssueCategory.class, ModelHelper.GET_ALL_ISSUE_CATEGORY_REQUEST_NAME);
        typesToRequestName.put(Tracker.class, ModelHelper.GET_ALL_TRACKER_REQUEST_NAME);
        typesToRequestName.put(User.class, ModelHelper.GET_ALL_USER_REQUEST_NAME);

        namesToType = Maps.newHashMap();
        namesToType.put("project", Project.class);
        namesToType.put("version", Version.class);
        namesToType.put("issue", Issue.class);
        namesToType.put("news", News.class);
        namesToType.put("issuestatus", IssueStatus.class);
        namesToType.put("issuepriority", IssuePriority.class);
        namesToType.put("issuecategory", IssueCategory.class);
        namesToType.put("tracker", Tracker.class);
        namesToType.put("user", User.class);
    }

    @Override
    protected boolean checkRunOnceDone() {
        return false;
    }

    @Override
    protected boolean isGoalSkip() {
        return false;
    }

    @Override
    protected boolean isRunOnce() {
        return false;
    }

    @Override
    protected void init() throws Exception {

        if (versionId != null) {
            // always remove snapshot (TODO-TC20100124 perharps should it be a flag ?)
            versionId = PluginHelper.removeSnapshotSuffix(versionId);
        }

        super.init();

        actions = getDownloadActions();
    }

    @Override
    public boolean isAnonymous() {
        return anonymous;
    }

    @Override
    public void setAnonymous(boolean anonymous) {
        this.anonymous = anonymous;
    }

    ///////////////////////////////////////////////////////////////////////////
    /// RedmineProjectAware
    ///////////////////////////////////////////////////////////////////////////

    @Override
    public String getProjectId() {
        return projectId;
    }

    @Override
    public void setProjectId(String projectId) {
        this.projectId = projectId;
    }

    ///////////////////////////////////////////////////////////////////////////
    /// RedmineVersionAware
    ///////////////////////////////////////////////////////////////////////////

    @Override
    public String getVersionId() {
        return versionId;
    }

    @Override
    public void setVersionId(String versionId) {
        this.versionId = versionId;
    }

    @Override
    protected boolean checkSkip() {
        boolean canContinue;
        if (actions == null || actions.isEmpty()) {
            // no data to treate
            getLog().warn("No data types detected, you must fill the " +
                          "required parameter types, will skip goal");
            canContinue = false;
        } else {

            canContinue = super.checkSkip();
        }
        return canContinue;
    }

    @Override
    protected void doAction() throws Exception {

        StringBuilder buffer = new StringBuilder("\n");
        for (Map.Entry<String, Class<?>> entry : actions.entrySet()) {

            Class<?> type = entry.getValue();
            String dataTypeName = entry.getKey();
            String scope = getDataScope(type);
            String requestName = typesToRequestName.get(type);

            IdAble[] datas;

            RedmineServiceImplementor implementor =
                    service.getDelegateImplementor();

            if (ModelHelper.VERSION_SCOPE.equals(scope)) {
                datas = (IdAble[]) implementor.getDatas(requestName, type, projectId, versionId);
            } else if (ModelHelper.PROJECT_SCOPE.equals(scope)) {
                datas = (IdAble[]) implementor.getDatas(requestName, type, projectId);
            } else {
                datas = (IdAble[]) implementor.getDatas(requestName, type);
            }
            List<IdAble> list = Arrays.asList(datas);
            if (sortById) {
                Collections.sort(list, ModelHelper.ID_ABLE_COMPARATOR);
            }
            buffer.append("\nValues of '").append(dataTypeName).append('\'');
            if (ModelHelper.VERSION_SCOPE.equals(scope) || ModelHelper.PROJECT_SCOPE.equals(scope)) {
                buffer.append(" for project '").append(projectId).append("'");
            }
            if (ModelHelper.VERSION_SCOPE.equals(scope)) {
                buffer.append(" and version '").append(versionId).append("'");
            }
            buffer.append('\n');
            for (IdAble data : list) {
                buffer.append("  - ");
                buffer.append(data.getId());
                buffer.append(" = ");
                buffer.append(((I18nAble) data).getName());
                buffer.append('\n');
            }
        }
        System.out.println(buffer.toString());
    }

    protected Map<String, Class<?>> getDownloadActions() throws MojoFailureException {

        Map<String, Class<?>> results = Maps.newTreeMap();

        List<Class<?>> universeList = Lists.newArrayList(typesToRequestName.keySet());

        for (String dataType : types.split(",")) {

            dataType = dataType.toLowerCase(Locale.ENGLISH);

            Class<?> result = namesToType.get(dataType);

            if (result == null || !universeList.contains(result)) {
                // can not accept these value
                throw new MojoFailureException(
                        "The data type '" + result +
                        "' can not be used, Accepted types are : " +
                        universeList);
            }

            String scope = getDataScope(result);

            if (ModelHelper.VERSION_SCOPE.equals(scope) || ModelHelper.PROJECT_SCOPE.equals(scope)) {

                // check projectId used
                if (projectId == null || projectId.isEmpty()) {
                    throw new MojoFailureException(
                            "The data type '" + result +
                            "', depends on a project, but the 'projectId' was " +
                            "not filled.");
                }
            }
            if (ModelHelper.VERSION_SCOPE.equals(scope)) {
                // check versionId use
                if (versionId == null || versionId.isEmpty()) {
                    throw new MojoFailureException(
                            "The data type '" + result +
                            "', depends on a version, but the 'versionId' was " +
                            "not filled.");
                }
            }
            results.put(dataType, result);
        }
        return results;
    }

    protected String getDataScope(Class<?> type) {

        RequestFactory requestFactory =
                service.getDelegateImplementor().getSession().getRequestFactory();

        String requestName = typesToRequestName.get(type);
        RestRequestBuilder requestBuilder =
                requestFactory.getRequestBuilder(requestName);

        return RedmineRequestFactory.getRequestScope(requestBuilder);
    }
}
