/*
 * #%L
 * JAXX :: Runtime
 * %%
 * Copyright (C) 2008 - 2014 Code Lutin, Tony Chemit
 * %%
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Lesser General Public License as 
 * published by the Free Software Foundation, either version 3 of the 
 * License, or (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Lesser Public License for more details.
 * 
 * You should have received a copy of the GNU General Lesser Public 
 * License along with this program.  If not, see
 * <http://www.gnu.org/licenses/lgpl-3.0.html>.
 * #L%
 */
package org.nuiton.jaxx.swing.extra.layer;

import javax.swing.Action;
import javax.swing.ImageIcon;
import javax.swing.JComponent;
import javax.swing.JLayer;
import javax.swing.plaf.LayerUI;
import java.awt.AWTEvent;
import java.awt.AlphaComposite;
import java.awt.Color;
import java.awt.Component;
import java.awt.Graphics;
import java.awt.Graphics2D;
import java.awt.Point;
import java.awt.RenderingHints;
import java.awt.event.ActionEvent;
import java.awt.event.KeyEvent;
import java.awt.event.MouseEvent;
import java.awt.event.MouseWheelEvent;
import java.awt.image.BufferedImage;
import java.beans.PropertyChangeEvent;

/**
 * A JXLayer ui implementation that permits to block a component but still
 * allow an action when clicking everywhere on the layer.
 * <p/>
 * Moreover, an icon can be added on the right-top icon painted and changed
 * when the mouse is over the layer.
 * <p/>
 * You can change the blocking and accepting icon.
 * <p/>
 * To hook an click on the layer's icon, you can :
 * <p/>
 * <ul><li>pass an Action via method {@link #setAcceptAction(Action)}</li>
 * <li>override the method {@link #acceptEvent(MouseEvent, JLayer)}</li>
 * </ul>
 *
 * @author Tony Chemit - chemit@codelutin.com
 * @since 1.3
 */
public class BlockingLayerUI2 extends LayerUI<JComponent> {

    public static final String CAN_CLICK_PROPERTY = "canClick";

    public static final String ACCEPT_ICON_PROPERTY = "acceptIcon";

    public static final String BLOCK_ICON_PROPERTY = "blockIcon";

    /** Action to be treated when click on icon */
    protected Action acceptAction;

    /** Icon when you can not click */
    protected BufferedImage blockIcon;

    /** Icon when you can click */
    protected BufferedImage acceptIcon;

    /** Optinal color to put fill background when blocking */
    protected Color blockingColor;

    /** Internal state to known when we can accept click */
    protected boolean canClick;

    public void setAcceptAction(Action acceptAction) {
        this.acceptAction = acceptAction;
    }

    public void setAcceptIcon(ImageIcon acceptIcon) {
        this.acceptIcon = prepareIcon(acceptIcon);
        firePropertyChange(ACCEPT_ICON_PROPERTY, null, acceptIcon);
        setDirty(true);
    }

    public void setBlockIcon(ImageIcon blockIcon) {
        this.blockIcon = prepareIcon(blockIcon);
        firePropertyChange(BLOCK_ICON_PROPERTY, null, blockIcon);
        setDirty(true);
    }

    public void setCanClick(boolean canClick) {
        boolean oldvalue = this.canClick;
        this.canClick = canClick;
        firePropertyChange(CAN_CLICK_PROPERTY, oldvalue, canClick);
        if (oldvalue != canClick) {
            setDirty(true);
        }
    }

    public void setDirty(boolean dirty) {
        firePropertyChange("dirty", null, dirty);
    }

    @Override
    public void applyPropertyChange(PropertyChangeEvent evt, JLayer<? extends JComponent> l) {
        super.applyPropertyChange(evt, l);
        if (!"dirty".equals(evt.getPropertyName())
            || evt.getNewValue() == Boolean.TRUE) {
            l.repaint();
        }
    }

    public void setBlockingColor(Color blockingColor) {
        this.blockingColor = blockingColor;
    }

    public void setBlockIcon(BufferedImage blockIcon) {
        this.blockIcon = blockIcon;
    }

    public BufferedImage getBlockIcon() {
        return blockIcon;
    }

    protected BufferedImage getAcceptIcon() {
        return acceptIcon;
    }

    public boolean isCanClick() {
        return canClick;
    }

    @Override
    public BlockingLayerUI2 clone() {
        BlockingLayerUI2 clone = new BlockingLayerUI2();
        clone.acceptAction = acceptAction;
        clone.acceptIcon = acceptIcon;
        clone.blockIcon = blockIcon;
        clone.blockingColor = blockingColor;
        clone.setCanClick(false);
        return clone;
    }

    @Override
    protected void processKeyEvent(KeyEvent e, JLayer<? extends JComponent> l) {
        e.consume();
    }

    @Override
    protected void processMouseMotionEvent(MouseEvent e, JLayer<? extends JComponent> l) {
        e.consume();
    }

    @Override
    protected void processMouseWheelEvent(MouseWheelEvent e, JLayer<? extends JComponent> l) {
        e.consume();
    }

    @Override
    protected void processMouseEvent(MouseEvent e, JLayer<? extends JComponent> l) {
        switch (e.getID()) {
            case MouseEvent.MOUSE_ENTERED:
                setCanClick(true);
                break;
            case MouseEvent.MOUSE_EXITED:
                setCanClick(false);
                break;
            case MouseEvent.MOUSE_CLICKED:
                if (canClick) {
                    acceptEvent(e, l);
                }
                break;
        }
        e.consume();
    }

    private static final long ACCEPTED_EVENTS =
            AWTEvent.COMPONENT_EVENT_MASK |
            AWTEvent.CONTAINER_EVENT_MASK |
            AWTEvent.FOCUS_EVENT_MASK |
            AWTEvent.KEY_EVENT_MASK |
            AWTEvent.MOUSE_WHEEL_EVENT_MASK |
            AWTEvent.MOUSE_MOTION_EVENT_MASK |
            AWTEvent.MOUSE_EVENT_MASK |
            AWTEvent.INPUT_METHOD_EVENT_MASK |
            AWTEvent.HIERARCHY_EVENT_MASK |
            AWTEvent.HIERARCHY_BOUNDS_EVENT_MASK;

    @Override
    public void installUI(JComponent c) {
        super.installUI(c);
        JLayer l = (JLayer) c;
        l.setLayerEventMask(ACCEPTED_EVENTS);
    }

    @Override
    public void uninstallUI(JComponent c) {
        super.uninstallUI(c);
        JLayer l = (JLayer) c;
        l.setLayerEventMask(0);
    }

    @Override
    public void paint(Graphics g, JComponent c) {
        super.paint(g, c);
        JLayer l = (JLayer) c;
        Graphics2D g2 = (Graphics2D) g;
        if (blockingColor != null) {
            // to be in sync with the view if the layer has a border
            /*Insets layerInsets = l.getInsets();
            g2.translate(layerInsets.left, layerInsets.top);

            JComponent view = l.getView();
            // To prevent painting on view's border
            Insets insets = view.getInsets();
            g2.clip(new Rectangle(insets.left, insets.top,
            view.getWidth() - insets.left - insets.right,
            view.getHeight() - insets.top - insets.bottom));
             */

            g2.setColor(blockingColor);
            g2.setComposite(AlphaComposite.getInstance(AlphaComposite.SRC_OVER, .1f));
            g2.fillRect(0, 0, l.getWidth(), l.getHeight());
        }
        if (getCurrentIcon() != null) {
            g2.drawImage(getCurrentIcon(), l.getWidth() - getCurrentIcon().getWidth() - 1, 0, null);
        }
    }

    protected void acceptEvent(MouseEvent e, JLayer<? extends JComponent> l) {
        if (acceptAction != null) {
            acceptAction.putValue("layer", l);
            Component source = l.getView();
            acceptAction.actionPerformed(new ActionEvent(source, 0, "accept"));
        }
    }

    protected BufferedImage getCurrentIcon() {
        return canClick ? acceptIcon : blockIcon;
    }

    protected BufferedImage prepareIcon(ImageIcon image) {
        BufferedImage icon = new BufferedImage(image.getIconWidth(), image.getIconHeight(), BufferedImage.TYPE_INT_ARGB);
        Graphics2D g2 = (Graphics2D) icon.getGraphics();
        g2.setRenderingHint(RenderingHints.KEY_ANTIALIASING, RenderingHints.VALUE_ANTIALIAS_ON);
        g2.setRenderingHint(RenderingHints.KEY_STROKE_CONTROL, RenderingHints.VALUE_STROKE_PURE);
        g2.drawImage(image.getImage(), 0, 0, null);
        g2.dispose();
        return icon;
    }

    protected void updateCanClickState(JLayer<JComponent> l, MouseEvent e) {
        // udpate toolTipText
        Point layerLocation = l.getView().getLocation();
        Point mousePoint = e.getPoint();
        BufferedImage currentIcon = getCurrentIcon();
        if (currentIcon == null) {
            setCanClick(false);
            return;
        }
        int minX = (int) layerLocation.getX() + l.getWidth() - currentIcon.getWidth();
        int maxX = (int) layerLocation.getX() + l.getWidth();
        int minY = 0;
        int maxY = currentIcon.getHeight();
        boolean accept = minX <= mousePoint.getX() && mousePoint.getX() <= maxX;
        accept &= minY <= mousePoint.getLocation().getY() && mousePoint.getLocation().getY() <= maxY;
        setCanClick(accept);
    }
}
