package org.nuiton.jaxx.application.swing.action;

/*
 * #%L
 * JAXX :: Application Swing
 * %%
 * Copyright (C) 2008 - 2014 Code Lutin, Tony Chemit
 * %%
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Lesser General Public License as 
 * published by the Free Software Foundation, either version 3 of the 
 * License, or (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Lesser Public License for more details.
 * 
 * You should have received a copy of the GNU General Lesser Public 
 * License along with this program.  If not, see
 * <http://www.gnu.org/licenses/lgpl-3.0.html>.
 * #L%
 */

import com.google.common.base.Preconditions;
import org.apache.commons.lang3.StringUtils;
import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.jdesktop.beans.AbstractBean;
import org.nuiton.decorator.Decorator;
import org.nuiton.jaxx.application.ApplicationConfiguration;
import org.nuiton.jaxx.application.swing.AbstractApplicationUIHandler;
import org.nuiton.jaxx.application.swing.ApplicationUI;
import org.nuiton.jaxx.application.swing.ApplicationUIContext;
import org.nuiton.jaxx.application.type.ApplicationProgressionModel;
import org.nuiton.jaxx.runtime.JaxxFileChooser;

import javax.swing.JOptionPane;
import java.awt.Component;
import java.io.File;
import java.util.Arrays;

import static org.nuiton.i18n.I18n.t;

/**
 * Tutti base action.
 *
 * @author Tony Chemit - chemit@codelutin.com
 * @since 2.8
 */
public abstract class AbstractApplicationAction<M extends AbstractBean, UI extends ApplicationUI<M, ?>, H extends AbstractApplicationUIHandler<M, UI>>
        extends AbstractBean {

    /** Logger. */
    private static final Log log = LogFactory.getLog(AbstractApplicationAction.class);

    public static final String PROPERTY_DONE = "done";

    public static final String MESAGE_FORMAT = "<html>%s</html>";

    protected final H handler;

    protected String actionDescription;

    protected final boolean hideBody;

    protected final Object lock = new Object();

    public abstract void doAction() throws Exception;

    protected AbstractApplicationAction(H handler, boolean hideBody) {
        this.handler = handler;
        this.hideBody = hideBody;
    }

    public boolean prepareAction() throws Exception {
        // by default nothing to prepare
        return true;
    }

    protected void releaseAction() {
        // by default nothing to clean
        firePropertyChange(PROPERTY_DONE, null, true);
    }

    public void postSuccessAction() {
        // by default nothing to do after action
    }

    public void postFailedAction(Throwable error) {
        // by default nothing to do after action
    }

    public H getHandler() {
        return handler;
    }

    public M getModel() {
        return handler.getModel();
    }

    public final UI getUI() {
        return handler.getUI();
    }

    public ApplicationUIContext getContext() {
        return handler.getContext();
    }

    public String getActionDescription() {
        return actionDescription;
    }

    public void setActionDescription(String actionDescription) {
        this.actionDescription = actionDescription;
    }

    public void setProgressionModel(ApplicationProgressionModel progressionModel) {
        getContext().getActionUI().getModel().setProgressionModel(progressionModel);
    }

    protected ApplicationProgressionModel getProgressionModel() {
        return getContext().getActionUI().getModel().getProgressionModel();
    }

    public boolean isHideBody() {
        return hideBody;
    }

    protected ApplicationConfiguration getConfig() {
        return getContext().getConfiguration();
    }

    protected abstract void sendMessage(String message);

    protected <O> Decorator<O> getDecorator(Class<O> type, String name) {
        Decorator<O> decorator = handler.getDecorator(type, name);
        Preconditions.checkNotNull(decorator);
        return decorator;
    }

    protected String decorate(Object object) {
        return getDecorator(object.getClass(), null).toString(object);
    }

    protected String decorate(Object object, String context) {
        return getDecorator(object.getClass(), context).toString(object);
    }

    /**
     * Choisir un fichier via un sélecteur graphique de fichiers.
     *
     * @param title       le titre du dialogue de sélection
     * @param buttonLabel le label du boutton d'acceptation
     * @param filters     les filtres + descriptions sur le sélecteur de
     *                    fichiers
     * @return le fichier choisi ou le fichier incoming si l'opération a été
     * annulée
     */
    protected File chooseFile(String title,
                              String buttonLabel,
                              String... filters) {

        File file = JaxxFileChooser.forLoadingFile()
                .setTitle(title)
                .setParent((Component) getUI())
                .setApprovalText(buttonLabel)
                .setPatternOrDescriptionFilters(Arrays.asList(filters))
                .choose();

//        File file = FileChooserUtil.getFile(title,
//                                            buttonLabel,
//                                            getContext().getMainUI(),
//                                            filters);
        if (log.isDebugEnabled()) {
            log.debug(title + " : " + file);
        }
        if (file != null) {
            File newDir = file.isDirectory() ? file : file.getParentFile();
            JaxxFileChooser.setCurrentDirectory(newDir);
        }
        return file;
    }

    /**
     * Sauver un fichier via un sélecteur graphique de fichiers.
     *
     * @param title       le titre du dialogue de sélection
     * @param buttonLabel le label du boutton d'acceptation
     * @param filters     les filtres + descriptions sur le sélecteur de
     *                    fichiers
     * @return le fichier choisi ou {@code null} si pas de fichier choisi ou
     * pas voulu écrasé un fichier existant.
     */
    protected File saveFile(File defaultFile,
                            String filename,
                            String extension,
                            String title,
                            String buttonLabel,
                            String... filters) {

        if (defaultFile != null && JaxxFileChooser.isCurrentDirectoryDefault()) {

            // set default directory to this one
            JaxxFileChooser.setCurrentDirectory(defaultFile);
        }
        File file = saveFile(filename, extension, title, buttonLabel, filters);
        return file;
    }

    /**
     * Sauver un fichier via un sélecteur graphique de fichiers.
     *
     * @param title       le titre du dialogue de sélection
     * @param buttonLabel le label du boutton d'acceptation
     * @param filters     les filtres + descriptions sur le sélecteur de
     *                    fichiers
     * @return le fichier choisi ou {@code null} si pas de fichier choisi ou
     * pas voulu écrasé un fichier existant.
     */
    protected File saveFile(String filename,
                            String extension,
                            String title,
                            String buttonLabel,
                            String... filters) {

        boolean withExtension = StringUtils.isNotBlank(extension);
        String filenameSuffix = withExtension ? "." + extension : "";

        File file = JaxxFileChooser.forSaving()
                .setTitle(title)
                .setParent((Component) getUI())
                .setApprovalText(buttonLabel)
                .setPatternOrDescriptionFilters(Arrays.asList(filters))
                .setFilename(filename + filenameSuffix)
                .choose();

        //File file = FileChooserUtil.saveFile(filename + filenameSuffix, title, buttonLabel, getContext().getMainUI(), filters);
        if (log.isDebugEnabled()) {
            log.debug(title + " : " + file);
        }
        if (file != null) {
            Preconditions.checkState(!file.isDirectory());

            // add extension if missing
            if (withExtension && !file.getName().endsWith(filenameSuffix)) {
                file = new File(file.getParentFile(), file.getName() + filenameSuffix);
            }

            // ask user to confirm overwrite.
            boolean confirm = askOverwriteFile(file);

            if (confirm) {

                // on conserve le répertoire (pour une prochaine utilisation)
                JaxxFileChooser.setCurrentDirectory(file.getParentFile());
            } else {

                // l'utilisateur n'a pas confirmé l'écrasement
                // donc pas de fichier en retour
                file = null;
            }
        }

        return file;
    }

    /**
     * Sauver un fichier via un sélecteur graphique de fichiers, en précisant
     * le répertoire où sauvegarder le fichier.
     *
     * @param startDirectory       le répertoire à utiliser pour choisir le fichier
     * @param keepCurrentDirectory un drapeau pour dire si on veut ou pas conserver le répertoire choisit comme nouveau répertoire par défaut
     * @param title                le titre du dialogue de sélection
     * @param buttonLabel          le label du boutton d'acceptation
     * @param filters              les filtres + descriptions sur le sélecteur de
     *                             fichiers
     * @return le fichier choisi ou {@code null} si pas de fichier choisi ou
     * pas voulu écrasé un fichier existant.
     * @since 2.17
     */
    protected File saveFileWithStartDirectory(File startDirectory,
                                              boolean keepCurrentDirectory,
                                              String filename,
                                              String extension,
                                              String title,
                                              String buttonLabel,
                                              String... filters) {

        boolean withExtension = StringUtils.isNotBlank(extension);
        String filenameSuffix = withExtension ? "." + extension : "";

        File file = JaxxFileChooser.forSaving()
                .setStartDirectory(startDirectory)
                .setKeepCurrentDirectory(keepCurrentDirectory)
                .setTitle(title)
                .setParent((Component) getUI())
                .setApprovalText(buttonLabel)
                .setPatternOrDescriptionFilters(Arrays.asList(filters))
                .setFilename(filename + filenameSuffix)
                .choose();

        //File file = FileChooserUtil.saveFile(filename + filenameSuffix, title, buttonLabel, getContext().getMainUI(), filters);
        if (log.isDebugEnabled()) {
            log.debug(title + " : " + file);
        }
        if (file != null) {
            Preconditions.checkState(!file.isDirectory());

            // add extension if missing
            if (withExtension && !file.getName().endsWith(filenameSuffix)) {
                file = new File(file.getParentFile(), file.getName() + filenameSuffix);
            }

            // ask user to confirm overwrite.
            boolean confirm = askOverwriteFile(file);

            if (confirm) {

                if (keepCurrentDirectory) {

                    // on conserve le répertoire (pour une prochaine utilisation)
                    JaxxFileChooser.setCurrentDirectory(file.getParentFile());
                    
                }

            } else {

                // l'utilisateur n'a pas confirmé l'écrasement
                // donc pas de fichier en retour
                file = null;
            }
        }

        return file;
    }

    protected boolean askOverwriteFile(File file) {
        boolean result;
        if (file.exists()) {

            // file exists ask user to overwrite
            String htmlMessage = String.format(
                    AbstractApplicationUIHandler.CONFIRMATION_FORMAT,
                    t("jaxx.application.common.askOverwriteFile.message", file),
                    t("jaxx.application.common.askOverwriteFile.help"));

            Component ui = getDialogParentComponent();

            result = JOptionPane.showConfirmDialog(
                    ui,
                    htmlMessage,
                    t("jaxx.application.common.askOverwriteFile.title"),
                    JOptionPane.OK_CANCEL_OPTION,
                    JOptionPane.QUESTION_MESSAGE) == JOptionPane.OK_OPTION;
        } else {

            // file does not exist
            result = true;
        }
        return result;
    }

    protected boolean askBeforeDelete(String title, String message) {
        String htmlMessage = String.format(
                AbstractApplicationUIHandler.CONFIRMATION_FORMAT,
                message,
                t("jaxx.application.common.askBeforeDelete.help"));
        Component ui = getDialogParentComponent();
        int i = JOptionPane.showConfirmDialog(
                ui,
                htmlMessage,
                title,
                JOptionPane.OK_CANCEL_OPTION,
                JOptionPane.QUESTION_MESSAGE);

        boolean result = i == JOptionPane.OK_OPTION;
        return result;
    }

    protected void displayInfoMessage(String title, String message) {
        String htmlMessage = String.format(MESAGE_FORMAT, message);
        Component ui = getDialogParentComponent();
        JOptionPane.showMessageDialog(
                ui,
                htmlMessage,
                title,
                JOptionPane.INFORMATION_MESSAGE);
    }

    protected void displayWarningMessage(String title, String message) {
        String htmlMessage = String.format(MESAGE_FORMAT, message);

        Component ui = getDialogParentComponent();
        JOptionPane.showMessageDialog(
                ui,
                htmlMessage,
                title,
                JOptionPane.WARNING_MESSAGE);
    }

    protected void displayErrorMessage(String title, String message) {
        String htmlMessage = String.format(MESAGE_FORMAT, message);
        Component ui = getDialogParentComponent();
        JOptionPane.showMessageDialog(
                ui,
                htmlMessage,
                title,
                JOptionPane.ERROR_MESSAGE);
    }

    protected Component getDialogParentComponent() {
        Component ui = getContext().getMainUI();
        if (ui==null) {
            ui = getHandler().getTopestUI();
        }
        return ui;
    }

    protected void createProgressionModelIfRequired(int total) {
        ApplicationProgressionModel progressionModel = getProgressionModel();
        if (progressionModel == null) {
            progressionModel = new ApplicationProgressionModel();
            progressionModel.setTotal(total);
            progressionModel.setMessage("");
            progressionModel.setCurrent(0);
            setProgressionModel(progressionModel);
        } else {
            progressionModel.adaptTotal(total);
        }
    }

    public ApplicationActionFactory getActionFactory() {
        return getContext().getActionFactory();
    }

    public ApplicationActionEngine getActionEngine() {
        return getContext().getActionEngine();
    }
}
