/*
 * #%L
 * I18n :: Maven Plugin
 * 
 * $Id: AbstractMakeI18nBundleMojo.java 1807 2010-11-12 11:48:31Z tchemit $
 * $HeadURL: http://svn.nuiton.org/svn/i18n/tags/i18n-2.0.1/maven-i18n-plugin/src/main/java/org/nuiton/i18n/plugin/bundle/AbstractMakeI18nBundleMojo.java $
 * %%
 * Copyright (C) 2007 - 2010 CodeLutin
 * %%
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Lesser General Public License as 
 * published by the Free Software Foundation, either version 3 of the 
 * License, or (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Lesser Public License for more details.
 * 
 * You should have received a copy of the GNU General Lesser Public 
 * License along with this program.  If not, see
 * <http://www.gnu.org/licenses/lgpl-3.0.html>.
 * #L%
 */
package org.nuiton.i18n.plugin.bundle;

import org.apache.maven.plugin.MojoFailureException;

import java.io.File;
import java.util.HashMap;
import java.util.Locale;
import java.util.Map;
import java.util.SortedSet;

/**
 * Common mojo to all final bundle maker.
 *
 * @author tchemit <chemit@codelutin.com>
 * @since 2.0
 */
public abstract class AbstractMakeI18nBundleMojo extends AbstractI18nBundleMojo {

    /**
     * Directory where to generate aggregated bundles.
     *
     * @parameter expression="${i18n.bundleOutputDir}" default-value="${basedir}/target/generated-sources/resources/META-INF"
     * @required
     * @since 1.0.0
     */
    protected File bundleOutputDir;

    /**
     * Name of the bundle to generate.
     *
     * @parameter expression="${i18n.bundleOutputName}" default-value="${project.artifactId}-i18n"
     * @required
     * @since 1.0.2
     */
    protected String bundleOutputName;

    /**
     * A flag to check that bundles are complete (no missing i18n translations).
     * <p/>
     * <b>Note :</b> This behaviour will be activated is {@link #failsIfWarning} is on.
     *
     * @parameter expression="${i18n.checkBundle}" default-value="true"
     * @since 1.0.0
     */
    protected boolean checkBundle;

    /**
     * A flag to show missing i18n translation.
     * <p/>
     * <b>Note :</b> Need the {@link #checkBundle} to be activated).
     *
     * @parameter expression="${i18n.showEmpty}" default-value="false"
     * @since 1.0.0
     */
    protected boolean showEmpty;

    /**
     * A flag to make the build fails if there is some warnings while generating
     * bundle, says when it misses some translations.
     * <p/>
     * <b>Note :</b> This parameter should be used in a release profile to
     * ensure bundles are complete.
     *
     * @parameter expression="${i18n.failsIfWarning}" default-value="false"
     * @since 2.0
     */
    protected boolean failsIfWarning;

    /** to keep all none translated i18n keys by locale. */
    protected Map<Locale, SortedSet<String>> unsafeMapping;

    @Override
    public void init() throws Exception {
        super.init();

        if (failsIfWarning) {

            // check bundle if wants to fail on unsafe bundles
            checkBundle = true;

            unsafeMapping = new HashMap<Locale, SortedSet<String>>();
        } else {
            unsafeMapping = null;
        }

        createDirectoryIfNecessary(bundleOutputDir);
    }

    protected void failsIfWarning() throws MojoFailureException {
        if (!failsIfWarning) {

            // no check
            return;
        }

        if (unsafeMapping != null && !unsafeMapping.isEmpty()) {

            // there is at least one not complete bundle, faisl the build
            throw new MojoFailureException(
                    "Bundles for locale(s) " + unsafeMapping.keySet() +
                    " are not complete. Use the -Di18n.showEmpty to see " +
                    "missing translations.");
        }
    }

}
