/**
 * *##% guix-compiler-swing
 * Copyright (C) 2009 CodeLutin
 *
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Lesser General Public License as
 * published by the Free Software Foundation, either version 3 of the
 * License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Lesser Public License for more details.
 *
 * You should have received a copy of the GNU General Lesser Public
 * License along with this program.  If not, see
 * <http://www.gnu.org/licenses/lgpl-3.0.html>. ##%*
 */
package org.nuiton.guix;

import java.beans.BeanInfo;
import java.beans.EventSetDescriptor;
import java.beans.IntrospectionException;
import java.beans.Introspector;
import java.lang.annotation.Annotation;
import java.lang.reflect.Method;
import java.util.HashMap;
import java.util.Map;
import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;

/**
 * Handles the Swing events
 *
 * @author kmorin
 */
public class SwingEventHandler {

    /** log */
    Log log = LogFactory.getLog(SwingEventHandler.class);
    /** Maps EventSetDescriptors with a map of the listener method and actions to execute */
    Map<EventSetDescriptor, Map<Method, String>> map = new HashMap<EventSetDescriptor, Map<Method, String>>();

    /**
     * Add an event to the map
     *
     * @param clazz the class of the object supposed to listen to the event
     * @param name name of the event
     * @param value code to execute when the events occurs
     * @return true if the name corresponds with an event
     * @throws java.beans.IntrospectionException if the JVM failed while getting the bean infos of the class
     */
    public boolean addEvent(Class clazz, String name, String value) throws IntrospectionException {
        if(clazz == null || name == null || value == null) {
            return false;
        }

        //get the bean info of the class
        BeanInfo bi = Introspector.getBeanInfo(clazz);
        //get the array of EventSetDescriptor of the class
        EventSetDescriptor[] esds = bi.getEventSetDescriptors();

        int i = 0;
        int j = 0;
        EventSetDescriptor esd = null;
        Method method = null;

        //while any event with the name "name" has been found, browse the EventSetDescriptors
        while (i < esds.length && method == null) {
            j = 0;
            //get the listener methods of the EventSetDescriptor
            Method[] methods = esds[i].getListenerMethods();
            while (j < esds[i].getListenerMethods().length && method == null) {
                if(methods[j].getName().equals(name)) {
                    method = methods[j];
                    esd = esds[i];
                }
                j++;
            }
            i++;
        }
        //if the event has been found in the bean info
        if (method != null) {
            //remove braces
            while(value.startsWith("{") && value.endsWith("}")) {
                value = value.substring(1, value.length() - 1);
            }
            //add final semicolon
            if(!value.endsWith(";"))
                value += ";";
            //if the EventSetDescriptor has not been yet recorded for another event
            if (map.get(esd) == null) {
                Map<Method, String> methods = new HashMap<Method, String>();
                methods.put(method, value);
                map.put(esd, methods);
            }
            else if(map.get(esd).get(method) != null) {
                map.get(esd).put(method, map.get(esd).get(method) + value);
            }
            else {
                map.get(esd).put(method, value);
            }
        }
        else
            if(log.isWarnEnabled())
                log.warn("Event " + name + " not found.");
        return method != null;
    }

    /**
     * Generates the code for all the events applied to an object
     *
     * @param object the objects to add the listeners
     * @return the generated code
     */
    public String generate(String object) {
        //result
        StringBuffer result = new StringBuffer();
        //for every EventSetDescriptor
        for (EventSetDescriptor e : map.keySet()) {
            //add new ListenerMethod
            result.append(object)
                    .append(".")
                    .append(e.getAddListenerMethod().getName())
                    .append("(new ")
                    //if only one event is handled for this EventSetDescriptor then add a listener, else add an adapter
                    .append(e.getListenerMethods().length == 1 ?
                        e.getListenerType().getCanonicalName() :
                        e.getListenerType().getCanonicalName().replace("Listener", "Adapter"))
                    .append("() {\n");
            //for every event handled for the EventSetDescriptor
            for(Method m : map.get(e).keySet()) {
                for(Annotation a : m.getDeclaredAnnotations()) {
                    result.append(a.toString()).append("\n");
                }
                //add the method that handles the event
                result.append("public ")
                        .append(m.getReturnType().getCanonicalName())
                        .append(" ")
                        .append(m.getName())
                        .append("(");
                Class clazz = m.getParameterTypes()[0];
                result.append(clazz.getCanonicalName())
                        .append(" event)")
                        .append("{\n")
                        .append(map.get(e).get(m))
                        .append("\n}\n");
            }
            result.append("});\n\n");
        }
        return result.toString();
    }
}
