/*
 * #%L
 * EUGene :: Maven plugin
 * 
 * $Id: BaseChainedFileWriter.java 1095 2011-09-21 18:02:48Z tchemit $
 * $HeadURL: http://svn.nuiton.org/svn/eugene/tags/eugene-2.4.2/maven-eugene-plugin/src/main/java/org/nuiton/eugene/plugin/writer/BaseChainedFileWriter.java $
 * %%
 * Copyright (C) 2006 - 2010 CodeLutin
 * %%
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Lesser General Public License as 
 * published by the Free Software Foundation, either version 3 of the 
 * License, or (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Lesser Public License for more details.
 * 
 * You should have received a copy of the GNU General Lesser Public 
 * License along with this program.  If not, see
 * <http://www.gnu.org/licenses/lgpl-3.0.html>.
 * #L%
 */

package org.nuiton.eugene.plugin.writer;

import org.apache.commons.collections.CollectionUtils;
import org.apache.maven.plugin.logging.Log;
import org.apache.maven.plugin.logging.SystemStreamLog;
import org.codehaus.plexus.util.DirectoryScanner;
import org.codehaus.plexus.util.FileUtils;
import org.nuiton.eugene.models.object.ObjectModel;
import org.nuiton.eugene.models.state.StateModel;
import org.nuiton.eugene.writer.AbstractChainedFileWriter;
import org.nuiton.eugene.writer.ChainedFileWriterConfiguration;
import org.nuiton.util.FileUtil;
import org.nuiton.util.Resource;

import java.io.File;
import java.io.IOException;
import java.io.InputStream;
import java.net.MalformedURLException;
import java.net.URI;
import java.net.URL;
import java.net.URLClassLoader;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.List;
import java.util.Map;
import java.util.Set;

/**
 * Surcharge de l'implentation abstraite pour avoir le logger de la console
 * maven.
 *
 * @author tchemit
 * @since 2.0.0
 */
public abstract class BaseChainedFileWriter extends AbstractChainedFileWriter {

    /** Logger */
    private Log log;

    protected BaseChainedFileWriter(String... propertyNameAndDescriptions) {
        super(propertyNameAndDescriptions);
    }

    public void setLog(Log log) {
        this.log = log;
    }

    public Log getLog() {
        if (log == null) {
            log = new SystemStreamLog();
        }

        return log;
    }

    @Override
    public final List<URL> getFiles(ChainedFileWriterConfiguration configuration,
                                    String inputPath,
                                    List<String> includePattern,
                                    boolean inClassPath) throws MalformedURLException, IllegalArgumentException {

        if (CollectionUtils.isEmpty(includePattern)) {
            throw new IllegalArgumentException("Must have at least one include pattern");
        }

        List<URL> result = new ArrayList<URL>();

        if (!inClassPath) {
            DirectoryScanner ds = new DirectoryScanner();
            File inputDirectory = new File(inputPath);
            ds.setBasedir(inputDirectory);
            ds.setIncludes(includePattern.toArray(new String[includePattern.size()]));
            ds.setExcludes(null);
            ds.addDefaultExcludes();
            ds.scan();
            for (String file : ds.getIncludedFiles()) {
                File in = new File(inputDirectory, file);
                result.add(in.toURI().toURL());
            }
            return result;
        }

        // search in class-path

        ClassLoader loader = configuration.getClassLoader();

        for (String pattern : includePattern) {

            String path = inputPath;

            //FIXME must change the file.separator to /
            if (!path.endsWith("/")) {
                path += "/";
            }
            path += pattern;

            if (path.startsWith("/")) {
                path = path.substring(1);
            }

            if (getLog().isDebugEnabled()) {
                getLog().debug("Try to seek class-path file " + path);
            }

            if (pattern.contains("*")) {

                // this is a multi-files to search
                List<URL> urlList = Resource.getURLs(path, (URLClassLoader) loader);
                if (CollectionUtils.isEmpty(urlList)) {

                    getLog().warn("Could not find in class-path files " + path);
                } else {
                    for (URL url : urlList) {

                        if (configuration.isVerbose()) {
                            getLog().info("Detected class-path file " + url);
                        }
                        result.add(url);
                    }
                }
            } else {

                // this is a simple unique search, improve performance
                // by searching directly in classloader the resource
                URL url = loader.getResource(path);
                if (url == null) {

                    getLog().warn("Could not find in class-path the file " + path);
                } else {

                    if (configuration.isVerbose()) {
                        getLog().info("Detected class-path file " + url);
                    }
                    result.add(url);
                }
            }
        }
        return result;
    }

    @Override
    public final List<URL> getResources(URL file) throws IOException {

        // obtain the properties files associated with the file
        String path = file.toString();

        String extension = "." + FileUtil.extension(path);

        String filename = FileUtil.basename(path, extension).concat(".properties");

        if (getLog().isDebugEnabled()) {
            getLog().info("path of file : " + path);
            getLog().info("path of resource : " + filename);
        }

        URL propertiesFile = URI.create(filename).toURL();

        if (path.startsWith("file:")) {

            // local file (not from class-path)
            // can test directly on resource if it exists
            if (!new File(propertiesFile.getFile()).exists()) {
                return null;
            }
            // resource exist, keep it
            return Arrays.asList(propertiesFile);
        }
        InputStream in = null;
        try {
            in = propertiesFile.openStream();

            // resource exist, keep it
            return Arrays.asList(propertiesFile);
        } catch (IOException eee) {

            // resource does not exists
            getLog().warn("Could not find resource " + propertiesFile);
            return null;
        } finally {
            if (in != null) {
                in.close();
            }
        }
    }

    @Override
    protected void initWriter(ChainedFileWriterConfiguration configuration) {
        super.initWriter(configuration);

        if (!configuration.isVerbose()) {

            // nothing else to do
            return;
        }

        // log writer config
        StringBuilder buffer = new StringBuilder();
        Set<Map.Entry<String, String>> set =
                getAuthorizedPropertyDescriptions().entrySet();
        if (set.isEmpty()) {
            buffer.append("Writer [");
            buffer.append(getInputProtocol());
            buffer.append("]");
            buffer.append(" does not use any specific properties.");
        } else {
            buffer.append("Writer [");
            buffer.append(getInputProtocol());
            buffer.append("]");
            buffer.append(" use ");
            buffer.append(properties.size());
            buffer.append(" properties :");
            if (getLog().isInfoEnabled()) {
                for (Map.Entry<String, String> e : set) {

                    String key = e.getKey();
                    Object value = properties.get(key);
                    if (value != null) {
                        buffer.append("\n");
                        buffer.append("  [");
                        buffer.append(key);
                        buffer.append("] (");
                        buffer.append(e.getValue());
                        buffer.append(") : ");
                        buffer.append(value);
                    }
                }
            }
        }
        getLog().info(buffer.toString());
    }

    protected boolean acceptObjectModelOrStateModel(String modelType) {
        modelType = modelType.trim().toLowerCase();
        return ObjectModel.NAME.equals(modelType) ||
               StateModel.NAME.equals(modelType);
    }

    protected void copyResources(ChainedFileWriterConfiguration configuration,
                                 File outputDirectory,
                                 File inputDirectory,
                                 File file,
                                 Map<File, List<File>> resourcesByFile) throws IOException {

        List<File> resources = resourcesByFile.get(file);

        if (CollectionUtils.isEmpty(resources)) {

            // no resource associated to the xmi file
            if (configuration.isVerbose()) {
                getLog().info("No resources to copy for file " + file);
            }
            return;
        }

        if (configuration.isVerbose()) {
            getLog().info("Copy " + resources.size() + " resource file(s).");
        }

        boolean overwrite = configuration.isOverwrite();

//        String parentPath = inputDirectory.getAbsolutePath();
        for (File in : resources) {

//            String relativePath = in.getAbsolutePath().substring(parentPath.length());

            File out = FileUtil.getRelativeFile(inputDirectory, outputDirectory, in);
//            File out = new File(outputDirectory, relativePath);
//            File out = new File(outputDirectory, in.getName());

            getLog().info("Copy file " + in + " to " + out);

            if (overwrite) {
                FileUtils.copyFile(in, out);
            } else {
                FileUtils.copyFileIfModified(in, out);
            }
        }
    }

}
