/* *##% Pollen
 * Copyright (C) 2009 CodeLutin
 *
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>. ##%*/

package org.chorem.pollen.votecounting.business;

import java.util.ArrayList;
import java.util.HashMap;
import java.util.HashSet;
import java.util.List;
import java.util.Map;
import java.util.Set;
import java.util.TreeSet;

import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.chorem.pollen.votecounting.dto.ChoiceDTO;
import org.chorem.pollen.votecounting.utils.PercentageBehavior;
import org.chorem.pollen.votecounting.utils.Utils;

/**
 * Méthode de dépouillement Condorcet.
 *
 * @author rannou
 * @version $Id: CondorcetMethod.java 2865 2010-02-09 16:17:54Z jruchaud $
 */
public class CondorcetMethod implements Method {

    /** Victoires pour chaque choix. */
    private Map<Choice, Double> victories;

    /** Choix de chaque personne. (personne >> choix >> classement) */
    private Map<String, Map<Choice, Double>> personVotes;

    /** Choix de chaque groupe. (groupe >> choix >> classement) */
    private Map<String, Map<Choice, Double>> groupVotes;

    /** log. */
    private static final Log log = LogFactory.getLog(CondorcetMethod.class);

    @Override
    public void executeCounting(List<Choice> choices, boolean byGroup) {
        if (byGroup) {
            initPersonVotes(choices);
            initGroupVotes(choices);
            calculateVictories(choices, groupVotes, byGroup);
            updateChoices(choices);
        } else {
            initPersonVotes(choices);
            calculateVictories(choices, personVotes, byGroup);
            updateChoices(choices);
        }
    }

    /**
     * Initialisation des votes par personne. Les votes sont initialisés avec
     * les valeurs des choix.
     *
     * @param choices Les choix utilisés pour le dépouillement.
     */
    private void initPersonVotes(List<Choice> choices) {
        personVotes = new HashMap<String, Map<Choice, Double>>();
        for (Choice choice : choices) {
            for (Group group : choice.getGroups()) {
                for (Vote vote : group.getVotes()) {
                    Map<Choice, Double> choicesMap = personVotes.get(vote
                            .getVotingID());
                    if (choicesMap == null) {
                        choicesMap = new HashMap<Choice, Double>();
                    }
                    choicesMap.put(choice, vote.getValue());
                    personVotes.put(vote.getVotingID(), choicesMap);
                }
            }
        }
    }

    /**
     * Initialisation des votes par groupes. Les votes sont initialisés avec les
     * victoires des membres du groupe. Des dépouillements sont donc effectués
     * au sein de chaque groupe avant d'effectuer le dépouillement global.
     *
     * @param choices Les choix utilisés pour le dépouillement.
     */
    private void initGroupVotes(List<Choice> choices) {
        groupVotes = new HashMap<String, Map<Choice, Double>>();

        // Initialisation de la liste des membres de chaque groupe (groupMembers)
        Map<String, Set<String>> groupMembers = new HashMap<String, Set<String>>();
        for (Choice choice : choices) {
            for (Group group : choice.getGroups()) {
                Set<String> members = groupMembers.get(group.getIdGroup());
                if (members == null) {
                    members = new HashSet<String>();
                }
                for (Vote vote : group.getVotes()) {
                    members.add(vote.getVotingID());
                }
                groupMembers.put(group.getIdGroup(), members);
            }
        }

        // Initialisation des votes de chaque groupe (groupVotes)
        for (String groupID : groupMembers.keySet()) {

            // Initialisation des votes par membre à partir des personVotes
            // membersVotes : liste des votes de chaque votant du groupe courant
            Map<String, Map<Choice, Double>> membersVotes = new HashMap<String, Map<Choice, Double>>();
            for (String votingID : groupMembers.get(groupID)) {
                Map<Choice, Double> choicesMap = personVotes.get(votingID);
                membersVotes.put(votingID, choicesMap);
            }

            // Calcul des victoires pour le groupe
            calculateVictories(choices, membersVotes, false);
            groupVotes.put(groupID, victories2votes(victories));

            //TODO joindre ce résultat (Map) au group.value (Double) ???
            TreeSet<Double> treeVictories = new TreeSet<Double>(victories
                    .values());
            for (Choice choice : choices) {
                for (Group group : choice.getGroups()) {
                    if (group.getIdGroup().equals(groupID)) {
                        group.setValue(treeVictories.last());
                    }
                }
            }

            // Affichage des victoires et des groupVotes
            if (log.isDebugEnabled()) {
                log.debug("> groupe " + groupID);
                for (Choice choice : victories.keySet()) {
                    log.debug(">> choix= " + choice.getIdChoice()
                            + " _ victoires= " + victories.get(choice)
                            + " _ classement= "
                            + groupVotes.get(groupID).get(choice));
                }
            }
        }
    }

    /**
     * Calcul des victoires. Parcours des votants (personnes ou groupes) et de
     * leurs votes. Pour chaque vote, calcul des victoires par rapport aux
     * autres votes du votant et mise à jour du nombre de victoires pour ce
     * vote.
     *
     * @see getVictories(String id, Choice choice, Map<String, Map<Choice,
     *      Double>> votes)
     * @param choices Les choix utilisés pour le dépouillement.
     * @param votes Les votes de chaque votant.
     * @param byGroup Calcul sur des groupes.
     */
    private void calculateVictories(List<Choice> choices,
            Map<String, Map<Choice, Double>> votes, boolean byGroup) {
        // Initialisation des victoires à 0.
        victories = new HashMap<Choice, Double>();
        for (Choice choice : choices) {
            victories.put(choice, 0.);
        }

        // Calcul des victoires
        for (String keyVotant : votes.keySet()) {
            for (Choice keyChoice : votes.get(keyVotant).keySet()) {

                // Victoires pour le votant + application du poids du votant
                Double v = getVictories(keyVotant, keyChoice, votes)
                        .doubleValue();
                if (byGroup) {
                    v *= getGroupWeight(choices, keyVotant);
                } else {
                    v *= getPersonWeight(choices, keyVotant);
                }

                victories.put(keyChoice, victories.get(keyChoice) + v);
            }
        }
    }

    /**
     * Calcul du nombre de duels gagnés par un choix par rapport aux autres
     * choix d'un votant (personne ou groupe).
     *
     * @param id L'identifiant du votant
     * @param choice Le choix de référence pour le calcul
     * @param votes Les votes de chaque votant.
     * @return Le nombre de duels gagnés
     */
    private Integer getVictories(String id, Choice choice,
            Map<String, Map<Choice, Double>> votes) {
        int res = 0; // résultat : nombre de victoires

        Double ref = votes.get(id).get(choice); // valeur de référence
        Double val; // valeur courante

        for (Choice keyChoice : victories.keySet()) {
            if (votes.get(id).containsKey(keyChoice)) {
                val = votes.get(id).get(keyChoice);
                if (ref < val) {
                    res++;
                }
            } else { // val == null (choix non renseigné par le votant)
                res++;
            }
        }

        return res;
    }

    /**
     * Transformation des victoires en votes. Les victoires listent le nombre de
     * duels gagnés pour chaque choix alors que les votes donnent leur
     * classement.
     *
     * @param victories Les victoires.
     * @return Les votes.
     */
    private Map<Choice, Double> victories2votes(Map<Choice, Double> victories) {
        Map<Choice, Double> victories_ = new HashMap<Choice, Double>(victories);
        Map<Choice, Double> votes_ = new HashMap<Choice, Double>();

        Integer d = 1;
        Map<Choice, Double> victoriesMax = getMax(victories_);
        while (!victoriesMax.isEmpty()) {
            for (Choice choice : victoriesMax.keySet()) {
                votes_.put(choice, d.doubleValue());
                victories_.remove(choice);
            }
            d++;
            victoriesMax = getMax(victories_);
        }

        return votes_;
    }

    /**
     * Retourne le ou les couples choix-valeur de valeur maxmimale.
     *
     * @param victories Les victoires.
     * @return Le ou les choix de valeur maximale.
     */
    private Map<Choice, Double> getMax(Map<Choice, Double> victories) {
        Map<Choice, Double> result = new HashMap<Choice, Double>();
        if (victories.isEmpty()) {
            return result;
        }

        TreeSet<Double> treeVictories = new TreeSet<Double>(victories.values());
        Double max = treeVictories.last();
        for (Choice choice : victories.keySet()) {
            if (victories.get(choice).equals(max)) {
                result.put(choice, victories.get(choice));
            }
        }
        return result;
    }

    /**
     * Récupération du poids d'une personne.
     *
     * @param choices Les choix utilisés pour le dépouillement.
     * @param id L'identifiant de la personne.
     * @return Le poids de la personne.
     */
    private Double getPersonWeight(List<Choice> choices, String id) {
        for (Choice choice : choices) {
            for (Group group : choice.getGroups()) {
                for (Vote vote : group.getVotes()) {
                    if (id.equals(vote.getVotingID())) {
                        return vote.getWeight();
                    }
                }
            }
        }
        return null;
    }

    /**
     * Récupération du poids d'un groupe.
     *
     * @param choices Les choix utilisés pour le dépouillement.
     * @param id L'identifiant du groupe.
     * @return Le poids du groupe.
     */
    private Double getGroupWeight(List<Choice> choices, String id) {
        for (Choice choice : choices) {
            for (Group group : choice.getGroups()) {
                if (id.equals(group.getIdGroup())) {
                    return group.getWeight();
                }
            }
        }
        return null;
    }

    /**
     * Mise à jour des valeurs des choix.
     *
     * @param choices Les choix utilisés pour le dépouillement.
     */
    private void updateChoices(List<Choice> choices) {
        for (Choice choice : choices) {
            choice.setValue(victories.get(choice));
            /*for (Group group : choice.getGroups()) {
                group.setValue(victories.get(choice));
            }*/
        }
    }

    @Override
    public void executeStats(List<Choice> choices, boolean groupCounting,
            Choice choice, ChoiceDTO choiceDTO) {
        List<PercentageBehavior> list = new ArrayList<PercentageBehavior>(choices);
        choiceDTO.setPercentage(Utils.calculatePercentage(choice, list));
    }
}