/* *##% Pollen
 * Copyright (C) 2009 CodeLutin
 *
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>. ##%*/

package org.chorem.pollen.votecounting.services;

import java.util.ArrayList;
import java.util.List;

import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.chorem.pollen.common.VoteCountingType;
import org.chorem.pollen.votecounting.business.Choice;
import org.chorem.pollen.votecounting.business.CondorcetMethod;
import org.chorem.pollen.votecounting.business.Context;
import org.chorem.pollen.votecounting.business.Method;
import org.chorem.pollen.votecounting.business.PercentageMethod;
import org.chorem.pollen.votecounting.business.StandardMethod;
import org.chorem.pollen.votecounting.dto.ChoiceDTO;
import org.chorem.pollen.votecounting.dto.PollChoiceDTO;
import org.chorem.pollen.votecounting.dto.PollDTO;
import org.chorem.pollen.votecounting.dto.VoteCountingResultDTO;
import org.chorem.pollen.votecounting.dto.VoteToChoiceDTO;
import org.chorem.pollen.votecounting.dto.VotingGroupDTO;
import org.chorem.pollen.votecounting.dto.VotingPersonDTO;
import org.chorem.pollen.votecounting.utils.Utils;

/**
 * Implémentation du service de dépouillement.
 *
 * @author fdesbois
 * @version $Id: ServiceVoteCountingImpl.java 2697 2009-08-11 09:39:09Z nrannou
 *          $
 */
public class ServiceVoteCountingImpl implements ServiceVoteCounting {
    /**
     * Identifiant du groupe courant
     */
    private String currentIdGroup;
    /**
     * Booléen déterminant si le dépouillement se fera sur les groupes
     */
    private boolean isByGroup;
    /**
     * Contexte de dépouillement
     */
    private Context context;
    /** log. */
    private static final Log log = LogFactory
            .getLog(ServiceVoteCountingImpl.class);

    /**
     * Constructeur
     */
    public ServiceVoteCountingImpl() {
        this.currentIdGroup = "";
        this.context = null;
    }

    /**
     * Service de dépouillement par votes (personnes)
     *
     * @param poll : sondage
     * @return resultat
     */
    @Override
    public VoteCountingResultDTO executeVoteCounting(PollDTO poll) {
        this.isByGroup = false;
        return this.execute(poll);
    }

    /**
     * Service de dépouillement par groupes
     *
     * @param poll : sondage
     * @return resultat
     */
    @Override
    public VoteCountingResultDTO executeGroupCounting(PollDTO poll) {
        this.isByGroup = true;
        return this.execute(poll);
    }

    /**
     * Execution du dépouillement
     *
     * @param poll : sondage
     * @return resultat
     */
    private VoteCountingResultDTO execute(PollDTO poll) {
        if (log.isInfoEnabled()) {
            log.info("Dépouillement (byGroup=" + isByGroup + ") du sondage "
                    + poll.getPollId());
        }

        // Création et remplissage du contexte
        this.createContext(poll.getVoteCounting());
        this.fillContext(poll);
        // Execution
        if (!this.context.execute()) {
            return null;
        }

        // Transformation des résultats
        List<ChoiceDTO> resChoices = new ArrayList<ChoiceDTO>();
        List choices = this.context.getChoices();

        for (Choice choice : this.context.getChoices()) {
            ChoiceDTO resChoice = new ChoiceDTO();
            resChoice.setIdChoice(choice.getIdChoice());
            resChoice.setNbVotes(this.calculateNbVotes(choice.getGroups(),
                    this.isByGroup));
            resChoice.setValue(choice.getValue());
            resChoice.setPercentage(Utils.calculatePercentage(choice, choices));
            resChoice.setResult(this.isChoiceResult(choice));
            if (log.isDebugEnabled()) {
                log.debug(choice + " _ result ? " + resChoice.isResult());
            }
            resChoices.add(resChoice);
        }

        VoteCountingResultDTO result = new VoteCountingResultDTO();
        result.setNbVotes(this.calculateNbVotes(poll.getVotingGroups(),
                this.isByGroup));
        result.setTypeVoteCounting(poll.getVoteCounting());
        result.setByGroup(this.isByGroup);
        result.setIdPoll(poll.getPollId());
        result.setChoices(resChoices);
        return result;
    }

    /**
     * Création du contexte en fonction du type de dépouillement
     *
     * @param type : type de dépouillement
     */
    private void createContext(VoteCountingType type) {
        Method method = null;
        switch (type) {
        case NORMAL:
            method = new StandardMethod();
            break;
        case PERCENTAGE:
            method = new PercentageMethod();
            break;
        case CONDORCET:
            method = new CondorcetMethod();
            break;
        default:
            method = new StandardMethod();
        }
        this.context = new Context(method, this.isByGroup);
    }

    /**
     * Remplissage du contexte
     *
     * @param poll : Sondage
     */
    private void fillContext(PollDTO poll) {
        if (log.isDebugEnabled()) {
            log.debug("Ajout poll : " + poll.getPollId());
        }
        for (Object o : poll.getChoices()) {
            PollChoiceDTO choice = (PollChoiceDTO) o;
            this.context.addChoice(choice.getIdChoice());
        }
        for (Object o : poll.getVotingGroups()) {
            VotingGroupDTO group = (VotingGroupDTO) o;
            this.routeGroup(group);
        }
    }

    /**
     * Parcours d'un groupe et de ses votants
     *
     * @param group : groupe lié au sondage
     */
    private void routeGroup(VotingGroupDTO group) {
        if (log.isDebugEnabled()) {
            log.debug("Ajout group : " + group.getIdGroup() + " _ weight="
                    + group.getWeight());
        }
        this.context.addGroup(group.getIdGroup(), group.getWeight());
        for (Object o : group.getVotingPersons()) {
            VotingPersonDTO person = (VotingPersonDTO) o;
            this.currentIdGroup = group.getIdGroup();
            this.routePerson(person);
        }
    }

    /**
     * Parcours d'un votant et de ses choix
     *
     * @param person : personne ayant voté
     */
    private void routePerson(VotingPersonDTO person) {
        if (log.isDebugEnabled()) {
            log.debug("Ajout person : " + person.getVotingId() + " _ weight="
                    + person.getWeight());
        }
        for (Object o : person.getChoices()) {
            VoteToChoiceDTO vote = (VoteToChoiceDTO) o;
            this.addVoteToContext(vote, person.getWeight(), person
                    .getVotingId());
        }
    }

    /**
     * Ajout du vote d'une personne au contexte pour un choix
     *
     * @param vote : vote lié au choix
     * @param weight : poids de la personne
     */
    private void addVoteToContext(VoteToChoiceDTO vote, double weight,
            String votingID) {
        if (log.isDebugEnabled()) {
            log.debug("Ajout vote : " + vote.getValue() + " _ choice="
                    + vote.getIdChoice());
        }
        this.context.getChoice(vote.getIdChoice())
                .getGroup(this.currentIdGroup).addVote(vote.getValue(), weight,
                        votingID);
    }

    /**
     * Calcul le nombre de votes d'une liste
     *
     * @param list : liste d'éléments (doivent impléméntés ObjectWithList)
     * @param byGroup : condition sur le calcul
     * @return nombre de votes
     */
    private int calculateNbVotes(List list, boolean byGroup) {
        return Utils.mayCalculateSubElements(list, byGroup);
    }

    /**
     * Test si le choix est un résultat d'après le contexte A utiliser après
     * dépouillement sinon aucun résultat
     *
     * @param choice : choix à tester
     * @return true si le choix est un résultat, false sinon
     */
    private boolean isChoiceResult(Choice choice) {
        for (Choice res : this.context.getResults()) {
            if (choice.getIdChoice().equals(res.getIdChoice())) {
                return true;
            }
        }
        return false;
    }

}