/* *##% Pollen
 * Copyright (C) 2009 CodeLutin
 *
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>. ##%*/

package org.chorem.pollen.ui.base;

import java.io.File;
import java.io.FileInputStream;
import java.io.IOException;
import java.io.InputStream;
import java.util.List;

import org.apache.tapestry5.BindingConstants;
import org.apache.tapestry5.ComponentResources;
import org.apache.tapestry5.StreamResponse;
import org.apache.tapestry5.annotations.InjectPage;
import org.apache.tapestry5.annotations.Parameter;
import org.apache.tapestry5.annotations.Property;
import org.apache.tapestry5.annotations.Retain;
import org.apache.tapestry5.annotations.SessionState;
import org.apache.tapestry5.beaneditor.BeanModel;
import org.apache.tapestry5.ioc.annotations.Inject;
import org.apache.tapestry5.services.BeanModelSource;
import org.apache.tapestry5.services.Response;
import org.chorem.pollen.business.dto.PollAccountDTO;
import org.chorem.pollen.business.dto.PollDTO;
import org.chorem.pollen.business.dto.UserDTO;
import org.chorem.pollen.business.services.ServicePoll;
import org.chorem.pollen.business.services.ServicePollAccount;
import org.chorem.pollen.business.services.ServiceResults;
import org.chorem.pollen.ui.data.EvenOdd;
import org.chorem.pollen.ui.data.Lien;
import org.chorem.pollen.ui.data.PollAction;
import org.chorem.pollen.ui.pages.admin.PollsAdmin;
import org.chorem.pollen.ui.pages.poll.ConfirmPoll;
import org.chorem.pollen.ui.pages.user.UserPollsCreated;

/**
 * Classe de la page d'affichage des sondages.
 *
 * @author kmorin
 * @author rannou
 * @version $Id: Polls.java 2699 2009-08-11 14:22:29Z nrannou $
 */
public class Polls {

    /**
     * Objet de session représentant l'utilisateur identifié.
     */
    @SuppressWarnings("unused")
    @SessionState
    @Property
    private UserDTO user;
    @SuppressWarnings("unused")
    @Property
    private boolean userExists;

    /**
     * Liste de sondages de l'utilisateur
     */
    @SuppressWarnings("unused")
    private List<PollDTO> polls;

    /**
     * Sondage servant à l'itération des sondages de la liste
     */
    @Property
    private PollDTO poll;

    /**
     * Objet servant à changer la couleur à chaque ligne de la liste des
     * sondages
     */
    @SuppressWarnings("unused")
    @Property
    private EvenOdd evenOdd;

    @Parameter(defaultPrefix = BindingConstants.MESSAGE, value = "title")
    @Property
    private String title;

    @SuppressWarnings("unused")
    @Property
    private Lien[] address;

    /**
     * Sert à créer pollsModel
     */
    @Inject
    private BeanModelSource beanModelSource;

    /**
     * Sert à passer les messages en paramètre de la création de pollsModel
     */
    @Inject
    private ComponentResources componentResources;

    /**
     * Modèle pour l'affichage de la liste des sondages
     */
    @SuppressWarnings("unchecked")
    @Property
    @Retain
    private BeanModel pollsModel;

    /** Page de validation de la fermeture d'un sondage */
    @InjectPage
    private ConfirmPoll confirmPoll;

    /** Injection des services */
    @Inject
    private ServicePoll servicePoll;
    @Inject
    private ServicePollAccount servicePollAccount;
    @Inject
    private ServiceResults serviceResults;

    /**
     * Récupération de l'identifiant de modification du sondage.
     *
     * @return l'identifiant.
     */
    public String getAdminId() {
        PollAccountDTO creator = servicePollAccount.findPollAccountById(poll
                .getCreatorId());
        return poll.getPollId() + ":" + creator.getAccountId();
    }

    /**
     * Fermeture du sondage.
     *
     * @return la page de validation
     */
    public Object onActionFromClose(String id) {
        return prepareConfirmPage(id, PollAction.CLOSE);
    }

    /**
     * Suppression du sondage.
     *
     * @return la page des sondages
     */
    public Object onActionFromDeletePoll(String id) {
        return prepareConfirmPage(id, PollAction.DELETE);
    }

    /**
     * Préparation de la page de confirmation.
     *
     * @return la page de confirmation
     */
    private Object prepareConfirmPage(String pollId, PollAction pollAction) {
        String backPage = "";
        if (this instanceof UserPollsCreated) {
            backPage = "user/PollsCreated";
        } else if (this instanceof PollsAdmin) {
            backPage = "admin/PollsAdmin";
        }
        confirmPoll.setBackPage(backPage);
        confirmPoll.setAction(pollAction);
        confirmPoll.setPoll(servicePoll.findPollByPollId(pollId));
        return confirmPoll;
    }

    /**
     * Export XML du sondage.
     *
     * @return le flux correspondant au fichier
     */
    public Object onActionFromExport(String id) {
        final String filename = serviceResults.exportPoll(id);

        // Retour d'un StreamResponse contenant le fichier XML
        return new StreamResponse() {
            public String getContentType() {
                return "text/xml";
            }

            public InputStream getStream() throws IOException {
                return new FileInputStream(new File(filename));
            }

            public void prepareResponse(Response response) {
            }
        };
    }

    /**
     * Méthode d'initialisation des sondages
     */
    public void onActivate() {
        evenOdd = new EvenOdd();
    }

    /**
     * Initialisation de l'affichage
     */
    public void setupRender() {
        address = new Lien[] { new Lien("Pollen", "Index"),
                new Lien(title, null) };

        pollsModel = beanModelSource.createDisplayModel(PollDTO.class,
                componentResources.getMessages());
        pollsModel.add("functions", null);
        pollsModel.include("title", "description", "beginChoiceDate",
                "beginDate", "endDate", "functions");
    }

}
