/* *##% Pollen
 * Copyright (C) 2009 CodeLutin
 *
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>. ##%*/

package org.chorem.pollen.ui.components;

import java.awt.Color;
import java.awt.Font;
import java.awt.image.BufferedImage;
import java.io.ByteArrayInputStream;
import java.io.ByteArrayOutputStream;
import java.io.IOException;
import java.io.InputStream;
import java.util.List;

import org.apache.tapestry5.ComponentResources;
import org.apache.tapestry5.Link;
import org.apache.tapestry5.MarkupWriter;
import org.apache.tapestry5.StreamResponse;
import org.apache.tapestry5.annotations.Parameter;
import org.apache.tapestry5.dom.Element;
import org.apache.tapestry5.ioc.annotations.Inject;
import org.apache.tapestry5.services.Response;
import org.jfree.chart.ChartUtilities;
import org.jfree.chart.JFreeChart;
import org.jfree.chart.labels.StandardPieSectionLabelGenerator;
import org.jfree.chart.plot.PiePlot;
import org.jfree.chart.plot.PiePlot3D;
import org.jfree.chart.plot.RingPlot;
import org.jfree.chart.title.TextTitle;
import org.jfree.data.DefaultKeyedValues;
import org.jfree.data.general.DefaultPieDataset;
import org.jfree.data.general.PieDataset;
import org.jfree.util.SortOrder;
import org.nuiton.util.ArrayUtil;

/**
 * Composant qui affiche un graphe de type PieChart.
 *
 * Utilisation : <t:chart width="210" height="190" values="choice" title="title"
 * type="1"/>
 *
 * choice est une liste de chaînes représentant les labels et leur valeurs
 * respectives. ({"val1", "20", "val2", "10"}...) type est le type de
 * représentation du diagramme (1:PIE, 2:PIE3D...)
 *
 * Pour généraliser à d'autres types de graphes il faudrait passer un JFreeChart
 * :
 *
 * Persist Parameter(required=true) private JFreeChart _chart;
 *
 * @author rannou
 * @version $Id: Chart.java 2615 2009-07-01 13:47:45Z nrannou $
 */
public class Chart {

    /** diagramme de type PIE */
    public static final int PIE = 1;
    /** diagramme de type PIE3D */
    public static final int PIE3D = 2;
    /** diagramme de type RING */
    public static final int RING = 3;

    /** titre du diagramme */
    @Parameter
    private String _title = "";

    /** type de représentation du diagramme */
    @Parameter(required = true)
    private int _type;

    @Parameter(required = true)
    private int _width;

    @Parameter(required = true)
    private int _height;

    /**
     * list(array) of paired values(label,value):
     * [String,Number,String,Number,...]
     */
    @Parameter(required = true)
    private List<Object> _values;

    @Inject
    private ComponentResources _resources;

    void beginRender(MarkupWriter writer) {

        // event link params
        Object[] params = new Object[] { _title, _type, _width, _height };
        params = ArrayUtil.concat(params, _values.toArray());

        // generate event link
        Link link = _resources.createEventLink("chart", params);
        Element img = writer.element("img", "src", link);

        _resources.renderInformalParameters(writer);
    }

    void afterRender(MarkupWriter writer) {
        writer.end();
    }

    public StreamResponse onChart(final String title, final int type,
            final int width, final int height, Object... rest) {
        DefaultKeyedValues values = new DefaultKeyedValues();
        for (int i = 4; i < rest.length; i += 2) {
            values.addValue(rest[i].toString(), Double.valueOf(rest[i + 1]
                    .toString()));
        }
        values.sortByValues(SortOrder.DESCENDING);
        PieDataset data = new DefaultPieDataset(values);

        // create the chart look according to the type
        PiePlot plot = null;
        switch (type) {
        case PIE:
            plot = new PiePlot(data);
            break;
        case PIE3D:
            plot = new PiePlot3D(data);
            break;
        case RING:
            plot = new RingPlot(data);
            break;
        default:
            plot = new PiePlot(data);
            break;
        }

        //plot.setForegroundAlpha(0.7f);
        plot.setCircular(true);
        plot.setLabelGenerator(new StandardPieSectionLabelGenerator("{0} {2}"));
        plot.setLegendLabelGenerator(new StandardPieSectionLabelGenerator(
                "{0} = {1}"));
        plot.setNoDataMessage("No Data");

        final JFreeChart chart = new JFreeChart(plot);
        chart.setTitle(new TextTitle(title, new Font(Font.SANS_SERIF,
                Font.BOLD, 12)));
        chart.setBackgroundPaint(new Color(255, 255, 255));

        // return the image
        return new StreamResponse() {
            public String getContentType() {
                return "image/png";
            }

            public InputStream getStream() throws IOException {
                BufferedImage image = chart.createBufferedImage(width, height);
                ByteArrayOutputStream byteArray = new ByteArrayOutputStream();
                ChartUtilities.writeBufferedImageAsPNG(byteArray, image);
                return new ByteArrayInputStream(byteArray.toByteArray());
            }

            public void prepareResponse(Response response) {
            }
        };
    }
}