/*
 * #%L
 * Pollen :: UI (struts2)
 * $Id: ManageFavoriteLists.java 3522 2012-06-19 10:18:41Z tchemit $
 * $HeadURL: http://svn.chorem.org/svn/pollen/tags/pollen-1.4/pollen-ui-struts2/src/main/java/org/chorem/pollen/ui/actions/user/ManageFavoriteLists.java $
 * %%
 * Copyright (C) 2009 - 2012 CodeLutin, Tony Chemit
 * %%
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Affero General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU Affero General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 * #L%
 */
package org.chorem.pollen.ui.actions.user;

import com.google.common.base.Preconditions;
import org.apache.commons.lang3.StringUtils;
import org.chorem.pollen.business.persistence.PersonList;
import org.chorem.pollen.business.persistence.PollAccount;
import org.chorem.pollen.services.FavoriteListImport;
import org.chorem.pollen.services.exceptions.FavoriteListAlreadyExistException;
import org.chorem.pollen.services.exceptions.FavoriteListImportException;
import org.chorem.pollen.services.exceptions.FavoriteListNotFoundException;
import org.chorem.pollen.services.exceptions.ParticipantAlreadyFoundInListException;
import org.chorem.pollen.services.impl.FavoriteListImportCSV;
import org.chorem.pollen.services.impl.FavoriteListImportLDAP;
import org.chorem.pollen.ui.actions.PageSkin;
import org.chorem.pollen.ui.actions.PollenActionSupport;

import java.io.File;
import java.util.List;

/**
 * Manage favorite lists of a connected user.
 *
 * @author tchemit <chemit@codelutin.com>
 * @since 1.3
 */
public class ManageFavoriteLists extends PollenActionSupport {

    private static final long serialVersionUID = 1L;

    protected PersonList favoriteList;

    protected File csvImport;

    protected String csvImportContentType;

    protected String csvImportFileName;

    protected String ldapImport;

    protected String action;

    public void setCsvImport(File csvImport) {
        this.csvImport = csvImport;
    }

    public void setCsvImportContentType(String csvImportContentType) {
        this.csvImportContentType = csvImportContentType;
    }

    public void setCsvImportFileName(String csvImportFileName) {
        this.csvImportFileName = csvImportFileName;
    }

    public String getLdapImport() {
        return ldapImport;
    }

    public void setLdapImport(String ldapImport) {
        this.ldapImport = ldapImport;
    }

    public PersonList getCreateFavoriteList() {
        return getFavoriteList();
    }

    public PersonList getDeleteFavoriteList() {
        return getFavoriteList();
    }

    public String getAction() {
        return action;
    }

    public void setAction(String action) {
        this.action = action;
    }

    @Override
    public PageSkin getSkin() {
        return PageSkin.EDITION;
    }

    @Override
    public void validate() {

        if ("create".equals(action)) {

            if( StringUtils.isBlank(getCreateFavoriteList().getName())) {
                addFieldError("createFavoriteList.name",
                              _("pollen.error.favoriteListName.required"));
            }
        }
    }

    public String create() throws Exception {

        Preconditions.checkNotNull(favoriteList);
        Preconditions.checkNotNull(favoriteList.getName());

        try {
            PersonList personList =
                    getFavoriteService().createFavoriteList(getPollenUserAccount(),
                                                            favoriteList.getName());

            if (csvImportFileName != null) {

                FavoriteListImport importService = newService(FavoriteListImportCSV.class);

                addImport(importService, csvImport.getAbsolutePath(), personList);
            }

            if (StringUtils.isNotBlank(ldapImport)) {

                FavoriteListImport importService = newService(FavoriteListImportLDAP.class);

                addImport(importService, ldapImport, personList);
            }

        } catch (FavoriteListAlreadyExistException ex) {
            addFieldError("createFavoriteList.name",
                          _("pollen.error.favoriteList.already.used"));
        }

        String result;
        if (hasAnyErrors()) {
            result = INPUT;

        } else {
            getTransaction().commitTransaction();

            addFlashMessage(_("pollen.information.favoriteList.created",
                              favoriteList.getName()));
            action = null;
            favoriteList = null;
            result = SUCCESS;
        }
        return result;
    }

    public String delete() throws Exception {

        Preconditions.checkNotNull(favoriteList);

        PersonList deletedFavoritedList =
                getFavoriteService().deleteFavoriteList(getPollenUserAccount(),
                                                        favoriteList);
        getTransaction().commitTransaction();

        addFlashMessage(_("pollen.information.favoriteList.deleted",
                          deletedFavoritedList.getName()));
        action = null;
        favoriteList = null;
        return SUCCESS;
    }

    protected PersonList getFavoriteList() {
        if (favoriteList == null) {
            favoriteList = getFavoriteService().newFavoriteList();
        }
        return favoriteList;
    }

    protected void addImport(FavoriteListImport service, String url, PersonList list)
            throws FavoriteListNotFoundException {

        List<PollAccount> importedAccounts;
        try {
            importedAccounts = service.execute(url);

            for (PollAccount importedAccount : importedAccounts) {
                try {
                    getFavoriteService().addPollAccountToFavoriteList(list, importedAccount);

                } catch (ParticipantAlreadyFoundInListException ex) {
                    // WARNING ?
                    addFlashError(
                            _("pollen.error.favoriteList.import.participantExists"
                                    , importedAccount.getEmail())
                    );
                }
            }

        } catch (FavoriteListImportException ex) {
            String message = ex.getLocalizedMessage(getLocale());
            addFlashError(message);
        }
    }
}
