/*
 * #%L
 * Pollen :: UI (strust2)
 * 
 * $Id: PollenActionSupport.java 3376 2012-05-18 14:50:18Z tchemit $
 * $HeadURL: http://svn.chorem.org/svn/pollen/tags/pollen-1.3.1/pollen-ui-struts2/src/main/java/org/chorem/pollen/ui/actions/PollenActionSupport.java $
 * %%
 * Copyright (C) 2009 - 2012 CodeLutin, Tony Chemit
 * %%
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Affero General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU Affero General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 * #L%
 */
package org.chorem.pollen.ui.actions;

import com.google.common.collect.Lists;
import com.opensymphony.xwork2.ActionContext;
import org.apache.commons.collections.CollectionUtils;
import org.apache.struts2.ServletActionContext;
import org.apache.struts2.StrutsStatics;
import org.chorem.pollen.PollenConfiguration;
import org.chorem.pollen.business.persistence.UserAccount;
import org.chorem.pollen.common.VoteCountingType;
import org.chorem.pollen.services.DefaultPollenServiceContext;
import org.chorem.pollen.services.PollenService;
import org.chorem.pollen.services.PollenServiceContext;
import org.chorem.pollen.services.PollenServiceFactory;
import org.chorem.pollen.ui.PollenApplicationContext;
import org.chorem.pollen.ui.PollenSession;
import org.nuiton.topia.TopiaContext;
import org.nuiton.topia.framework.TopiaTransactionAware;
import org.nuiton.web.filter.TopiaTransactionFilter;
import org.nuiton.web.struts2.BaseAction;

import javax.servlet.http.HttpServletRequest;
import java.net.URL;
import java.util.Collection;
import java.util.Date;
import java.util.List;

/**
 * Pollen action support.
 * <p/>
 * <b>Note :</b> All actions must implement this class to have a better i18n
 * support, when a i18n key is not translated it has a empty translation and
 * we do NOT want this behaviour in gui, prefer to return the marked
 * untranslated key.
 *
 * @author tchemit <chemit@codelutin.com>
 * @since 1.3
 */
public class PollenActionSupport extends BaseAction implements TopiaTransactionAware {

    private static final long serialVersionUID = 1L;

    public static PollenApplicationContext getPollenApplicationContext() {
        PollenApplicationContext applicationContext =
                PollenApplicationContext.get(getActionContext());
        return applicationContext;
    }

    public PollenConfiguration getConfiguration() {
        return getPollenApplicationContext().getConfiguration();
    }

    /** Pollen User getPollenSession(). */
    private transient PollenSession pollenSession;

    /**
     * Provides a way to get a service.
     * <p/>
     * Actions may <strong>not</strong> call it directly by use
     * {@link #newService(Class)} instead.
     */
    protected transient PollenServiceFactory serviceFactory;

    protected transient PollenServiceContext serviceContext;

    protected transient TopiaContext transaction;

    public PageSkin getSkin() {
        return PageSkin.INDEX;
    }

    public String getPageLogo() {
        return getSkin().getCssSuffix();
    }

    public String getJqueryTheme() {
        return getSkin().getTheme();
    }

    public PollenSession getPollenSession() {
        if (pollenSession == null) {
            pollenSession = PollenSession.get(getActionContext());
        }
        return pollenSession;
    }

    public UserAccount getPollenUserAccount() {
        return getPollenSession().getUserAccount();
    }

    public boolean isUserLoggued() {
        UserAccount userAccount = getPollenUserAccount();
        return userAccount != null;
    }

    public boolean isUserAdmin() {
        UserAccount userAccount = getPollenUserAccount();
        return userAccount != null && userAccount.isAdministrator();
    }

    public static String getApplicationVersion() {
        return getPollenApplicationContext().getConfiguration().getVersion().toString();
    }

    public URL getApplicationUrl() {
        URL url = getServiceContext().getApplicationURL();
        return url;
    }

    /**
     * Fabrique pour récupérer le ServiceContext tel qu'il devrait être fourni
     * à la fabrication d'un service.
     *
     * @return service context
     */
    protected PollenServiceContext getServiceContext() {
        if (serviceContext == null) {
            serviceContext = DefaultPollenServiceContext.newContext(
                    getLocale(),
                    getTransaction(),
                    getConfiguration(),
                    getServiceFactory()
            );
        }
        return serviceContext;
    }

    public PollenServiceFactory getServiceFactory() {
        if (serviceFactory == null) {
            serviceFactory = new PollenServiceFactory();
        }
        return serviceFactory;
    }

    /**
     * Sub-classes should use this method to easily get a service instance.
     *
     * @param serviceClass The type of service to instantiate
     * @return A newly created service of the expected type with necessary data set
     */
    public <E extends PollenService> E newService(Class<E> serviceClass) {
        E service = getServiceFactory().newService(serviceClass,
                                                   getServiceContext());
        return service;
    }

    @Override
    public TopiaContext getTransaction() {
        if (transaction == null) {
            HttpServletRequest request = (HttpServletRequest)
                    getActionContext().get(StrutsStatics.HTTP_REQUEST);
            transaction = TopiaTransactionFilter.getTransaction(request);
        }
        return transaction;
    }

    @Override
    public void setTransaction(TopiaContext transaction) {
        this.transaction = transaction;
    }

    public static final List<String> SAFE_LANGUAGES =
            Lists.newArrayList("en", "fr");

    public String getSafeLanguage() {
        String language = getLocale().getLanguage();
        if (!SAFE_LANGUAGES.contains(language)) {
            language = SAFE_LANGUAGES.get(0);
        }
        return language;
    }

    protected static ActionContext getActionContext() {
        return ActionContext.getContext();
    }

    public static Date getCurrentDateTime() {
        return new Date();
    }

    public String getVoteCountingTypeHelp(VoteCountingType voteCountingType) {
        String result;
        switch (voteCountingType) {

            default:
            case NORMAL:
                result =
                        _("pollen.common.voteCountingTypeHelp.normal");
                break;
            case PERCENTAGE:
                result =
                        _("pollen.common.voteCountingTypeHelp.percentage");
                break;
            case CONDORCET:
                result =
                        _("pollen.common.voteCountingTypeHelp.condorcet");
                break;
            case NUMBER:
                result =
                        _("pollen.common.voteCountingTypeHelp.number");
        }
        return result;
    }

    public String getCurrentUrl() {
        HttpServletRequest request = ServletActionContext.getRequest();
        String result = request.getRequestURL().toString();
        return result;
    }

    public void addFlashMessage(String message) {
        getPollenSession().addMessage(PollenSession.SESSION_TOKEN_MESSAGES, message);
    }

    public void addFlashError(String message) {
        getPollenSession().addMessage(PollenSession.SESSION_TOKEN_ERRORS, message);
    }

    public void addFlashWarning(String message) {
        getPollenSession().addMessage(PollenSession.SESSION_TOKEN_WARNINGS, message);
    }

    public Collection<String> getFlashMessages() {

        List<String> result = getPollenSession().consumeDynamicData(PollenSession.SESSION_TOKEN_MESSAGES);
//        if (result == null) {
//            result = Collections.emptyList();
//        } else {
//            getPollenSession().removeDynamicData(PollenSession.SESSION_TOKEN_MESSAGES);
//        }
        return result;
    }

    public Collection<String> getFlashErrors() {
        List<String> result = getPollenSession().consumeDynamicData(PollenSession.SESSION_TOKEN_ERRORS);
//        if (result == null) {
//            result = Collections.emptyList();
//        }  else {
//            getPollenSession().removeDynamicData(PollenSession.SESSION_TOKEN_ERRORS);
//        }
        return result;
    }

    public Collection<String> getFlashWarnings() {
        List<String> result = getPollenSession().consumeDynamicData(PollenSession.SESSION_TOKEN_WARNINGS);
//        if (result == null) {
//            result = Collections.emptyList();
//        }  else {
//            getPollenSession().removeDynamicData(PollenSession.SESSION_TOKEN_ERRORS);
//        }
        return result;
    }

    public boolean hasFlashMessages() {
        List<String> result = getPollenSession().getDynamicData(PollenSession.SESSION_TOKEN_MESSAGES);
        return CollectionUtils.isNotEmpty(result);
    }

    public boolean hasFlashErrors() {
        List<String> result = getPollenSession().getDynamicData(PollenSession.SESSION_TOKEN_ERRORS);
        return CollectionUtils.isNotEmpty(result);
    }

    public boolean hasFlashWarnings() {
        List<String> result = getPollenSession().getDynamicData(PollenSession.SESSION_TOKEN_WARNINGS);
        return CollectionUtils.isNotEmpty(result);
    }

    public void clearFlashMessages() {
        List<String> result = getPollenSession().getDynamicData(PollenSession.SESSION_TOKEN_MESSAGES);
        if (result != null) {
            result.clear();
        }
    }

    public void clearFlashErrors() {
        List<String> result = getPollenSession().getDynamicData(PollenSession.SESSION_TOKEN_ERRORS);
        if (result != null) {
            result.clear();
        }
    }

    public boolean hasAnyErrors() {
        boolean result = super.hasErrors();
        return result || hasFlashErrors();
    }
}
