/*
 * #%L
 * IsisFish
 * 
 * $Id$
 * $HeadURL$
 * %%
 * Copyright (C) 2009 - 2010 Ifremer, CodeLutin
 * %%
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as
 * published by the Free Software Foundation, either version 2 of the 
 * License, or (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public 
 * License along with this program.  If not, see
 * <http://www.gnu.org/licenses/gpl-2.0.html>.
 * #L%
 */

package fr.ifremer.isisfish.ui.input.model;

import static org.nuiton.i18n.I18n._;

import java.awt.Component;
import java.util.List;

import javax.swing.JButton;
import javax.swing.JCheckBox;
import javax.swing.JLabel;
import javax.swing.JTable;
import javax.swing.table.AbstractTableModel;
import javax.swing.table.TableCellRenderer;

import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;

import fr.ifremer.isisfish.entities.Equation;
import fr.ifremer.isisfish.entities.MetierSeasonInfo;
import fr.ifremer.isisfish.entities.TargetSpecies;
import fr.ifremer.isisfish.ui.sensitivity.SensitivityTableModel;

/**
 * Table model for {@link MetierSeasonInfo}#{@link TargetSpecies}.
 * 
 * Columns :
 * <li>target species name</li>
 * <li>target species equation</li>
 * <li>target species primaryCatch</li>
 *
 * @author chatellier
 * @version $Revision: 3276 $
 * 
 * Last update : $Date: 2011-04-20 17:36:48 +0200 (mer., 20 avril 2011) $
 * By : $Author: chatellier $
 */
public class MetierSeasonInfoTargetSpeciesTableModel extends AbstractTableModel implements TableCellRenderer, SensitivityTableModel {

    /** Log. */
    private static Log log = LogFactory.getLog(MetierSeasonInfoTargetSpeciesTableModel.class);
    
    /** serialVersionUID. */
    private static final long serialVersionUID = 3169786638868209920L;

    /** Columns names. */
    public final static String[] COLUMN_NAMES = {
            _("isisfish.metierSeasonInfoSpecies.species"),
            _("isisfish.metierSeasonInfoSpecies.targetFactor"),
            _("isisfish.metierSeasonInfoSpecies.mainSpecies") };

    protected List<TargetSpecies> targetSpeciesList;

    /**
     * Empty constructor.
     */
    public MetierSeasonInfoTargetSpeciesTableModel() {
        this(null);
    }

    /**
     * Constructor with data.
     *  
     * @param targetSpeciesList initial target species
     */
    public MetierSeasonInfoTargetSpeciesTableModel(
            List<TargetSpecies> targetSpeciesList) {
        this.targetSpeciesList = targetSpeciesList;
    }

    /**
     * Set target species list.
     * 
     * @param targetSpeciesList the targetSpecies to set
     */
    public void setTargetSpecies(List<TargetSpecies> targetSpeciesList) {
        this.targetSpeciesList = targetSpeciesList;
    }

    /*
     * @see javax.swing.table.TableModel#getColumnCount()
     */
    @Override
    public int getColumnCount() {
        return COLUMN_NAMES.length;
    }

    /*
     * @see javax.swing.table.TableModel#getRowCount()
     */
    @Override
    public int getRowCount() {
        int rows = 0;
        if (targetSpeciesList != null) {
            rows = targetSpeciesList.size();
        }
        return rows;
    }

    /*
     * @see javax.swing.table.TableModel#getValueAt(int, int)
     */
    @Override
    public Object getValueAt(int rowIndex, int columnIndex) {

        Object result = null;

        TargetSpecies targetSpecies = targetSpeciesList.get(rowIndex);
        switch (columnIndex) {
        case 0:
            result = targetSpecies.getSpecies().getName();
            break;
        case 1:
            result = targetSpecies.getTargetFactorEquation();
            break;
        case 2:
            result = targetSpecies.getPrimaryCatch();
            break;
        default:
            throw new IndexOutOfBoundsException("No such column " + columnIndex);
        }

        return result;
    }

    /*
     * @see javax.swing.table.TableModel#getColumnClass(int)
     */
    @Override
    public Class<?> getColumnClass(int columnIndex) {

        Class<?> result = null;
        
        switch (columnIndex) {
        case 0:
            result = String.class;
            break;
        case 1:
            result = Equation.class;
            break;
        case 2:
            result = Boolean.class;
            break;
        default:
            throw new IndexOutOfBoundsException("No such column " + columnIndex);
        }
        
        return result;
    }

    /*
     * @see javax.swing.table.TableModel#getColumnName(int)
     */
    @Override
    public String getColumnName(int columnIndex) {
        return COLUMN_NAMES[columnIndex];
    }

    /*
     * @see javax.swing.table.TableModel#isCellEditable(int, int)
     */
    @Override
    public boolean isCellEditable(int rowIndex, int columnIndex) {
        return columnIndex > 0;
    }

    /*
     * @see javax.swing.table.TableModel#setValueAt(java.lang.Object, int, int)
     */
    @Override
    public void setValueAt(Object value, int rowIndex, int columnIndex) {
        
        if (log.isDebugEnabled()) {
            log.debug("Cell edition (column " + columnIndex + ") = " + value);
        }
        
        TargetSpecies targetSpecies = targetSpeciesList.get(rowIndex);
        switch (columnIndex) {
        case 1:
            Equation eq = (Equation)value;
            // two events for event to be fired
            targetSpecies.setTargetFactorEquation(null);
            targetSpecies.setTargetFactorEquation(eq);
            break;
        case 2:
            Boolean bValue = (Boolean)value;
            targetSpecies.setPrimaryCatch(bValue);
            break;
        default:
            throw new IndexOutOfBoundsException("Can't edit column " + columnIndex);
        }

    }

    /*
     * @see javax.swing.table.TableCellRenderer#getTableCellRendererComponent(javax.swing.JTable, java.lang.Object, boolean, boolean, int, int)
     */
    @Override
    public Component getTableCellRendererComponent(JTable table, Object value,
            boolean isSelected, boolean hasFocus, int row, int column) {

        Component c = null;
        switch (column) {
        case 0:
            c = new JLabel(value.toString());
            break;
        case 1:
            Equation equation = (Equation)value;
            c = new JButton(equation.getName() + "(" + equation.getCategory() + ")");
            break;
        case 2:
            Boolean bValue = (Boolean)value;
            c = new JCheckBox();
            ((JCheckBox)c).setSelected(bValue);
            break;
        default:
            throw new IndexOutOfBoundsException("No such column " + column);
        }
        return c;
    }

    /*
     * @see fr.ifremer.isisfish.ui.sensitivity.SensitivityTableModel#getPropertyAtColumn(int)
     */
    @Override
    public String getPropertyAtColumn(int column) {
        String property = null;
        if (column == 1) {
            property = "targetFactorEquation";
        }
        return property;
    }

    /*
     * @see fr.ifremer.isisfish.ui.sensitivity.SensitivityTableModel#getBeanAtRow(int)
     */
    @Override
    public Object getBeanAtRow(int rowIndex) {
        Object value = null;
        value = targetSpeciesList.get(rowIndex);
        return value;
    }
}
