/*
 * #%L
 * IsisFish
 * 
 * $Id: SimulationInformation.java 3460 2011-10-06 20:08:09Z chatellier $
 * $HeadURL$
 * %%
 * Copyright (C) 2007 - 2011 Ifremer, CodeLutin, Chatellier Eric
 * %%
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as
 * published by the Free Software Foundation, either version 2 of the 
 * License, or (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public 
 * License along with this program.  If not, see
 * <http://www.gnu.org/licenses/gpl-2.0.html>.
 * #L%
 */

package fr.ifremer.isisfish.datastore;

import static org.nuiton.i18n.I18n._;

import java.io.File;
import java.io.FileReader;
import java.io.FileWriter;
import java.io.IOException;
import java.io.PrintWriter;
import java.io.StringWriter;
import java.text.ParseException;
import java.text.SimpleDateFormat;
import java.util.Date;
import java.util.HashMap;
import java.util.HashSet;
import java.util.Map;
import java.util.Properties;
import java.util.Set;

import org.apache.commons.io.IOUtils;
import org.apache.commons.lang3.time.DurationFormatUtils;
import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;

/**
 * Cette classe permet de conserver des informations sur le deroulement d'une
 * simulation. La plupart des informations sont automatiquement renseignees,
 * mais l'utilisateur peut lui aussi ajouter des informations avec la methode
 * {@link #addInformation(String)}.
 * 
 * @author poussin
 * @version $Revision: 3460 $
 *
 * Last update: $Date: 2011-10-06 22:08:09 +0200 (jeu., 06 oct. 2011) $
 * by : $Author: chatellier $
 */
public class SimulationInformation {

    /** Class logger. */
    private static Log log = LogFactory.getLog(SimulationInformation.class);

    private static final String START_SIMULATION = "simulationStart";
    private static final String END_SIMULATION = "simulationEnd";
    private static final String EXPORT_TIME = "exportTime";
    private static final String RULE_TIME = "ruleTime";
    private static final String RULE_TIME_INIT = RULE_TIME + ".init";
    private static final String RULE_TIME_PRE = RULE_TIME + ".pre";
    private static final String RULE_TIME_POST = RULE_TIME + ".post";
    private static final String OTHER_INFO = "otherInfo";
    private static final String STATISTIC = "statistic";
    private static final String OPTIMIZATION_USAGE = "optimizationUsage";
    private static final String SIMULATION_EXCEPTION = "exception";

    private static final SimpleDateFormat dateFormat = new SimpleDateFormat(
            "yyyy.MM.dd HH:mm:ss");

    protected Properties info = new Properties();

    protected File file = null;

    /**
     * Constructor.
     * 
     * If file already exists, load his content into current instance.
     * 
     * @param file simulation information output file
     */
    public SimulationInformation(File file) {
        this.file = file;
        if (file.exists()) {
            FileReader reader = null;
            try {
                reader = new FileReader(file);
                info.load(reader);
            } catch (IOException eee) {
                if (log.isWarnEnabled()) {
                    log.warn(_("isisfish.error.read.simulation", file
                                    .getPath()), eee);
                }
            } finally {
                IOUtils.closeQuietly(reader);
            }
        }
    }

    /*
     * @see java.lang.Object#toString()
     */
    @Override
    public String toString() {
        String result = "Simulation Information:\n";

        // date start/end
        result += "Start: " + dateFormat.format(getSimulationStart())
                + " End: " + dateFormat.format(getSimulationEnd()) + "\n";

        // exports
        Map<String, Long> exportTime = getExportTimes();
        if (exportTime.size() > 0) {
            result += "Export time:\n";
            for (Map.Entry<String, Long> entry : exportTime.entrySet()) {
                result += "\t"
                        + entry.getKey()
                        + " : "
                        + DurationFormatUtils.formatDuration(entry.getValue(),
                                "s'.'S") + "s\n";
            }
        }

        // rules
        Set<String> ruleNames = getRuleNames();
        if (ruleNames.size() > 0) {
            result += "Rule time:\n";
            for (String ruleName : ruleNames) {
                
                String details = "";
                long total = 0;

                long time = getRuleInitTime(ruleName);
                if (time > 0) {
                    total += time;
                    details += "init:" + DurationFormatUtils.formatDuration(time, "s'.'S");
                }
                
                // can be 0 if condition always return false, never entrer pre/post
                time = getRulePreTime(ruleName);
                if (time > 0) {
                    if (total > 0) {
                        details += ", ";
                    }
                    total += time;
                    details += "pre:" + DurationFormatUtils.formatDuration(time, "s'.'S");
                }
                time = getRulePostTime(ruleName);
                if (time > 0) {
                    if (total > 0) {
                        details += ", ";
                    }
                    total += time;
                    details += "post:" + DurationFormatUtils.formatDuration(time, "s'.'S");
                }

                if (total > 0) {
                    result += "\t" + ruleName + " : " + DurationFormatUtils.formatDuration(total, "s'.'S");
                    result += "s (" + details + ")";
                    result += "\n";
                }
            }
        }

        // general information
        String info = getInformation();
        if (info != null && !"".equals(info)) {
            result += "Information:\n" + info + "\n";
        }

        // Statistic
        String v = getStatistic();
        if (v != null) {
            result += "Statistic:\n" + v + "\n";
        }

        // Optimisation usage
        v = getOptimizationUsage();
        if (v != null) {
            result += "Optimisation usage:\n" + v + "\n";
        }

        // Exception
        v = getException();
        if (v != null) {
            result += "Exception:\n" + v + "\n";
        }

        return result;
    }

    protected void store() {
        FileWriter writer = null;
        try {
            writer = new FileWriter(file);
            info.store(writer, "Simulation Information");
        } catch (IOException eee) {
            if (log.isWarnEnabled()) {
                log.warn(_("isisfish.error.write.simulation", file.getPath()),
                        eee);
            }
        } finally {
            IOUtils.closeQuietly(writer);
        }
    }

    protected void setInfo(String key, String value) {
        info.setProperty(key, value);
        store();
    }

    /**
     * Get the date of simulation start.
     * 
     * @return simulation start date
     */
    public Date getSimulationStart() {
        String d = info.getProperty(START_SIMULATION);
        Date result = null;
        if (d != null) {
            try {
                result = dateFormat.parse(d);
            } catch (ParseException eee) {
                if (log.isWarnEnabled()) {
                    log.warn(_("isisfish.error.parse.date", d), eee);
                }
            }
        }
        if (result == null) {
            result = new Date(0);
        }
        return result;
    }

    public void setSimulationStart(Date date) {
        setInfo(START_SIMULATION, dateFormat.format(date));
    }

    /**
     * Get the date of simulation start.
     * 
     * @return simulation end date
     */
    public Date getSimulationEnd() {
        String d = info.getProperty(END_SIMULATION);
        Date result = null;
        if (d != null) {
            try {
                result = dateFormat.parse(d);
            } catch (ParseException eee) {
                if (log.isWarnEnabled()) {
                    log.warn(_("isisfish.error.parse.date", d), eee);
                }
            }
        }
        if (result == null) {
            result = new Date(0);
        }
        return result;
    }

    public void setSimulationEnd(Date date) {
        setInfo(END_SIMULATION, dateFormat.format(date));
    }

    public void addExportTime(String exportName, long time) {
        setInfo(EXPORT_TIME + "." + exportName, String.valueOf(time));
    }

    /**
     * Get all export time in map.
     * 
     * @return a map with all export time
     */
    protected Map<String, Long> getExportTimes() {
        Map<String, Long> result = new HashMap<String, Long>();
        for (String key : info.stringPropertyNames()) {
            if (key.startsWith(EXPORT_TIME + ".")) {
                String exportName = key.substring(EXPORT_TIME.length() + 1);
                String t = info.getProperty(key);
                Long value = null;
                if (t != null) {
                    try {
                        value = Long.valueOf(t);
                    } catch (NumberFormatException eee) {
                        if (log.isWarnEnabled()) {
                            log.warn(_("isisfish.error.parse.long", t), eee);
                        }
                    }
                }
                if (value == null) {
                    value = Long.valueOf(0);
                }
                result.put(exportName, value);
            }
        }
        return result;
    }

    /**
     * Get all export time in map.
     * 
     * @return a map with all export time
     * @deprecated since 3.2.0.5, use {@link #getExportTimes()} instead
     */
    public Map<String, Long> getExportTime() {
        return getExportTimes();
    }

    public long getExportTime(String exportName) {
        String t = info.getProperty(EXPORT_TIME + "." + exportName);
        long result = 0;
        if (t != null) {
            try {
                result = Long.parseLong(t);
            } catch (NumberFormatException eee) {
                if (log.isWarnEnabled()) {
                    log.warn(_("isisfish.error.parse.long", t), eee);
                }
            }
        }
        return result;
    }

    /**
     * Add rule time.
     * 
     * If a time already exists for ruleName, add time to previous time.
     * (usefull because pre/post action are called multiples time)
     * 
     * @param keyName (ie {@link #RULE_TIME_INIT}, {@link #RULE_TIME_PRE}, {@link #RULE_TIME_POST})
     * @param ruleName rule name
     * @param time time to add
     * 
     * @since 3.2.0.5
     */
    protected void addRuleTime(String keyName, String ruleName, long time) {

        // get previous time
        String previousTimeAsString = info
                .getProperty(keyName + "." + ruleName);
        long previousTime = 0;
        if (previousTimeAsString != null) {
            try {
                previousTime = Long.parseLong(previousTimeAsString);
            } catch (NumberFormatException eee) {
                if (log.isWarnEnabled()) {
                    log.warn(_("isisfish.error.parse.long",
                            previousTimeAsString), eee);
                }
            }
        }

        // add
        long newTime = previousTime + time;

        // set new time in
        setInfo(keyName + "." + ruleName, String.valueOf(newTime));
    }

    /**
     * Add rule init time.
     * 
     * If a time already exists for ruleName, add time to previous time.
     * 
     * @param ruleName rule name
     * @param time time to add
     * @since 3.2.0.5
     */
    public void addRuleInitTime(String ruleName, long time) {
        addRuleTime(RULE_TIME_INIT, ruleName, time);
    }

    /**
     * Add rule pre operation time time.
     * 
     * If a time already exists for ruleName, add time to previous time.
     * 
     * @param ruleName rule name
     * @param time time to add
     * @since 3.2.0.5
     */
    public void addRulePreTime(String ruleName, long time) {
        addRuleTime(RULE_TIME_PRE, ruleName, time);
    }

    /**
     * Add rule post operation time.
     * 
     * If a time already exists for ruleName, add time to previous time.
     * 
     * @param ruleName rule name
     * @param time time to add
     * @since 3.2.0.5
     */
    public void addRulePostTime(String ruleName, long time) {
        addRuleTime(RULE_TIME_POST, ruleName, time);
    }

    /**
     * Get rule init operation time.
     * 
     * @param ruleName rule name
     * @return time
     * @since 3.2.0.5
     */
    public long getRuleInitTime(String ruleName) {
        return getRuleTime(RULE_TIME_INIT, ruleName);
    }

    /**
     * Get rule pre operation time.
     * 
     * @param ruleName rule name
     * @return time
     * @since 3.2.0.5
     */
    public long getRulePreTime(String ruleName) {
        return getRuleTime(RULE_TIME_PRE, ruleName);
    }

    /**
     * Get rule post operation time.
     * 
     * @param ruleName rule name
     * @return time
     * @since 3.2.0.5
     */
    public long getRulePostTime(String ruleName) {
        return getRuleTime(RULE_TIME_POST, ruleName);
    }

    /**
     * Get rule operation time.
     * 
     * @param ruleName rule name
     * @since 3.2.0.5
     */
    protected long getRuleTime(String prefixName, String ruleName) {
        String t = info.getProperty(prefixName + "." + ruleName);
        long result = 0;
        if (t != null) {
            try {
                result = Long.parseLong(t);
            } catch (NumberFormatException eee) {
                if (log.isWarnEnabled()) {
                    log.warn(_("isisfish.error.parse.long", t), eee);
                }
            }
        }
        return result;
    }

    /**
     * Get rules names.
     * 
     * @return a map with all export time
     */
    protected Set<String> getRuleNames() {
        Set<String> result = new HashSet<String>();
        for (String key : info.stringPropertyNames()) {
            if (key.startsWith(RULE_TIME + ".")) {
                // recupere le nom apres le deuxieme "."
                String ruleName = key.substring(key.indexOf('.', RULE_TIME.length() + 1) + 1);
                result.add(ruleName);
            }
        }
        return result;
    }

    public String getStatistic() {
        String result = info.getProperty(STATISTIC);
        return result;
    }

    public void setStatistic(String v) {
        setInfo(STATISTIC, v);
    }

    public String getOptimizationUsage() {
        String result = info.getProperty(OPTIMIZATION_USAGE);
        return result;
    }

    public void setOptimizationUsage(String v) {
        setInfo(OPTIMIZATION_USAGE, v);
    }

    public String getException() {
        String result = info.getProperty(SIMULATION_EXCEPTION);
        return result;
    }

    public void setException(Throwable eee) {
        StringWriter w = new StringWriter();
        PrintWriter pw = new PrintWriter(w);
        eee.printStackTrace(pw);
        String v = w.getBuffer().toString();
        setInfo(SIMULATION_EXCEPTION, v);
    }

    /**
     * Return {@code true} if an exception has been set.
     * 
     * @return {@code true} if there is an exception
     */
    public boolean hasError() {
        boolean result = getException() != null && getException().length() > 0;
        return result;
    }

    /**
     * @deprecated since 3.2.0.5, use {@link #getInformation()} instead
     * @return other information
     */
    public String getInfomation() {
        return getInformation();
    }

    /**
     * Get other information.
     * 
     * @return other information
     */
    protected String getInformation() {
        String result = info.getProperty(OTHER_INFO);
        if (result == null) {
            result = "";
        }
        return result;
    }

    /**
     * Add additional simulation information.
     * 
     * @param info new info
     */
    public void addInformation(String info) {
        setInfo(OTHER_INFO, getInformation() + info + "\n");
    }
}
