/*
 * #%L
 * IsisFish
 * 
 * $Id: AspectClassLoader.java 3124 2010-11-29 18:14:09Z chatellier $
 * $HeadURL$
 * %%
 * Copyright (C) 1999 - 2010 Ifremer, Code Lutin, Cédric Pineau, Benjamin Poussin
 * %%
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as
 * published by the Free Software Foundation, either version 2 of the 
 * License, or (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public 
 * License along with this program.  If not, see
 * <http://www.gnu.org/licenses/gpl-2.0.html>.
 * #L%
 */

package fr.ifremer.isisfish.aspect;

import java.io.IOException;
import java.net.URL;
import java.net.URLClassLoader;
import java.util.HashSet;
import java.util.Iterator;
import java.util.Set;

import org.codehaus.aspectwerkz.annotation.AspectAnnotationParser;
import org.codehaus.aspectwerkz.definition.AdviceDefinition;
import org.codehaus.aspectwerkz.definition.AspectDefinition;
import org.codehaus.aspectwerkz.definition.DeploymentScope;
import org.codehaus.aspectwerkz.definition.SystemDefinition;
import org.codehaus.aspectwerkz.definition.SystemDefinitionContainer;
import org.codehaus.aspectwerkz.expression.ExpressionInfo;
import org.codehaus.aspectwerkz.hook.impl.ClassPreProcessorHelper;
import org.codehaus.aspectwerkz.reflect.ClassInfo;
import org.codehaus.aspectwerkz.reflect.impl.asm.AsmClassInfo;
import org.codehaus.aspectwerkz.reflect.impl.java.JavaClassInfo;
import org.codehaus.aspectwerkz.transform.inlining.AspectModelManager;

import sun.misc.Resource;
import sun.misc.URLClassPath;

/**
 * Aspectwerkz class loader.
 * 
 * http://aspectwerkz.codehaus.org/xref/org/codehaus/aspectwerkz/transform/inlining/deployer/Deployer.html
 * 
 * Created: 30 mars 07 00:31:22
 *
 * @author poussin
 * @version $Revision: 3124 $
 *
 * Last update: $Date: 2010-11-29 19:14:09 +0100 (lun., 29 nov. 2010) $
 * by : $Author: chatellier $
 */
public class AspectClassLoader extends URLClassLoader {

    /**
     * Class loader constructor.
     * 
     * @param urls parent urlclassloader url
     * @param parent parent class loader
     */
    public AspectClassLoader(URL[] urls, ClassLoader parent) {
        super(urls, parent);
        ClassPreProcessorHelper.initializePreProcessor();
    }

    @Override
    protected Class<?> findClass(String name) throws ClassNotFoundException {
        String path = name.replace('.', '/').concat(".class");
        URLClassPath ucp = new URLClassPath(getURLs());
        Resource res = ucp.getResource(path, false);
        if (res != null) {
            //definePackage(name.substring(0, name.lastIndexOf(".")), null, null);
            try {
                byte[] b = res.getBytes();
                byte[] transformed = ClassPreProcessorHelper.defineClass0Pre(this, name, b, 0, b.length, null);
                return defineClass(name, transformed, 0, transformed.length);
            } catch (IOException e) {
                throw new ClassNotFoundException("Can't build aspect for class", e);
            }
        } else {
            throw new ClassNotFoundException(name);
        }
    }

    /**
     * Register new aspect.
     * 
     * @param aspectClass aspect class to register
     */
    public void deploy(Class<?> aspectClass) {
        String className = aspectClass.getName();
        try {
            aspectClass = Class.forName(className, false, this);
        } catch (ClassNotFoundException e) {
            throw new RuntimeException("could not load class [" + className + "] in class loader [" + this + "]");
        }

        final ClassInfo aspectClassInfo = JavaClassInfo.getClassInfo(aspectClass);

        // create a new aspect def and fill it up with the annotation def from the aspect class
        final SystemDefinition systemDef = SystemDefinitionContainer.getVirtualDefinitionAt(this);
        final AspectDefinition newAspectDef = new AspectDefinition(className, aspectClassInfo, systemDef);
        getNewExpressionsForAspect(aspectClass, newAspectDef, systemDef, DeploymentScope.MATCH_ALL);
    }

    /**
     * Returns a set with the new expressions for the advice in the aspect to deploy.
     *
     * @param aspectClass
     * @param newAspectDef
     * @param newAspectDef
     * @param systemDef
     * @param deploymentScope
     * @return a set with the new expressions
     */
    private Set getNewExpressionsForAspect(final Class aspectClass,
            final AspectDefinition newAspectDef,
            final SystemDefinition systemDef,
            final DeploymentScope deploymentScope) {
        final ClassLoader aspectLoader = aspectClass.getClassLoader();
        final String aspectName = aspectClass.getName();

        final ClassInfo classInfo = AsmClassInfo.getClassInfo(aspectName, aspectLoader);

        AspectModelManager.defineAspect(classInfo, newAspectDef, aspectLoader);

        AspectAnnotationParser.parse(classInfo, newAspectDef, aspectLoader);

        AspectDefinition aspectDef = systemDef.getAspectDefinition(aspectName);
        if (aspectDef != null) {
            // if in def already reuse some of the settings that can have been overridded by XML def
            newAspectDef.setContainerClassName(aspectDef.getContainerClassName());
            newAspectDef.setDeploymentModel(aspectDef.getDeploymentModel());
        }

        systemDef.addAspectOverwriteIfExists(newAspectDef);

        final Set newExpressions = new HashSet();
        for (Iterator it2 = newAspectDef.getAdviceDefinitions().iterator(); it2.hasNext();) {
            AdviceDefinition adviceDef = (AdviceDefinition) it2.next();
            ExpressionInfo oldExpression = adviceDef.getExpressionInfo();
            if (oldExpression == null) {
                continue;
            }
            // deploymentHandle.registerDefinitionChange(adviceDef, oldExpression);

            final ExpressionInfo newExpression = deploymentScope.newExpressionInfo(oldExpression);
            adviceDef.setExpressionInfo(newExpression);
            newExpressions.add(newExpression);
        }
        return newExpressions;
    }
}
