/*
 * #%L
 * Vradi :: Swing
 * 
 * $Id: UIHelper.java 1778 2010-11-15 13:43:46Z sletellier $
 * $HeadURL: svn+ssh://sletellier@labs.libre-entreprise.org/svnroot/vradi/vradi/tags/vradi-0.4.0/vradi-swing/src/main/java/com/jurismarches/vradi/ui/helpers/UIHelper.java $
 * %%
 * Copyright (C) 2009 - 2010 JurisMarches, Codelutin
 * %%
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as
 * published by the Free Software Foundation, either version 3 of the 
 * License, or (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public 
 * License along with this program.  If not, see
 * <http://www.gnu.org/licenses/gpl-3.0.html>.
 * #L%
 */
package com.jurismarches.vradi.ui.helpers;

import static org.nuiton.i18n.I18n._;

import java.awt.Component;
import java.awt.Desktop;
import java.io.File;
import java.io.FileWriter;
import java.io.IOException;
import java.net.URI;
import java.net.URISyntaxException;
import java.net.URL;
import java.text.NumberFormat;
import java.util.Arrays;
import java.util.Collection;
import java.util.Collections;
import java.util.List;

import com.jurismarches.vradi.VradiContext;
import com.jurismarches.vradi.entities.Group;
import javax.swing.ComboBoxModel;
import javax.swing.DefaultComboBoxModel;
import javax.swing.DefaultListCellRenderer;
import javax.swing.JFileChooser;
import javax.swing.JFormattedTextField;
import javax.swing.JLabel;
import javax.swing.JList;
import javax.swing.JOptionPane;
import javax.swing.JPanel;
import javax.swing.JTable;
import javax.swing.JTree;
import javax.swing.ListCellRenderer;
import javax.swing.ListModel;
import javax.swing.SwingUtilities;
import javax.swing.event.TreeSelectionEvent;
import javax.swing.event.TreeSelectionListener;
import javax.swing.filechooser.FileFilter;
import javax.swing.text.NumberFormatter;
import javax.swing.tree.TreeNode;
import javax.swing.tree.TreePath;

import jaxx.runtime.JAXXContext;
import jaxx.runtime.SwingUtil;
import jaxx.runtime.context.JAXXInitialContext;
import jaxx.runtime.swing.CardLayout2;
import jaxx.runtime.swing.ErrorDialogUI;

import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.jdesktop.swingx.JXTreeTable;

import au.com.bytecode.opencsv.CSVWriter;

import com.jurismarches.vradi.VradiHelper;
import com.jurismarches.vradi.entities.Status;
import com.jurismarches.vradi.entities.XmlStream;
import com.jurismarches.vradi.services.VradiDataService;
import com.jurismarches.vradi.services.VradiService;
import com.jurismarches.vradi.ui.admin.AdminHandler;
import com.jurismarches.vradi.ui.models.EntityModel;
import com.jurismarches.vradi.ui.models.FormTypeModel;
import com.jurismarches.vradi.ui.offer.models.OfferListTableModel;
import com.jurismarches.vradi.ui.renderers.VradiI18nTableCellRenderer;
import org.nuiton.widget.SwingSession;

/**
 * Classe de méthodes utiles pour les ui.
 * <p/>
 * User: chemit
 * Date: 23 nov. 2009
 * Time: 11:51:14
 */
public class UIHelper extends SwingUtil {

    /** Logger. */
    private static final Log log = LogFactory.getLog(UIHelper.class);

    public static final String INFOGENE_KEY_PREFIX = "vradi.infogene.";
    public static final String FORM_KEY_PREFIX = "vradi.form.";

    private UIHelper() {
        // pas d'instance
    }

    /**
     * Return traduction of infogene field pass in param
     *
     * @param key to traduct
     * @return traducted
     */
    public static String getInfogeneTraduction(String key) {
        return _(INFOGENE_KEY_PREFIX + key);
    }

    /**
     * Return traduction of form field pass in param
     *
     * @param key to traduct
     * @return traducted
     */
    public static String getFormTraduction(String key) {
        return _(FORM_KEY_PREFIX + key);
    }

    /**
     * Met à jour le modèle de sélection {@code list} donnée avec l'objet {@code value} entrant.
     * <p/>
     * {@code value} represente normalement une collection de données.
     *
     * @param list  la liste à updater
     * @param value la liste des valeurs de la selection de la liste.
     */
    public static void updateListSelection(JList list, Object value) {

        list.setValueIsAdjusting(true);
        list.clearSelection();

        try {
            if (value != null) {

                // la valeur doit être une collection, on met a jour le model de selection
                int size = list.getModel().getSize();
                Collection<?> valuesToSelect = (Collection<?>) value;
                int[] indexToSelect = new int[valuesToSelect.size()];
                int index = 0;
                for (Object v : valuesToSelect) {
                    for (int i = 0; i < size; i++) {
                        Object o = list.getModel().getElementAt(i);
                        if (o.equals(v)) {
                            indexToSelect[index++] = i;
                            break;
                        }
                    }
                }
                if (log.isTraceEnabled()) {
                    log.trace("index to select = " + Arrays.toString(indexToSelect));
                }
                list.setSelectedIndices(indexToSelect);
            }

        } finally {

            list.setValueIsAdjusting(false);
        }
    }

    // I18n
    public static void setI18nTableHeaderRenderer(JTable table, String... libelles) {
        table.getTableHeader().setDefaultRenderer(new VradiI18nTableCellRenderer(table.getTableHeader().getDefaultRenderer(), libelles));
    }

    /**
     * Ouvre un navigateur sur google maps.
     * 
     * Utilisé dans les champs "address" des clients et des utilisateurs.
     * 
     * @param address address to browse
     */
    public static void openAddressInBrowser(String address) {
        if(address != null) {
            try {
                browseURI("http://maps.google.fr/maps?f=q&source=s_q&hl=fr&geocode=&q="
                    + getStringValue(address) + ",France");
            } catch (IOException eee) {
                log.error("Cant display page for address : " +
                        getStringValue(address), eee);
                JOptionPane.showMessageDialog(null,
                        _("vradi.message.cannot.display.address"));

            }
        }
    }

    public static void browseURI(URI uri) throws IOException {
        if(uri != null) {
            if (log.isDebugEnabled()) {
                log.debug("Browse URI : " + uri.toString());
            }
            Desktop desktop = Desktop.getDesktop();
            desktop.browse(uri);
        }
    }

    public static void browseURI(String uri) throws IOException {
        if (uri != null) {
            try {
                browseURI(new URI(uri.replace(" ", "+")));
            } catch (URISyntaxException eee) {
                log.error("Cant display uri : " + uri, eee);
                ErrorDialogUI.showError(eee);
            }
        }
    }

    public static void browseURI(URL url) throws IOException {
        if (url != null) {
            String externalForm = url.toExternalForm();
            browseURI(externalForm);
        }
    }

    public static void createEmail(String emailAdress)
            throws IOException, URISyntaxException, UnsupportedOperationException {
        URI mailtoURI = new URI("mailto:" + emailAdress);
        Desktop.getDesktop().mail(mailtoURI);
    }

    /**
     * Export la table des résultats de recherche au format CSV.
     * 
     * @param file file to export to
     * @param model model to export
     */
    public static void exportToCSV(File file, OfferListTableModel model) {
        try {
            FileWriter fileWriter = new FileWriter(file);
            CSVWriter csvWriter = new CSVWriter(fileWriter);

            int columnCount = model.getColumnCount();
            String[] nextLine = new String[columnCount];
            
            // writes headers
            for (int i = 0; i < model.getColumnCount(); i++) {
                nextLine[i] = _(model.getColumnName(i));
            }
            csvWriter.writeNext(nextLine);
            
            // writes row values
            for (int i = 0; i < model.getRowCount(); i++) {
                nextLine = new String[columnCount];
                
                for (int j = 0; j < model.getColumnCount(); j++) {
                    Object valueAt = model.getValueAt(i, j);
                    if (valueAt != null) {
                        String value = String.valueOf(valueAt);
                        if (valueAt instanceof Status) {
                            value = ((Status) valueAt).getName();
                        } else if (valueAt instanceof XmlStream) {
                            value = ((XmlStream) valueAt).getName();
                        }

                        nextLine[j] = value;
                    }
                }
                csvWriter.writeNext(nextLine);
                
            }
            csvWriter.close();

        } catch (IOException e) {
            log.error(e.getMessage(), e);
            ErrorDialogUI.showError(e);
        }
    }

    /**
     * Ouvre un file chooser configuré sur un certain type de fichier et
     * retourne le fichier choisit par l'utilisateur.
     * 
     * @param fileExtension extension with begin "."
     * @param description filter description
     * @param component parent component
     * @param validationLabel ok button label
     * @return selected file or {@code null}
     */
    public static File openFileChooser(final String fileExtension, final String description, Component component, String validationLabel) {
        JFileChooser fileChooser = new JFileChooser();
        fileChooser.setFileSelectionMode(JFileChooser.FILES_ONLY);
        fileChooser.setAcceptAllFileFilterUsed(false);
        fileChooser.setFileFilter(new FileFilter() {

            @Override
            public boolean accept(File file) {
                if (file.isDirectory()) {
                    return true;
                }

                String ext = null;
                String fileName = file.getName();
                int dot = fileName.lastIndexOf('.');

                if (dot > 0 && dot < fileName.length() - 1) {
                    ext = fileName.substring(dot).toLowerCase();
                }
                return ext != null && ext.equals(fileExtension);
            }

            @Override
            public String getDescription() {
                return description;
            }
        });

        int returnVal = fileChooser.showDialog(component, validationLabel);
        if (returnVal == JFileChooser.APPROVE_OPTION) {
            File file = fileChooser.getSelectedFile();
            if (!file.getName().endsWith(fileExtension)) {
                file = new File(file.getAbsolutePath() + fileExtension);
            }
            log.debug("saving : " + file.getName() + ".");
            return file;
        } else {
            log.debug("No file chosen by user.");
            return null;
        }
    }

    public static ListCellRenderer getTranslationListCellRenderer() {
        return new DefaultListCellRenderer() {

            @Override
            public Component getListCellRendererComponent(
                    JList list,
                    Object value,
                    int index,
                    boolean isSelected,
                    boolean cellHasFocus) {
                if (value != null) {
                    return super.getListCellRendererComponent(list, _(value.toString()), index,
                            isSelected, cellHasFocus);
                } else {
                    return super.getListCellRendererComponent(list, value, index,
                            isSelected, cellHasFocus);
                }
            }

        };
    }

    public static <T> T getHandler(JAXXContext context, Class<T> handlerClass) {
        T handler = context.getContextValue(handlerClass);
        if (handler != null) {
            return handler;
        }
        try {
            handler = handlerClass.newInstance();
        } catch (InstantiationException e) {
            log.error("Cant create handler instance of type " + handlerClass.getName() + " : ", e);
            ErrorDialogUI.showError(e);
        } catch (IllegalAccessException e) {
            log.error("Cant create handler instance of type " + handlerClass.getName() + " : ", e);
            ErrorDialogUI.showError(e);
        }

        if (context instanceof JAXXInitialContext) {
            ((JAXXInitialContext) context).add(handler);
        } else {
            context.setContextValue(handler);
        }
        return handler;
    }

    /**
     * Return an initialized ComboBoxModel with int from startIndex and filled
     * with nbData.
     * (used in xmlstream import time config)
     * 
     * @param startIndex start index
     * @param nbData nb data
     * @return ComboBoxModel
     */
    public static ComboBoxModel getNumberComboBoxModel(int startIndex, int nbData) {
        Object[] data = new Integer[nbData];
        for (int i = startIndex; i < data.length; i++) {
            data[i] = i;
        }
        return new DefaultComboBoxModel(data);
    }

    /**
     * Return an initialized ComboBoxModel with int from 0 to nbData.
     * (used in xmlstream import time config)
     * 
     * @param nbData nb data
     * @return ComboBoxModel
     */
    public static ComboBoxModel getNumberComboBoxModel(int nbData) {
        return getNumberComboBoxModel(0, nbData);
    }

    /**
     * Return an initialized ComboBoxModel with int from 0 to 24.
     * (used in xmlstream import time config)
     * 
     * @return ComboBoxModel
     */
    public static ComboBoxModel getHourModel() {
        return getNumberComboBoxModel(24);
    }

    /**
     * Return an initialized ComboBoxModel with int from 0 to 60.
     * (used in xmlstream import time config)
     * 
     * @return ComboBoxModel
     */
    public static ComboBoxModel getMinuteModel() {
        return getNumberComboBoxModel(60);
    }

    /**
     * Get time list renderer (hour or minutes).
     * 
     * @return ListCellRenderer
     */
    public static ListCellRenderer get2DigitsRenderer() {
        return new DefaultListCellRenderer() {
            /** serialVersionUID. */
            private static final long serialVersionUID = 1L;

            @Override
            public Component getListCellRendererComponent(JList list, Object value,
                                                          int index, boolean isSelected, boolean cellHasFocus) {

                String stringValue = null;
                if (value != null) {
                    stringValue = (Integer) value < 10 ? "0" + value.toString() : value.toString();
                }
                return super.getListCellRendererComponent(list, stringValue, index, isSelected, cellHasFocus);
            }
        };
    }

    /**
     * Return i18n translation for {@code value}.
     * 
     * @return i18n transaltion
     */
    public static ListCellRenderer getTranslationRenderer() {
        return new DefaultListCellRenderer() {
            /** serialVersionUID. */
            private static final long serialVersionUID = 1L;
            @Override
            public Component getListCellRendererComponent(JList list, Object value,
                                                          int index, boolean isSelected, boolean cellHasFocus) {
                String stringValue = null;
                if (value != null) {
                    stringValue = value.toString();
                }
                return super.getListCellRendererComponent(list, _(stringValue), index, isSelected, cellHasFocus);
            }
        };
    }

    public static VradiDataService getDataService() {
        return VradiService.getVradiDataService();
    }

    public static EntityModel<Group> getGroupModel(boolean firstNull) {
        List<Group> groupList = getDataService().findAllGroups();

        // Sort
        Collections.sort(groupList, VradiComparators.GROUP_COMPARATOR);
        return new EntityModel<Group>(Group.class, Group.EXT_GROUP, groupList, firstNull);
    }

    public static EntityModel<XmlStream> getXmlStreamsModel() {
        List<XmlStream> streamList = getDataService().findAllXmlStreams();

        // Sort
        Collections.sort(streamList, VradiComparators.XML_STREAM_COMPARATOR);
        return new EntityModel<XmlStream>(XmlStream.class, XmlStream.EXT_XMLSTREAM, streamList);
    }

    public static ListModel getStatusesModel(boolean firstNull) {
        List<Status> status = AdminHandler.getAllStatuses();
        return new EntityModel<Status>(Status.class, Status.EXT_STATUS, status, firstNull);
    }

    /**
     * Return status cell renderer (translate i18n status name key).
     * 
     * @return ListCellRenderer
     */
    public static ListCellRenderer getStatusCellRenderer() {
        return new DefaultListCellRenderer(){
            /** serialVersionUID. */
            private static final long serialVersionUID = 1L;
            @Override
            public Component getListCellRendererComponent(JList list, Object value, int index, boolean isSelected, boolean cellHasFocus) {
                Status status = (Status)value;
                return super.getListCellRendererComponent(list, _(status.getName()), index, isSelected, cellHasFocus) ;
            }
        };
    }

    public static ComboBoxModel getFormTypesModel() {
        return new FormTypeModel(false);
    }

    public static ComboBoxModel getFormTypesModel(boolean firstNull) {
        return new FormTypeModel(firstNull);
    }

    /**
     * Get content if exist in content, else return null
     * @param contentLayout layout to search
     * @param content concerned
     * @param uiClass class of ui searched
     * @return content if found
     */
    public static <E extends Component> E getContentIfExist(CardLayout2 contentLayout, JPanel content, Class<E> uiClass) {
        String contentName = uiClass.getName();
        if (log.isDebugEnabled()) {
            log.debug("Get content if exist " + contentName);
        }
        if (!contentLayout.contains(contentName)) {
            return null;
        }
        return (E)contentLayout.getComponent(content, contentName);
    }

    /**
     * NumberFormatterFactory creates a new NumberFormatter for Integer of variable length.
     */
    public static class NumberFormatterFactory extends JFormattedTextField.AbstractFormatterFactory {
        @Override
        public JFormattedTextField.AbstractFormatter getFormatter(JFormattedTextField jFormattedTextField) {
            NumberFormat numberFormat = NumberFormat.getIntegerInstance();
            NumberFormatter numberFormatter = new NumberFormatter(numberFormat);
            numberFormatter.setAllowsInvalid(false);
            numberFormatter.setValueClass(Integer.class);
            return numberFormatter;
        }
    }

    /**
     * Used to cross of text
     *
     * @param text to cross of
     * @return text crossed of
     */
    public static String crossOf(String text) {
        return "<html><strike>" + text + "</strike></html>";
    }

    /**
     * Used to cross of component
     *
     * @param component to cross of
     * @return component crossed of
     */
    public static Component crossOf(Component component) {
        if (component instanceof JLabel) {
            JLabel label = (JLabel) component;
            label.setText(crossOf(label.getText()));

            return label;
        }
        return component;
    }

    /**
     * To keep ui diposition
     *
     * @param c to register
     */
    public static void registerComponentToSaveDispositionConfig(Component c) {
        VradiContext.SWING_SESSION_ENTRY_DEF.getContextValue(VradiContext.get()).add(c);
    }

    /**
     * To save ui diposition
     */
    public static void saveComponentDispositionConfig() {
        SwingSession swingSession = VradiContext.SWING_SESSION_ENTRY_DEF.getContextValue(VradiContext.get());
        swingSession.save();
    }

    /**
     * Add a listener of tree selection model to expand a new selected node
     * when it is selected.
     *
     * @param tree the tree to treate
     */
    public static void addThesaurusExpandOnClickListener(final JTree tree) {

        tree.getSelectionModel().addTreeSelectionListener(
            new TreeSelectionListener() {

            @Override
            public void valueChanged(final TreeSelectionEvent e) {

                boolean isThesaurusExpanded = VradiHelper.isOnClickThesaurusExpanded();

                if (!isThesaurusExpanded) {
                    return;
                }

                TreeNode node = (TreeNode)
                        e.getPath().getLastPathComponent();
                if (node != null && !node.isLeaf()) {

                    SwingUtilities.invokeLater(new Runnable() {

                        @Override
                        public void run() {
                            for (TreePath path : e.getPaths()) {
                                if (e.isAddedPath(path) &&
                                    !tree.isExpanded(path)) {
                                    if (log.isDebugEnabled()) {
                                        log.debug("expand node [" + path
                                             + "]");
                                    }

                                    // will expand the node
                                    tree.expandPath(path);
                                }
                            }
                        }
                    });
                }
            }
        });
    }

    /**
     * Add a listener of tree selection model to expand a new selected node
     * when it is selected.
     *
     * @param treeTable the treeTable to treate
     */
    public static void addThesaurusExpandOnClickListener(final JXTreeTable treeTable) {

        treeTable.addTreeSelectionListener(
            new TreeSelectionListener() {

            @Override
            public void valueChanged(final TreeSelectionEvent e) {

                TreeNode node = (TreeNode)
                        e.getPath().getLastPathComponent();
                if (node != null && !node.isLeaf()) {

                    SwingUtilities.invokeLater(new Runnable() {

                        @Override
                        public void run() {
                            boolean isThesaurusExpanded = VradiHelper.isOnClickThesaurusExpanded();
                            for (TreePath path : e.getPaths()) {
                                if (e.isAddedPath(path) &&
                                    !treeTable.isExpanded(path)) {

                                    if (log.isDebugEnabled()) {
                                        log.debug("expand node [" + path
                                             + "]");
                                    }
                                    
                                    if (isThesaurusExpanded) {

                                        treeTable.expandPath(path);
                                    }

                                    // Keep selection
                                    treeTable.getTreeSelectionModel().setSelectionPath(path);
                                }
                            }
                        }
                    });
                }
            }
        });
    }
}
