package com.jurismarches.vradi.ui.helpers;

import com.jurismarches.vradi.entities.*;
import com.jurismarches.vradi.services.VradiService;
import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.nuiton.wikitty.*;
import org.nuiton.wikitty.search.Element;
import org.nuiton.wikitty.search.Search;

import java.util.*;

public class ThesaurusDataHelper {

    /**
     * Logger
     */
    static private final Log log = LogFactory.getLog(ThesaurusDataHelper.class);

    /**
     * Get the wikitty proxy
     * @return WikittyProxy
     */
    public static WikittyProxy getProxy() {
        return VradiService.getWikittyProxy();
    }

    /**
     * Get corresponding thesaurus
     *
     * @param thesaurusId thesaurus id
     * @return Thesaurus from base
     */
    public static Thesaurus restoreThesaurus(String thesaurusId){
        return getProxy().restore(Thesaurus.class, thesaurusId);
    }

    /**
     * Get corresponding list of thesaurus
     *
     * @param thesaurusIds thesaurus ids
     * @return List of thesaurus from base
     */
    public static List<Thesaurus> restoreThesaurus(List<String> thesaurusIds){
        return getProxy().restore(Thesaurus.class, thesaurusIds);
    }

    public static List<Thesaurus> restoreThesaurus(Set<String> thesaurusIds) {
        return restoreThesaurus(new ArrayList<String>(thesaurusIds));
    }

    /**
     * Get corresponding root thesaurus
     *
     * @param rootThesaurusId root thesaurus id
     * @return Thesaurus from base
     */
    public static RootThesaurus restoreRootThesaurus(String rootThesaurusId){
        return getProxy().restore(RootThesaurus.class, rootThesaurusId);
    }

    /**
     * Get corresponding list root thesaurus
     *
     * @param rootThesaurusIds root thesaurus ids
     * @return List root thesaurus from base
     */
    public static List<RootThesaurus> restoreRootThesaurus(List<String> rootThesaurusIds){
        return getProxy().restore(RootThesaurus.class, rootThesaurusIds);
    }

    /**
     * Get the path with name of thesaurus ({@code root/thesaurus...})
     * @param thesaurus thesaurus concerned
     * @param pathSeparator separator
     * @return path with name
     */
    public static String getNamePath(Thesaurus thesaurus, String pathSeparator) {
        if (thesaurus == null) {
            return "";
        }
        String parentId = thesaurus.getParent();

        // If root
        if (isFirstChild(thesaurus)) {
            return restoreRootThesaurus(thesaurus.getRootThesaurus()).getName() + pathSeparator + concateOrderName(thesaurus);
        }
        Thesaurus parentThesaurus = restoreThesaurus(parentId);
        return getNamePath(parentThesaurus, pathSeparator) + pathSeparator + concateOrderName(thesaurus);
    }

    /**
     * Concatenate order and name of thesaurus in param ({@code order - name}}
     *
     * @param thesaurus thesaurus concerned
     * @return concatenate order and name
     */
    public static String concateOrderName(Thesaurus thesaurus) {
        String result = thesaurus.getName();
        int order = thesaurus.getOrder();
        if (order != 0){
            result = order + " - " + result;
        }
        return result;
    }

    /**
     * Get tags as string ({@code tag1, tag2, ...})
     *
     * @param thesaurus thesaurus concerned
     * @return tags as string
     */
    public static String getTagsAsString(Thesaurus thesaurus) {
        if (thesaurus == null) {
            return "";
        }
        Set<String> tags = thesaurus.getTags();

        // Tags as a string
        StringBuilder tagAsString = new StringBuilder();

        if (tags != null) {
            for (String tag : tags) {
                tagAsString.append(tag);
                tagAsString.append(", ");
            }
            int length = tagAsString.length();
            if (length > 0) {
                tagAsString.delete(length - 2, length);
            }
        }

        return tagAsString.toString();
    }

    /**
     * Set tags as string ({@code tag1, tag2, ...})
     *
     * @param thesaurus    thesaurus concerned
     * @param tagsAsString tags as string
     */
    public static void setTagsAsString(Thesaurus thesaurus, String tagsAsString) {
        if (tagsAsString != null) {

            // Clear existing
            thesaurus.clearTags();

            // Resore tags from tags as string
            for (String tag : tagsAsString.split(",")) {
                thesaurus.addTags(tag.trim());
            }
        }
    }

    /**
     * Return number of form associated to thesaurus pass in param
     *
     * @param thesaurus thesaurus concerned
     * @return number of form associated
     */
    public static int getNbFormsForThesaurus(Thesaurus thesaurus) {

        Map.Entry<Thesaurus, Integer> entry =
                getProxy().restoreNode(Thesaurus.class, thesaurus.getWikittyId(), null);

        if (entry == null) {
            return 0;
        }

        return entry.getValue();
    }

    protected static List<String> getStringAsList(String s) {
        List<String> result = new ArrayList<String>();
        result.add(s);
        return result;
    }

    /**
     * Return the children thesaurus of one pass in param
     * @param thesaurusId id of thesaurus
     * @return list of children
     */
    public static List<Thesaurus> getChildrenOfThesaurus(String thesaurusId) {

        Search query = Search.query();
        query.eq(Element.ELT_EXTENSION, Thesaurus.EXT_THESAURUS);
        query.eq(Thesaurus.FQ_FIELD_TREENODE_PARENT, thesaurusId);

        Criteria criteria = query.criteria();
        PagedResult<Thesaurus> nodes = getProxy()
                .findAllByCriteria(Thesaurus.class, criteria);
        List<Thesaurus> all = new ArrayList<Thesaurus>(nodes.getAll());

        if (log.isDebugEnabled()) {
            Thesaurus thesaurus = restoreThesaurus(thesaurusId);
            log.debug("[Get children of thesaurus : " + thesaurus.getName() + "] Found " + all.size() + " children");
            for (Thesaurus child : all) {
                log.debug("[Get children of thesaurus : " + thesaurus.getName() + "] Found " + child.getName());
            }
        }

        return all;
    }

    /**
     * Return the children root thesaurus of one pass in param
     * @param rootThesaurusId id of root thesaurus
     * @return list of children
     */
    public static List<Thesaurus> getChildrenOfRootThesaurus(String rootThesaurusId) {

        Search query = Search.query();
        query.eq(Element.ELT_EXTENSION, Thesaurus.EXT_THESAURUS);
        query.eq(Thesaurus.FQ_FIELD_TREENODE_PARENT, rootThesaurusId);

        Criteria criteria = query.criteria();
        PagedResult<Thesaurus> nodes = getProxy()
                .findAllByCriteria(Thesaurus.class, criteria);
        List<Thesaurus> all = new ArrayList<Thesaurus>(nodes.getAll());

        if (log.isDebugEnabled()) {
            RootThesaurus rootThesaurus = restoreRootThesaurus(rootThesaurusId);
            log.debug("[Get children of root thesaurus : " + rootThesaurus.getName() + "] Found " + all.size() + " children");
            for (Thesaurus child : all) {
                log.debug("[Get children of root thesaurus : " + rootThesaurus.getName() + "] Found " + child.getName());
            }
        }

        return all;
    }

    public static List<Thesaurus> getAllThesaurus() {

        Criteria criteria = Search.query()
                .eq(Element.ELT_EXTENSION, ThesaurusImpl.EXT_THESAURUS)
                .criteria();

        PagedResult<Thesaurus> thesaurusResult =
                getProxy().findAllByCriteria(Thesaurus.class, criteria);

        List<Thesaurus> all = new ArrayList<Thesaurus>(thesaurusResult.getAll());
        return all;
    }

    /**
     * Return all root thesaurus
     *
     * @return list of root thesaurus
     */
    public static List<RootThesaurus> getAllRootThesaurus() {

        Criteria criteria = Search.query()
                .eq(Element.ELT_EXTENSION, RootThesaurusImpl.EXT_ROOTTHESAURUS)
                .criteria();

        PagedResult<RootThesaurus> rootThesaurusResult =
                getProxy().findAllByCriteria(RootThesaurus.class, criteria);

        List<RootThesaurus> all = new ArrayList<RootThesaurus>(rootThesaurusResult.getAll());
        if (log.isDebugEnabled()) {
            log.debug("[Get all root thesaurus] Found " + all.size() + " children");
            for (RootThesaurus child : all) {
                log.debug("[Get all root thesaurus] Found " + child.getName());
            }
        }
        return all;
    }

    /**
     * Check if name of thesaurus is existing
     *
     * @return thesaurus with name if found, else return null
     */
    public static boolean checkRootNameExist(String name) {

        List<RootThesaurus> rootThesaurusList = getAllRootThesaurus();
        for (RootThesaurus rootThesaurus : rootThesaurusList) {
            if (name.equalsIgnoreCase(rootThesaurus.getName())) {
                return true;
            }
        }
        return false;
    }

    /**
     * Check if name of thesaurus is existing
     *
     * @param rootThesaurusConcerned to check
     * @return thesaurus with name if found, else return null
     */
    public static boolean checkRootNameExist(RootThesaurus rootThesaurusConcerned) {

        String name = rootThesaurusConcerned.getName();
        String id = rootThesaurusConcerned.getWikittyId();

        List<RootThesaurus> rootThesaurusList = getAllRootThesaurus();
        for (RootThesaurus rootThesaurus : rootThesaurusList) {
            if (!id.equals(rootThesaurus.getWikittyId()) &&
                    name.equalsIgnoreCase(rootThesaurus.getName())) {
                return true;
            }
        }
        return false;
    }

    protected static boolean checkNameExist(Thesaurus thesaurus, String name) {

        if (thesaurus.getName().equalsIgnoreCase(name)){

            return true;
        }

        // Get children
        List<Thesaurus> children = getChildrenOfThesaurus(thesaurus.getWikittyId());
        for (Thesaurus child : children) {
            boolean result = checkNameExist(child, name);
            if (result){
                return true;
            }
        }
        return false;
    }

    /**
     * Check if name of thesaurus is existing
     *
     * @param rootThesaurus to check
     * @return thesaurusConcerned with name if found, else return null
     */
    public static boolean checkNameExist(RootThesaurus rootThesaurus, String name) {


        // Get children
        List<Thesaurus> children = getChildrenOfRootThesaurus(rootThesaurus.getWikittyId());
        for (Thesaurus child : children) {
            boolean result = checkNameExist(child, name);
            if (result){
                return true;
            }
        }
        return false;
    }

    /**
     * Check if name of thesaurus is existing
     *
     * @param rootThesaurus to check
     * @return thesaurusConcerned with name if found, else return null
     */
    public static boolean checkNameExist(RootThesaurus rootThesaurus, Thesaurus thesaurusConcerned) {

        String name = thesaurusConcerned.getName();
        String id = thesaurusConcerned.getWikittyId();

        // Get children
        List<Thesaurus> children = getChildrenOfRootThesaurus(rootThesaurus.getWikittyId());
        for (Thesaurus child : children) {
            boolean result = checkNameExist(child, name, id);
            if (result){
                return true;
            }
        }
        return false;
    }

    protected static boolean checkNameExist(Thesaurus thesaurus, String name, String id) {

        if (!id.equals(thesaurus.getWikittyId())
            && thesaurus.getName().equalsIgnoreCase(name)){

            return true;
        }

        // Get children
        List<Thesaurus> children = getChildrenOfThesaurus(thesaurus.getWikittyId());
        for (Thesaurus child : children) {
            boolean result = checkNameExist(child, name, id);
            if (result){
                return true;
            }
        }
        return false;
    }

    public static String getRecursifName(Thesaurus thesaurus) {
        return getRecursifName(thesaurus, concateOrderName(thesaurus));
    }

    protected static String getRecursifName(Thesaurus thesaurus, String name) {

        StringBuilder builder = new StringBuilder(name);

        List<Thesaurus> children = getChildrenOfThesaurus(thesaurus.getWikittyId());
        for (Thesaurus child : children) {
            builder.append(", ");
            builder.append(getRecursifName(child));
        }
        return builder.toString();
    }

    public static String getParentPath(Thesaurus thesaurus, String pathSeparator) {
        if (thesaurus == null) {
            return pathSeparator;
        }
        if (isFirstChild(thesaurus)) {
            RootThesaurus rootThesaurus = restoreRootThesaurus(thesaurus.getRootThesaurus());
            return rootThesaurus.getName();
        }
        String parentId = thesaurus.getParent();
        Thesaurus parentThesaurus = restoreThesaurus(parentId);
        return getParentPath(parentThesaurus, pathSeparator) + pathSeparator + concateOrderName(parentThesaurus);
    }

    public static int getDepth(Thesaurus thesaurus){
        if (thesaurus == null) {
            return 0;
        }
        if (isFirstChild(thesaurus)) {
            return 2;
        }
        String parentId = thesaurus.getParent();
        Thesaurus parent = restoreThesaurus(parentId);
        return 1 + getDepth(parent);
    }

    /**
     * Get all children of thesaurus recursively
     * Note : thesaurus in param will be added to children list
     *
     * @param thesaurus thesaurus to start
     * @return list of all children
     */
    public static List<String> getChildrenRecursif(Thesaurus thesaurus) {
        return getChildrenRecursif(thesaurus, true);
    }

    protected static List<String> getChildrenRecursif(Thesaurus thesaurus, boolean putFirst) {
        List<String> result = new ArrayList<String>();
        if (putFirst) {
            result.add(thesaurus.getWikittyId());
        }
        for (Thesaurus child : getChildrenOfThesaurus(thesaurus.getWikittyId())) {
            result.addAll(getChildrenRecursif(child, false));
        }
        return result;
    }

    /**
     * Get all children of root thesaurus recursively
     *
     * @param thesaurus thesaurus to start
     * @return list of all children
     */
    public static List<String> getChildrenRecursif(RootThesaurus thesaurus) {
        List<String> result = new ArrayList<String>();
        for (Thesaurus child : getChildrenOfRootThesaurus(thesaurus.getWikittyId())) {
            result.addAll(getChildrenRecursif(child));
        }
        return result;
    }

    public static boolean isFirstChild(Thesaurus thesaurus) {
        String rootThesaurus = thesaurus.getRootThesaurus();
        String parent = thesaurus.getParent();
        if (parent == null) {
            return false;
        }
        return parent.equals(rootThesaurus);
    }

    public static String getRequestName(Thesaurus thesaurus) {
        String rootId = thesaurus.getRootThesaurus();
        RootThesaurus rootThesaurus = restoreRootThesaurus(rootId);
        return rootThesaurus.getName() + ":" + thesaurus.getName();
    }
}