/*
 * *##%
 * vradi-services
 * Copyright (C) 2009 JurisMarches
 *
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as
 * published by the Free Software Foundation, either version 3 of the
 * License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Lesser Public License for more details.
 *
 * You should have received a copy of the GNU General Public
 * License along with this program.  If not, see
 * <http://www.gnu.org/licenses/gpl-3.0.html>.
 * ##%*
 */
package com.jurismarches.vradi.services.dto;

import com.jurismarches.vradi.entities.Client;
import com.jurismarches.vradi.entities.ClientBean;
import com.jurismarches.vradi.entities.ModificationTag;
import com.jurismarches.vradi.entities.VradiUser;
import com.jurismarches.vradi.services.ServiceHelper;
import com.jurismarches.vradi.services.VradiStorageService;
import com.jurismarches.vradi.services.VradiStorageServiceImpl;
import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.sharengo.exceptions.TechnicalException;
import org.sharengo.wikitty.FieldType;
import org.sharengo.wikitty.TreeNode;
import org.sharengo.wikitty.WikittyExtension;

import java.beans.PropertyChangeListener;
import java.beans.PropertyChangeSupport;
import java.util.*;

/**
 * User: sletellier
 * Date: 30 dec. 2009
 * Time: 12:23:27
 */
public class VradiThesaurusDTO implements VradiDTO<TreeNode> {

    private static final long serialVersionUID = 1L;

    /**
     * Logger
     */
    static private final Log log = LogFactory.getLog(VradiThesaurusDTO.class);

    protected PropertyChangeSupport propertyChange = new PropertyChangeSupport(this);

    public static final String EXT_THESAURUS = "Thesaurus";
    public static final String VERSION_THESAURUS = "1.0";
    public static final String TAGS_FIELD = "tags";
    public static final String FQ_TAGS_FIELD = "Thesaurus.tags";
    public static final String COMMENT_FIELD = "comment";
    public static final String FQ_COMMENT_FIELD = "Thesaurus.comment";

    protected boolean toSave = false;
    protected boolean toCreate = false;
    protected boolean toDelete = false;
    protected TreeNode bean;
    protected String name;
    protected String wikittyId;
    protected int formsForThesaurus = 0;
    protected String comment;

    protected VradiThesaurusDTO parentThesaurus;

    protected String tagAsString;

    protected List<String> tags = new ArrayList<String>();

    protected Map<String, VradiThesaurusDTO> children = new HashMap<String, VradiThesaurusDTO>();

    public VradiThesaurusDTO() {
    }

    public boolean isToDelete() {
        return toDelete;
    }

    public void setToDelete(boolean toDelete) {
        boolean toDeleteOld = this.toDelete;
        this.toDelete = toDelete;
        for (VradiThesaurusDTO child : getChildren()) {
            child.setToDelete(toDelete);
        }
        propertyChange.firePropertyChange("toDelete", toDeleteOld, toDelete);
    }

    public boolean isToCreate() {
        return toCreate;
    }

    public void setToCreate(boolean toCreate) {
        boolean old = this.toCreate;
        this.toCreate = toCreate;
        propertyChange.firePropertyChange("toCreate", old, toCreate);
    }

    public boolean isToSave() {
        return toSave;
    }

    public void setToSave(boolean toSave) {
        boolean old = this.toSave;
        this.toSave = toSave;
        propertyChange.firePropertyChange("toDelete", old, toSave);
    }

    public String getName() {
        return name;
    }

    public void setName(String name) {
        String old = this.name;
        this.name = name;
        propertyChange.firePropertyChange("name", old, name);
    }

    public String getWikittyId() {
        return wikittyId;
    }

    public void setWikittyId(String wikittyId) {
        String old = this.wikittyId;
        this.wikittyId = wikittyId;
        propertyChange.firePropertyChange("wikittyId", old, wikittyId);
    }

    public VradiThesaurusDTO getParentThesaurus() {
        return parentThesaurus;
    }

    public void setParentThesaurus(VradiThesaurusDTO parentThesaurus) {
        String oldPath = getParentPath();
        VradiThesaurusDTO old = this.parentThesaurus;
        this.parentThesaurus = parentThesaurus;
        propertyChange.firePropertyChange("parentThesaurus", old, parentThesaurus);
        propertyChange.firePropertyChange("parentPath", oldPath, getParentPath());
    }

    public String getTagsAsString() {
        return tagAsString;
    }

    public void setTagsAsString(String values) {
        String old = this.tagAsString;
        tagAsString = values;
        propertyChange.firePropertyChange("tagAsString", old, tagAsString);
    }

    protected List<String> getTags() {
        return tags;
    }

    protected void setTags(List<String> tags) {
        this.tags = tags;
    }

    protected void addTag(String tag) {
        this.tags.add(tag);
    }

    public int getFormsForThesaurus() {
        // Number of forms atached to thesaurus
        // TODO : Find other method to refresh
        VradiStorageService service = ServiceHelper.getVradiStorageService();
        try {
            formsForThesaurus = service.getNbFormsForThesaurus(wikittyId);
        } catch (TechnicalException e) {
            log.error("Cant find number of froms assiciated : ", e);
        }
        return formsForThesaurus;
    }

    public void setFormsForThesaurus(int formsForThesaurus) {
        this.formsForThesaurus = formsForThesaurus;
    }

    public String getComment() {
        return comment;
    }

    public void setComment(String comment) {
        String old = this.comment;
        this.comment = comment;
        propertyChange.firePropertyChange("comment", old, comment);
    }

    public Collection<VradiThesaurusDTO> getChildren() {
        return children == null ? Collections.<VradiThesaurusDTO>emptyList() : children.values();
    }

    public void addChild(VradiThesaurusDTO element) {
        Map<String, VradiThesaurusDTO> old = this.children;
        children.put(element.getWikittyId(), element);
        propertyChange.firePropertyChange("children", old, children);
    }

    public void clearChildren() {
        Map<String, VradiThesaurusDTO> old = this.children;
        children.clear();
        propertyChange.firePropertyChange("children", old, children);
    }

    public void removeChild(VradiThesaurusDTO child) {
        Map<String, VradiThesaurusDTO> old = this.children;
        children.remove(child.getWikittyId());
        propertyChange.firePropertyChange("children", old, children);
    }

    public boolean addChildRecursif(VradiThesaurusDTO child) {
        VradiThesaurusDTO p = child.getParentThesaurus();
        if (bean.getWikittyId().equals(p.getWikittyId())) {
            addChild(child);
            return true;
        } else {
            for (VradiThesaurusDTO c : getChildren()) {
                return c.addChildRecursif(child);
            }
        }
        return false;
    }

    public boolean removeThesaurusRecursivly(VradiThesaurusDTO toRemove) {
        if (children.get(toRemove.getWikittyId()) != null) {
            removeChild(toRemove);
            return true;
        } else {
            for (VradiThesaurusDTO c : getChildren()) {
                return c.removeThesaurusRecursivly(toRemove);
            }
        }
        return false;
    }

    public VradiThesaurusDTO findThesaurus(String id) {
        VradiThesaurusDTO thesaurus = children.get(id);
        if (thesaurus != null) {
            return thesaurus;
        } else {
            for (VradiThesaurusDTO c : children.values()) {
                return c.findThesaurus(id);
            }
        }
        return null;
    }

    public String getRecursifName() {
        return getRecursifName(getName());
    }

    protected String getRecursifName(String name) {
        Collection<VradiThesaurusDTO> children = getChildren();
        for (VradiThesaurusDTO child : children) {
            name += ", " + child.getRecursifName();
        }
        return name;
    }

    public String getParentPath() {
        String result = "";
        if (getParentThesaurus() != null && !getParentThesaurus().getName().equals(VradiStorageServiceImpl.ROOT_THESAURUS_NAME)) {
            result += getParentThesaurus().getParentPath() + "/" + getParentThesaurus().getName();
        }
        return result;
    }

    public static final WikittyExtension getThesaurusExtension() {
        LinkedHashMap<String, FieldType> fields = new LinkedHashMap<String, FieldType>();
        fields.put(TAGS_FIELD, new FieldType(FieldType.TYPE.STRING, 0, FieldType.NOLIMIT));
        fields.put(COMMENT_FIELD, new FieldType(FieldType.TYPE.STRING, 0, 1));
        WikittyExtension thesaurusExtension = new WikittyExtension(EXT_THESAURUS, VERSION_THESAURUS,
                TreeNode.EXT_TREENODE, fields);
        return thesaurusExtension;
    }

    public void revertFromWikitty(){
        reset();
        fromWikitty(bean);
    }

    @Override
    public void fromWikitty(TreeNode wikitty){
        // Attach bean
        this.bean = wikitty;
        setName(wikitty.getName());
        this.setWikittyId(wikitty.getWikittyId());

        // Extentions
        if (wikitty.getExtensionNames().contains(EXT_THESAURUS)) {
            // Tags
            setTags((List<String>) wikitty.getField(EXT_THESAURUS, TAGS_FIELD));

            // Comment
            setComment((String) wikitty.getField(EXT_THESAURUS, COMMENT_FIELD));

            // Tags as a string
            String tagAsString = "";
            if (tags != null) {
                for (String tag : tags) {
                    tagAsString += tag + ", ";
                }
                if (!tagAsString.isEmpty()) {
                    tagAsString = tagAsString.substring(0, tagAsString.length() - 2);
                }
            }
            setTagsAsString(tagAsString);
        }

        // Get service
        VradiStorageService service = ServiceHelper.getVradiStorageService();
        try {
            // Number of forms atached to thesaurus
            formsForThesaurus = service.getNbFormsForThesaurus(wikittyId);

            // Get children
            List<TreeNode> childrenTreeNode = service.getChildrenThesaurus(wikittyId);

            // Creating thesaurus recursivly
            for (TreeNode child : childrenTreeNode){
                VradiThesaurusDTO childThesaurus = new VradiThesaurusDTO();
                childThesaurus.fromWikitty(child, this);
                addChild(childThesaurus);
            }
        } catch (TechnicalException e) {
            log.error("Cant find thesaurus ", e);
        }
    }

    // Clone recursively
    public VradiThesaurusDTO clone(){
        VradiThesaurusDTO clone = new VradiThesaurusDTO();
        clone.setWikittyId(wikittyId);
        clone.setToCreate(toCreate);
        clone.setToSave(toSave);
        clone.setComment(comment);
        clone.setName(name);
        clone.setFormsForThesaurus(formsForThesaurus);
        clone.setParentThesaurus(parentThesaurus);
        for (VradiThesaurusDTO child : getChildren()){
            clone.addChild(child.clone());
        }
        clone.setTags(tags);
        clone.setTagsAsString(tagAsString);
        return clone;
    }

    private void fromWikitty(TreeNode treeNodeChild, VradiThesaurusDTO vradiThesaurusDTO) {
        fromWikitty(treeNodeChild);
        setParentThesaurus(vradiThesaurusDTO);
    }

    @Override
    public void toWikitty(TreeNode wikitty) {
        // Resore name
        wikitty.setName(getName());
        // Resore parent
        VradiThesaurusDTO parentThesaurus = getParentThesaurus();
        // If it's root
        if (parentThesaurus != null){
            wikitty.setParent(parentThesaurus.getWikittyId());
        }
        // Resore extentions
        if (!wikitty.getExtensionNames().contains(EXT_THESAURUS)) {
            wikitty.addExtension(getThesaurusExtension());
        }

        // Resore tags from tags as string
        if (tags == null){
            tags = new ArrayList<String>();
        } else {
            tags.clear();
        }
        if (tagAsString != null) {
            for (String tag : tagAsString.split(",")) {
                addTag(tag.trim());
            }
        }
        wikitty.setField(EXT_THESAURUS, TAGS_FIELD, tags);
        // Resore comment
        wikitty.setField(EXT_THESAURUS, COMMENT_FIELD, comment);
    }

    // TODO : Remove it ?
    public TreeNode getBean() {
        toWikitty(bean);
        return bean;
    }

    @Override
    public void reset() {
        setWikittyId(null);
        setName(null);
        setTags(null);
        setComment(null);
        setToDelete(false);
        setToSave(false);
        setToCreate(false);
        setFormsForThesaurus(0);
        children =  Collections.emptyMap();
    }

    @Override
    public void addPropertyChangeListener(PropertyChangeListener listener) {
        propertyChange.addPropertyChangeListener(listener);
    }

    @Override
    public void removePropertyChangeListener(PropertyChangeListener listener) {
        propertyChange.removePropertyChangeListener(listener);
    }

    @Override
    public void addPropertyChangeListener(String propertyName, PropertyChangeListener listener) {
        propertyChange.addPropertyChangeListener(propertyName, listener);
    }

    @Override
    public void removePropertyChangeListener(String propertyName, PropertyChangeListener listener) {
        propertyChange.removePropertyChangeListener(propertyName, listener);
    }
}
