/*
 * #%L
 * Wikitty :: wikitty-solr-impl
 * 
 * $Id: SolrUtil.java 650 2010-12-23 11:44:57Z sletellier $
 * $HeadURL: http://svn.nuiton.org/svn/wikitty/tags/wikitty-3.0/wikitty-solr-impl/src/main/java/org/nuiton/wikitty/storage/solr/SolrUtil.java $
 * %%
 * Copyright (C) 2009 - 2010 CodeLutin, Benjamin Poussin
 * %%
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Lesser General Public License as 
 * published by the Free Software Foundation, either version 3 of the 
 * License, or (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Lesser Public License for more details.
 * 
 * You should have received a copy of the GNU General Lesser Public 
 * License along with this program.  If not, see
 * <http://www.gnu.org/licenses/lgpl-3.0.html>.
 * #L%
 */
package org.nuiton.wikitty.storage.solr;

import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.apache.solr.client.solrj.SolrQuery;
import org.apache.solr.client.solrj.SolrServer;
import org.apache.solr.client.solrj.SolrServerException;
import org.apache.solr.client.solrj.response.QueryResponse;
import org.apache.solr.common.SolrDocument;
import org.apache.solr.common.SolrDocumentList;
import org.apache.solr.common.SolrInputDocument;
import org.nuiton.wikitty.WikittyException;
import org.nuiton.wikitty.entities.FieldType.TYPE;

import java.util.Collection;
import java.util.HashSet;
import java.util.Set;

import static org.nuiton.wikitty.storage.solr.WikittySolrConstant.SOLR_ID;
import static org.nuiton.wikitty.storage.solr.WikittySolrConstant.SUFFIX_BINARY;
import static org.nuiton.wikitty.storage.solr.WikittySolrConstant.SUFFIX_BOOLEAN;
import static org.nuiton.wikitty.storage.solr.WikittySolrConstant.SUFFIX_DATE;
import static org.nuiton.wikitty.storage.solr.WikittySolrConstant.SUFFIX_NUMERIC;
import static org.nuiton.wikitty.storage.solr.WikittySolrConstant.SUFFIX_STRING;
import static org.nuiton.wikitty.storage.solr.WikittySolrConstant.SUFFIX_WIKITTY;

/**
 *
 * @author poussin
 * @version $Revision: 650 $
 *
 * Last update: $Date: 2010-12-23 12:44:57 +0100 (jeu., 23 déc. 2010) $
 * by : $Author: sletellier $
 */
public class SolrUtil {

    /** to use log facility, just put in your code: log.info(\"...\"); */
    static private Log log = LogFactory.getLog(SolrUtil.class);

    /**
     * Find solr document by id
     */
    static public SolrDocument findById(SolrServer solrServer, String id) {
        SolrQuery query = new SolrQuery(SOLR_ID + ":" + id);
        QueryResponse response;
        try {
            response = solrServer.query(query);
        } catch (SolrServerException eee) {
            throw new WikittyException("Error during find", eee);
        }

        SolrDocumentList results = response.getResults();
        long numFound = results.getNumFound();
        if(numFound == 1) {
            return results.get(0);
        }

        return null;
    }

    /**
     * if you change this method, change
     * {@link TypeFieldModifer#convertToField(org.nuiton.wikitty.services.WikittyTransaction, java.lang.String)}
     * too
     *
     * @param fqfieldName
     * @param type
     * @return
     */
    static public String getSolrFieldName(String fqfieldName, TYPE type) {
        switch (type) {
            case BINARY:
                return fqfieldName + SUFFIX_BINARY;
            case BOOLEAN:
                return fqfieldName + SUFFIX_BOOLEAN;
            case DATE:
                return fqfieldName + SUFFIX_DATE;
            case STRING:
                return fqfieldName + SUFFIX_STRING;
            case NUMERIC:
                return fqfieldName + SUFFIX_NUMERIC;
            case WIKITTY:
                return fqfieldName + SUFFIX_WIKITTY;
            default:
                return fqfieldName;
        }
    }

    /**
     * copy all field of source in new document.
     * If include is true copy only field specified in fields
     * if include is false copy all field except field in fields.
     *
     * example:
     * if doc contains field: abc, aabbcc, aaabbbccc, toto
     * copySolrDocument(doc, true, "aa.*", ".*bbb.*")
     * field copied are: aabbcc, aaabbbccc
     *
     * copySolrDocument(doc, false, "aa.*", ".*bbb.*")
     * field copied are: abc, toto
     *
     * @param source
     * @param include
     * @param fields
     * @return
     */
    static public SolrInputDocument copySolrDocument(
            SolrDocument source, boolean include, String... fields) {
        SolrInputDocument result = new SolrInputDocument();
        Collection<String> fieldNames = source.getFieldNames();

        Set<String> fieldToCopy = new HashSet<String>();
        if (include) {
            for (String fieldName : fieldNames) {
                for (String fieldRegexp : fields) {
                    if (fieldName.matches(fieldRegexp)) {
                        fieldToCopy.add(fieldName);
                        break;
                    }
                }
            }
        } else { // exclude
            fieldToCopy.addAll(fieldNames);
            for (String fieldName : fieldNames) {
                for (String fieldRegexp : fields) {
                    if (fieldName.matches(fieldRegexp)) {
                        fieldToCopy.remove(fieldName);
                        break;
                    }
                }
            }
        }

        if (log.isDebugEnabled()) {
            log.debug(String.format(
                    "Copiable field are %s but only field %s are copied",
                    fieldNames, fieldToCopy));
        }

        for (String fieldName : fieldToCopy) {
            Collection<Object> fieldValues = source.getFieldValues(fieldName);
            for (Object fieldValue : fieldValues) {
                result.addField(fieldName, fieldValue);
            }
        }
        return result;
    }
}
