/*
 * #%L
 * Wikitty :: wikitty-solr-impl
 * 
 * $Id: WikittySearchEngineSolrIndexInTreeNode.java 650 2010-12-23 11:44:57Z sletellier $
 * $HeadURL: http://svn.nuiton.org/svn/wikitty/tags/wikitty-3.0.1/wikitty-solr-impl/src/main/java/org/nuiton/wikitty/storage/solr/WikittySearchEngineSolrIndexInTreeNode.java $
 * %%
 * Copyright (C) 2009 - 2010 CodeLutin, Benjamin POUSSIN
 * %%
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Lesser General Public License as 
 * published by the Free Software Foundation, either version 3 of the 
 * License, or (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Lesser Public License for more details.
 * 
 * You should have received a copy of the GNU General Lesser Public 
 * License along with this program.  If not, see
 * <http://www.gnu.org/licenses/lgpl-3.0.html>.
 * #L%
 */

package org.nuiton.wikitty.storage.solr;

import static org.nuiton.wikitty.storage.solr.WikittySolrConstant.*;

import java.util.ArrayList;
import java.util.Collection;
import java.util.HashMap;
import java.util.HashSet;
import java.util.Iterator;
import java.util.List;
import java.util.Map;
import java.util.Set;

import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.apache.solr.client.solrj.SolrQuery;
import org.apache.solr.client.solrj.SolrServer;
import org.apache.solr.client.solrj.SolrServerException;
import org.apache.solr.client.solrj.embedded.EmbeddedSolrServer;
import org.apache.solr.client.solrj.response.FacetField;
import org.apache.solr.client.solrj.response.QueryResponse;
import org.apache.solr.common.SolrDocument;
import org.apache.solr.common.SolrDocumentList;
import org.apache.solr.common.SolrInputDocument;
import org.apache.solr.core.CoreContainer;
import org.nuiton.wikitty.search.Criteria;
import org.nuiton.wikitty.search.FacetTopic;
import org.nuiton.wikitty.entities.FieldType;
import org.nuiton.wikitty.entities.FieldType.TYPE;
import org.nuiton.wikitty.search.PagedResult;
import org.nuiton.wikitty.entities.WikittyTreeNode;
import org.nuiton.wikitty.entities.Wikitty;
import org.nuiton.wikitty.WikittyException;
import org.nuiton.wikitty.storage.WikittyExtensionStorage;
import org.nuiton.wikitty.storage.WikittySearchEngine;
import org.nuiton.wikitty.services.WikittyTransaction;

import java.io.File;
import java.util.Collections;
import java.util.LinkedHashSet;
import java.util.LinkedList;
import org.nuiton.util.ApplicationConfig;
import org.nuiton.wikitty.WikittyConfig;
import org.nuiton.wikitty.WikittyUtil;
import org.nuiton.wikitty.entities.WikittyTreeNodeHelper;
import org.nuiton.wikitty.search.Search;

/**
 *
 * @author poussin
 * @version $Revision: 650 $
 *
 * Last update: $Date: 2010-12-23 12:44:57 +0100 (jeu., 23 déc. 2010) $
 * by : $Author: sletellier $
 */
public class WikittySearchEngineSolrIndexInTreeNode implements WikittySearchEngine {

    /** to use log facility, just put in your code: log.info(\"...\"); */
    static private Log log = LogFactory.getLog(WikittySearchEngineSolr.class);

//    static final public String[] fieldNotToCopyPattern = {
//        Pattern.quote(TREENODE_PREFIX) + ".*"
//    };

    /** pattern to copy field from solr document to another solr document
     *  copy field s_c and s_t are not copied
     *  tree fields are not copied
     *  #all. fields are not copied
     */
    static final public String[] fieldToCopyPattern = {
         // match: id, extensions, not_null_fields
        SOLR_ID, SOLR_EXTENSIONS, SOLR_NOT_NULL_FIELDS,
        // match: ".*_bi" accepte ce qui fini par _bi
        ".*" + SUFFIX_BINARY,
        ".*" + SUFFIX_BOOLEAN,
        ".*" + SUFFIX_DATE,
        ".*" + SUFFIX_NUMERIC,
        ".*" + SUFFIX_STRING,
        ".*" + SUFFIX_WIKITTY
    };

    /**
     * NOTE: On ne pourra utiliser ces patterns pour la copie que lorsque la config
     * solr acceptera de creer des copyField avec des expressions regulieres
     * Ce qui permettra de genere les champs #all.* via la config solr et non
     * pas de devoir les ajouter via le code Java.
     * <copyField source="[^.]+\.(.*)"   dest="#all.#1"/>
     */
    static final public String[] fieldToCopyPatternWithExcludeAll = {
         // match: id, extensions, not_null_fields
        SOLR_ID, SOLR_EXTENSIONS, SOLR_NOT_NULL_FIELDS,
        // match: "(?!(all\.)).*_bi" accept ce qui fini par _bi sauf si ca commence par "all."
        "(?!(" + SOLR_ALL_EXTENSIONS + "\\.)).*" + SUFFIX_BINARY,
        "(?!(" + SOLR_ALL_EXTENSIONS + "\\.)).*" + SUFFIX_BOOLEAN,
        "(?!(" + SOLR_ALL_EXTENSIONS + "\\.)).*" + SUFFIX_DATE,
        "(?!(" + SOLR_ALL_EXTENSIONS + "\\.)).*" + SUFFIX_NUMERIC,
        "(?!(" + SOLR_ALL_EXTENSIONS + "\\.)).*" + SUFFIX_STRING,
        "(?!(" + SOLR_ALL_EXTENSIONS + "\\.)).*" + SUFFIX_WIKITTY
    };

    /** solr server */
    protected SolrServer solrServer;

    /** Field modifier use to transform to solr format */
    protected TypeFieldModifier fieldModifier;

    /** JTA resource */
    protected SolrResource solrResource;

    /**
     * Init wikitty search engine on solr embedded server.
     * 
     * @param extensionStorage extension storage
     * @param properties properties (can be null)
     */
    public WikittySearchEngineSolrIndexInTreeNode(
            ApplicationConfig config, WikittyExtensionStorage extensionStorage) {

        // init system env solr.data.dir
        if (config != null) {
            // choix du storage (file or Ram)
            String solrDirFactoryKey =
                    WikittyConfig.WikittyOption.WIKITTY_SEARCHENGINE_SOLR_DIRECTORY_FACTORY.getKey();
            String solrDirFactory = config.getOption(solrDirFactoryKey);
            if (solrDirFactory != null) {
                System.setProperty(solrDirFactoryKey, solrDirFactory);
            }

            // on utilise le directory que si on est pas en Ram
            if (solrDirFactory != null && !solrDirFactory.contains("RAMDirectoryFactory")) {
                String solrDataDirKey =
                        WikittyConfig.WikittyOption.WIKITTY_SEARCHENGINE_SOLR_DIRECTORY_DATA.getKey();
                String solrDataDir = config.getOption(solrDataDirKey);
                // make sure that dir exists
                if (solrDataDir != null) {
                    File file = new File(solrDataDir);
                    file.mkdirs();
                    System.setProperty(solrDataDirKey, solrDataDir);
                }
            }
        }

        try {
            CoreContainer.Initializer initializer = new CoreContainer.Initializer();
            CoreContainer coreContainer = initializer.initialize();
            solrServer = new EmbeddedSolrServer(coreContainer, "");

            fieldModifier = new TypeFieldModifier(extensionStorage);
            solrResource = new SolrResource(solrServer);
            
        } catch (Exception eee) {
            throw new WikittyException("SolR initialization error", eee);
        }
    }

    @Override
    public void clear(WikittyTransaction transaction) {
        try {
            // FIXME poussin 20100618 pourquoi n'est pas fait dans la transaction ?
            solrResource.init();
            solrServer.deleteByQuery("*:*");
        } catch (Exception eee) {
            throw new WikittyException("Error during clearing SolR data", eee);
        }
    }

    @Override
    public void store(WikittyTransaction transaction, Collection<Wikitty> wikitties) {
        try {
            solrResource.init();
            LinkedHashSet<String> treeNodeToIndex = new LinkedHashSet<String>();
            for (Wikitty w : wikitties) {
                String id = w.getId();
                // Index
                SolrInputDocument doc = createIndexDocument(w);
                solrResource.addDoc(id, doc);

                if (WikittyTreeNodeHelper.hasExtension(w)) {
                    if (w.getDirty().contains(WikittyTreeNode.FQ_FIELD_WIKITTYTREENODE_PARENT)
                            || null == WikittyTreeNodeHelper.getParent(w)
                            || WikittyUtil.versionGreaterThan("1", w.getVersion())) {
                        // si le pere a change
                        // ou qu'il est null (creation d'un nouvel arbre)
                        // ou que l'objet n'a jamais ete sauve 1 > version
                        // il faut indexer le noeud
                        treeNodeToIndex.add(id);
                    } else {
                        // on recupere l'ancienne indexation
                        SolrDocument oldDoc = SolrUtil.findById(solrServer, id);
                        String root = (String)oldDoc.getFieldValue(TREENODE_ROOT);
                        Collection parents = oldDoc.getFieldValues(TREENODE_PARENTS);

                        doc.addField(TREENODE_ROOT, root);
                        for (Object parent : parents) {
                            doc.addField(TREENODE_PARENTS, parent);
                        }
                    }
                }

            }

            // Reindex child in tree node
            reindexTreeNode(solrResource, treeNodeToIndex);
        } catch (Exception eee) {
            throw new WikittyException("Can't store wikitty", eee);
        }
    }

    protected String getTreeRoot(SolrResource solrResource, SolrServer solrServer,
            String node) throws Exception {
        String result = null;
        String parent = node;
        while (parent != null) {
            SolrInputDocument doc = solrResource.getAddedDoc(parent);
            if (doc != null) {
                result = (String)doc.getFieldValue(TREENODE_ROOT);
                parent = (String) doc.getFieldValue(WikittyTreeNode
                        .FQ_FIELD_WIKITTYTREENODE_PARENT + SUFFIX_WIKITTY);
            } else {
                SolrDocument parentDoc = SolrUtil.findById(solrServer, parent);
                result = (String) parentDoc.getFirstValue(TREENODE_ROOT);
                parent = (String) parentDoc.getFieldValue(WikittyTreeNode
                        .FQ_FIELD_WIKITTYTREENODE_PARENT + SUFFIX_WIKITTY);
            }
        }
        // on a pas retrouve le root, cela veut dire que node et le root
        if (result == null) {
            result = node;
        }
        return result;
    }

    /**
     * Reindexe les noeuds passe en parametre et reindexe automatiquement leurs
     * sous noeuds
     *
     * @param solrResource
     * @param treeNodeToIndex
     * @throws Exception
     */
    protected void reindexTreeNode(SolrResource solrResource,
            LinkedHashSet<String> treeNodeToIndex) throws Exception {
        LinkedList<String> todo = new LinkedList<String>(treeNodeToIndex);
        Set<String> done = new HashSet<String>();

        Set<String> deleted = new HashSet<String>(solrResource.getDeletedDocs());

        // key: child id, value: parent id
        HashMap<String, String> tree = new HashMap<String, String>();
        while(todo.size() > 0) {
            String id = todo.poll();
            if (done.contains(id)) {
                continue;
            }
            done.add(id);

            // on recherche tout les fils du noeud que l'on reindex pour les reindexer aussi
            SolrQuery query = new SolrQuery(SOLR_QUERY_PARSER + TREENODE_PARENTS + ":" + id);
            QueryResponse response = solrServer.query(query);
            SolrDocumentList updateDocs = response.getResults();
            for (Iterator<SolrDocument> iterator = updateDocs.iterator(); iterator.hasNext();) {
                SolrDocument solrDocument = iterator.next();
                String childId = (String) solrDocument.getFieldValue(SOLR_ID);
                todo.offer(childId);
            }

            if (deleted.contains(id)) {
                continue;
            }

            // Get documents associated with this id
            SolrInputDocument doc = solrResource.getAddedDoc(id);
            if (doc == null) {
                // Copy old field value
                SolrDocument found = SolrUtil.findById(solrServer, id);
                if (found != null) {
                    // copy only necessary field (other are regenerated)
                    doc = SolrUtil.copySolrDocument(found, true, fieldToCopyPattern);
                    solrResource.addDoc(id, doc);
                } else {
                    if (log.isWarnEnabled()) {
                        log.warn("Can't find wikitty id '" + id + "' in index. Skip this wikitty.");
                    }
                }
            }

            // ajout du nouveau champs tree.root
            String root = getTreeRoot(solrResource, solrServer, id);
            doc.addField(TREENODE_ROOT, root);

            // ajout dans le nouveau champs tree.parents tous les parents du noeud
            // et le noeud lui meme
            String parent = id;
            while (parent != null) {
                String oldParent = parent;
                doc.addField(TREENODE_PARENTS, parent);
                if (tree.containsKey(parent)) {
                    parent = tree.get(parent);
                } else if (null != solrResource.getAddedDoc(parent)) {
                    SolrInputDocument parentDoc = solrResource.getAddedDoc(id);
                    parent = (String) parentDoc.getFieldValue(WikittyTreeNode
                            .FQ_FIELD_WIKITTYTREENODE_PARENT + SUFFIX_WIKITTY);
                } else {
                    SolrDocument parentDoc = SolrUtil.findById(solrServer, parent);
                    if (parentDoc != null) {
                        parent = (String) parentDoc.getFieldValue(WikittyTreeNode
                                .FQ_FIELD_WIKITTYTREENODE_PARENT + SUFFIX_WIKITTY);
                    } else {
                        parent = null;
                    }
                }
                tree.put(oldParent, parent);
            }
        }
    }

    @Override
    public void delete(WikittyTransaction transaction, Collection<String> ids) throws WikittyException {
        try {
            solrResource.init();
            LinkedHashSet<String> treeNodeToIndex = new LinkedHashSet<String>();
            for (String id : ids) {
                SolrDocument doc = SolrUtil.findById(solrServer, id);
                if (doc.containsKey(TREENODE_ROOT)) {
                    // child are automaticaly done in reindexTreeNode method
                    treeNodeToIndex.add(id);
                }
                solrResource.deleteDoc(id);
            }

            // Reindex child in tree node
            reindexTreeNode(solrResource, treeNodeToIndex);
        } catch (Exception eee) {
            throw new WikittyException("Can't delete wikitty in index", eee);
        }
    }

    @Override
    public PagedResult<String> findAllByCriteria(WikittyTransaction transaction, Criteria criteria) {
        try {
            // Create query with restriction
            Restriction2Solr restriction2Solr = new Restriction2Solr(transaction, fieldModifier);
            String queryString = restriction2Solr.toSolr(criteria.getRestriction(), solrServer);
            SolrQuery query = new SolrQuery(SOLR_QUERY_PARSER + queryString);

            // Add paged
            int firstIndex = criteria.getFirstIndex();
            int endIndex = criteria.getEndIndex();

            query.setStart(firstIndex);
            int nbRows;
            if (endIndex == -1) {
                // WARNING It is necessary to substract 'start' otherwise,
                // there is a capacity overlow in solR
                nbRows = Integer.MAX_VALUE - firstIndex;
            } else {
                nbRows = endIndex - firstIndex + 1;
            }
            query.setRows(nbRows);

            // Add sorting
            List<String> sortAscending = criteria.getSortAscending();
            if(sortAscending != null) {
                for (String sort : sortAscending) {
                    String tranform = fieldModifier.convertToSolr(transaction, sort);
                    query.addSortField(tranform, SolrQuery.ORDER.asc);
                }
            }
            
            List<String> sortDescending = criteria.getSortDescending();
            if(sortDescending != null) {
                for (String sort : sortDescending) {
                    String tranform = fieldModifier.convertToSolr(transaction, sort);
                    query.addSortField(tranform, SolrQuery.ORDER.desc);
                }
            }

            // Add faceting
            List<String> facetField = criteria.getFacetField();
            log.debug("facetField : " + facetField);
            List<Criteria> facetCriteria = criteria.getFacetCriteria();

            // use to map query string to criteria facet name
            Map<String, String> facetQueryToName = new HashMap<String, String>();

            if ((facetField != null && !facetField.isEmpty())
                    || (facetCriteria != null && !facetCriteria.isEmpty())) {
                query.setFacet(true);
                query.setFacetMinCount(1);
                // query.setFacetLimit(8); // no limit actualy

                // field facetisation
                if (facetField != null) {
                    for (String fqfieldName : facetField) {
                        String tranform = fieldModifier.convertToSolr(transaction, fqfieldName);
                        query.addFacetField(tranform);
                    }
                }

                // query facetisation
                if (facetCriteria != null) {
                    for (Criteria facet : facetCriteria) {
                        String queryFacet =
                                restriction2Solr.toSolr(facet.getRestriction());
                        facetQueryToName.put(queryFacet, facet.getName());
                        query.addFacetQuery(queryFacet);
                    }
                }
            }

            if(log.isDebugEnabled()) {
                log.debug(String.format("Try to execute query %s", query));
            }
            QueryResponse resp = solrServer.query(query);
            SolrDocumentList solrResults = resp.getResults();

            Map<String, List<FacetTopic>> facets = new HashMap<String, List<FacetTopic>>();
            if (facetField != null && !facetField.isEmpty()) {
                for (FacetField facet : resp.getFacetFields()) {
                    String facetName = fieldModifier.convertToField(transaction, facet.getName());
                    List<FacetTopic> topics = new ArrayList<FacetTopic>();
                    if (facet.getValues() != null) {
                        for (FacetField.Count value : facet.getValues()) {
                            String topicName = value.getName();
//                            if(!topicName.endsWith(TREENODE_EMPTY)) {
                                int topicCount = (int) value.getCount();
                                FacetTopic topic = new FacetTopic(facetName, topicName, topicCount);
                                topics.add(topic);
//                            }
                        }
                    }
                    facets.put(facetName, topics);
                }
            }
            if (facetCriteria != null && !facetCriteria.isEmpty()) {
                for (Map.Entry<String, Integer> facet : resp.getFacetQuery().entrySet()) {
                    String facetName = facet.getKey();
                    // don't use contains because, map can have key with null value
                    if (null != facetQueryToName.get(facetName)) {
                        facetName = facetQueryToName.get(facetName);
                    }
                    Integer count = facet.getValue();
                    List<FacetTopic> topics = new ArrayList<FacetTopic>();
                    FacetTopic topic = new FacetTopic(facetName, facetName, count);
                    topics.add(topic);
                    facets.put(facetName, topics);
                }
            }

            List<String> ids = new ArrayList<String>(solrResults.size());
            for (SolrDocument doc : solrResults) {
                String id = (String) doc.getFieldValue(SOLR_ID);
                ids.add(id);
            }

            int numFound = (int)resp.getResults().getNumFound();
            PagedResult<String> result = new PagedResult<String>(
                    firstIndex, numFound, queryString, facets, ids);

            return result;
        } catch (SolrServerException eee) {
            throw new WikittyException("Error during find", eee);
        }
    }

    /**
     * Cette implantation n'est pas vraiment optimale en place memoire si
     * le 'filter' retourne enormement de resultat ou qu'il y a beaucoup
     * d'attachment dans l'arbre.
     *
     * On a en memoire l'ensemble des ids des attachments du sous arbre et
     * l'ensemble des ids du resultat du filter.
     *
     * @param transaction
     * @param w
     * @param filter
     * @return
     */
    @Override
    public Integer findNodeCount(WikittyTransaction transaction, Wikitty w, Criteria filter) {
        try {
            int result = 0;

            String id = w.getId();
            // we used hashSet to count only once same attachments in many node
            Collection allAttachments = new HashSet();

            // on recherche tout les fils du noeud
            SolrQuery query = new SolrQuery(SOLR_QUERY_PARSER + TREENODE_PARENTS + ":" + id);
            QueryResponse response = solrServer.query(query);
            SolrDocumentList updateDocs = response.getResults();
            for (Iterator<SolrDocument> iterator = updateDocs.iterator(); iterator.hasNext();) {
                SolrDocument solrDocument = iterator.next();
                Collection attachments = solrDocument.getFieldValues(
                        WikittyTreeNode.FQ_FIELD_WIKITTYTREENODE_ATTACHMENT
                        + SUFFIX_WIKITTY);
                if (attachments != null) {
                    allAttachments.addAll(attachments);
                }
            }
            if (filter != null) {
                Search search = Search.query(filter);
                Search and = search.and().contains(SOLR_ID, allAttachments);
                Criteria criteria = search.criteria();

                PagedResult<String> criteriaResult = findAllByCriteria(transaction, filter);
                allAttachments.retainAll(criteriaResult.getAll());
            }
            result = allAttachments.size();
            return result;
        } catch (SolrServerException eee) {
            throw new WikittyException("Can't find node count", eee);
        }
    }

    /**
     * Cette implantation n'est pas vraiment optimale en place memoire si
     * le 'filter' retourne enormement de resultat ou qu'il y a beaucoup
     * d'attachment dans l'arbre.
     *
     * On a en memoire l'ensemble des ids des attachments du sous arbre et
     * l'ensemble des ids du resultat du filter.
     *
     * @param transaction
     * @param w
     * @param filter
     * @return
     */
    @Override
    public Map<String, Integer> findAllChildrenCount(
            WikittyTransaction transaction, Wikitty w, Criteria filter) {
        try {
            // key: id node; value: attachment count for this node (with sub*node)
            Map<String, Integer> result = new HashMap<String, Integer>();

            String id = w.getId();
                // key: id node; value: attachment count for this node (with sub*node)
            Map<String, Collection> allAttachments = new HashMap<String, Collection>();

            // les id des fils direct du wikitty passe en parametre
            Set<String> child = new HashSet<String>();

            // on recherche tout les fils et sous fils, lui compris
            SolrQuery query = new SolrQuery(SOLR_QUERY_PARSER + TREENODE_PARENTS + ":" + id);
            QueryResponse response = solrServer.query(query);
            SolrDocumentList updateDocs = response.getResults();

            for (Iterator<SolrDocument> iterator = updateDocs.iterator(); iterator.hasNext();) {
                SolrDocument solrDocument = iterator.next();
                String childId = (String) solrDocument.getFieldValue(SOLR_ID);
                String parentId = (String) solrDocument.getFieldValue(
                        WikittyTreeNode.FQ_FIELD_WIKITTYTREENODE_PARENT + SUFFIX_WIKITTY);
                Collection attachments = solrDocument.getFieldValues(
                        WikittyTreeNode.FQ_FIELD_WIKITTYTREENODE_ATTACHMENT + SUFFIX_WIKITTY);
                Collection parents = solrDocument.getFieldValues(
                        TREENODE_PARENTS);

                if (id.equals(parentId)) {
                    // c'est un fils direct on l'ajoute a la liste des fils
                    child.add(childId);
                }

                for (Object p : parents) {
                    String parent = (String) p;
                    Collection col = allAttachments.get(parent);
                    if (col == null) {
                        // we used hashSet to count only once same attachments in many node
                        col = new HashSet();
                        allAttachments.put(parent, col);
                    }
                    if (attachments != null) {
                        col.addAll(attachments);
                    }
                }
            }

            // creation des id repondants au filtre si besoin
            Collection<String> filteredId = Collections.EMPTY_LIST;
            if (filter != null) {

                // stocke toutes les ids des attachments du sous arbres
                Collection<String> attachmentIds = new LinkedList<String>();
                // 1ere passe, on ne garde que les fils directs.
                // comme ca se sera deja fait pour la construction du result
                // moins de chose a reparcourir
                for (Iterator<Map.Entry<String, Collection>> i = allAttachments.entrySet().iterator(); i.hasNext();) {
                    Map.Entry<String, Collection> e = i.next();
                    // on ne garde que les fils directs
                    if (child.contains(e.getKey())) {
                        attachmentIds.addAll(e.getValue());
                    } else {
                        i.remove();
                    }
                }

                // ca ne sert a rien de faire la requete si au final, il n'y a
                // aucun attachment
                if (attachmentIds.size() > 0) {                    
                    Search search = Search.query(filter);
                    Search and = search.and().contains(SOLR_ID, attachmentIds);
                    Criteria criteria = search.criteria();

                    PagedResult<String> criteriaResult = findAllByCriteria(transaction, criteria);
                    filteredId = criteriaResult.getAll();
                }
            }

            // Construction du resultat
            for (Map.Entry<String, Collection> e : allAttachments.entrySet()) {
                // on ne garde que les fils directs
                // c'est peut-etre fait si filter est non null, mais pas sinon
                if (child.contains(e.getKey())) {
                    Collection attachments = e.getValue();
                    // on ne garde dans le comptage des attachments que les id
                    // qui reponde aussi au filter
                    attachments.retainAll(filteredId);
                    result.put(e.getKey(), attachments.size());
                }
            }

            return result;
        } catch (SolrServerException eee) {
            throw new WikittyException("Can't find node count", eee);
        }

    }

    /**
     * Create all index document to used to modify indexation.
     * this method don't modify index.
     * 
     * The document looks like :
     * SolrId : wikittyId
     * extensions : extensionNames
     * fieldName : fieldValue
     *
     * @param w all wikitties object to index
     * @return solrInputDocument used to modify index
     */
    protected SolrInputDocument createIndexDocument(Wikitty w) {
        if (log.isDebugEnabled()) {
            log.debug("index wikitty " + w.getId());
        }

        SolrInputDocument doc = new SolrInputDocument();
        String id = w.getId();
        doc.addField(SOLR_ID, id);

        for (String name : w.getExtensionNames()) {
            doc.addField(SOLR_EXTENSIONS, name);
        }

        for (String fqfieldName : w.fieldNames()) {
            FieldType fieldType = w.getFieldType(fqfieldName);
            TYPE type = fieldType.getType();
            String solrFqFieldName = SolrUtil.getSolrFieldName(fqfieldName, type);
            
            String solrAllFieldName = SOLR_ALL_EXTENSIONS
                    + WikittyUtil.FQ_FIELD_NAME_SEPARATOR
                    + WikittyUtil.getFieldNameFromFQFieldName(solrFqFieldName);
            
            Object objectValue = w.getFqField(fqfieldName);
            if(objectValue != null) {
                if (fieldType.isCollection()) {
                    Collection collectionValue = (Collection) objectValue;
                    for (Object itemValue : collectionValue) {
                        if (itemValue != null) {
                            doc.addField(solrFqFieldName, itemValue);
                            doc.addField(solrAllFieldName, itemValue);                            
                            doc.addField(SOLR_NOT_NULL_FIELDS, fqfieldName);
                            if (log.isDebugEnabled()) {
                                log.debug("index field " + solrFqFieldName +
                                        " with value '" + itemValue + "'");
                            }
                        }
                    }
                } else {
                    doc.addField(solrFqFieldName, objectValue);
                    doc.addField(solrAllFieldName, objectValue);
                    doc.addField(SOLR_NOT_NULL_FIELDS, fqfieldName);
                    if (log.isDebugEnabled()) {
                        log.debug("index field " + solrFqFieldName +
                                " with value '" + objectValue + "'");
                    }
                }
            }
        }
        return doc;
    }

}
