/* *##%
 * Copyright (c) 2009 morin. All rights reserved.
 * 
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 *##%*/

package org.nuiton.wikitty.jdbc;

import static org.nuiton.wikitty.jdbc.WikittyJDBCUtil.COL_FIELDTYPE;
import static org.nuiton.wikitty.jdbc.WikittyJDBCUtil.COL_ID;
import static org.nuiton.wikitty.jdbc.WikittyJDBCUtil.COL_NAME;
import static org.nuiton.wikitty.jdbc.WikittyJDBCUtil.COL_REQUIRES;
import static org.nuiton.wikitty.jdbc.WikittyJDBCUtil.COL_TAGVALUES;
import static org.nuiton.wikitty.jdbc.WikittyJDBCUtil.COL_VERSION;
import static org.nuiton.wikitty.jdbc.WikittyJDBCUtil.QUERY_CLEAR_EXTENSION;
import static org.nuiton.wikitty.jdbc.WikittyJDBCUtil.QUERY_CREATION_EXTENSION_ADMIN_TEST;
import static org.nuiton.wikitty.jdbc.WikittyJDBCUtil.QUERY_CREATION_EXTENSION_ADMIN;
import static org.nuiton.wikitty.jdbc.WikittyJDBCUtil.QUERY_CREATION_EXTENSION_DATA_TEST;
import static org.nuiton.wikitty.jdbc.WikittyJDBCUtil.QUERY_CREATION_EXTENSION_DATA;
import static org.nuiton.wikitty.jdbc.WikittyJDBCUtil.QUERY_INSERT_EXTENSION_ADMIN;
import static org.nuiton.wikitty.jdbc.WikittyJDBCUtil.QUERY_INSERT_EXTENSION_DATA;
import static org.nuiton.wikitty.jdbc.WikittyJDBCUtil.QUERY_SELECT;
import static org.nuiton.wikitty.jdbc.WikittyJDBCUtil.QUERY_SELECT_WHERE;
import static org.nuiton.wikitty.jdbc.WikittyJDBCUtil.TABLE_EXTENSION_ADMIN;
import static org.nuiton.wikitty.jdbc.WikittyJDBCUtil.TABLE_EXTENSION_DATA;
import static org.nuiton.wikitty.jdbc.WikittyJDBCUtil.commitJDBCConnection;
import static org.nuiton.wikitty.jdbc.WikittyJDBCUtil.doQuery;
import static org.nuiton.wikitty.jdbc.WikittyJDBCUtil.getConnection;
import static org.nuiton.wikitty.jdbc.WikittyJDBCUtil.getJDBCConnection;
import static org.nuiton.wikitty.jdbc.WikittyJDBCUtil.loadProperties;
import static org.nuiton.wikitty.jdbc.WikittyJDBCUtil.rollbackJDBCConnection;

import java.sql.Connection;
import java.sql.PreparedStatement;
import java.sql.ResultSet;
import java.sql.SQLException;
import java.sql.Statement;
import java.util.ArrayList;
import java.util.Collection;
import java.util.HashMap;
import java.util.LinkedHashMap;
import java.util.List;
import java.util.Map;
import java.util.Properties;

import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.nuiton.wikitty.FieldType;
import org.nuiton.wikitty.UpdateResponse;
import org.nuiton.wikitty.WikittyException;
import org.nuiton.wikitty.WikittyExtension;
import org.nuiton.wikitty.WikittyExtensionStorage;
import org.nuiton.wikitty.WikittyTransaction;
import org.nuiton.wikitty.WikittyUtil;

/**
 *
 * @author morin
 * @version $Revision: 311 $
 *
 * Last update: $Date: 2010-09-15 12:14:23 +0200 (mer., 15 sept. 2010) $
 * by : $Author: jruchaud $
 */
public class WikittyExtensionStorageJDBC implements WikittyExtensionStorage {

    /** to use log facility, just put in your code: log.info(\"...\"); */
    static protected Log log = LogFactory.getLog(WikittyExtensionStorageJDBC.class);
    
    /** Properties file */
    protected final Properties conf;
    
    /** cache for extension key: ext id (extname[extversion]) value: WikittyExtension */
    protected Map<String, WikittyExtension> extensionCache =
            new HashMap<String, WikittyExtension>();
    
    /** cache for last extension version; key: extName value: extVersion */
    protected Map<String, String> lastVersion = null;

    public WikittyExtensionStorageJDBC() {
        this(null);
    }
    
    public WikittyExtensionStorageJDBC(Properties properties) {
        conf = loadProperties(properties);
        Connection connectionTest = getJDBCConnection(conf);
        try {
            // If test of existance work, no exception and do nothing
            // if exception try to create databse
            Statement statementTest = connectionTest.createStatement();
            statementTest.execute(conf.getProperty(QUERY_CREATION_EXTENSION_ADMIN_TEST));
            statementTest.execute(conf.getProperty(QUERY_CREATION_EXTENSION_DATA_TEST));
        } catch(SQLException silentError) {
            if (log.isInfoEnabled()) {
                log.info("try to create extension database");
            }
            Connection connection = getConnection(conf);
            try {
                Statement statement = connection.createStatement();
                statement.execute(conf.getProperty(QUERY_CREATION_EXTENSION_ADMIN));
                statement.execute(conf.getProperty(QUERY_CREATION_EXTENSION_DATA));
                commitJDBCConnection(connection);
            } catch (SQLException eee) {
                rollbackJDBCConnection(connection);
                throw new WikittyException(eee);
            } finally {
                WikittyJDBCUtil.closeQuietly(connection);
            }
        } finally {
            WikittyJDBCUtil.closeQuietly(connectionTest);
        }
    }

    @Override
    public UpdateResponse store(WikittyTransaction transaction,
            Collection<WikittyExtension> extensions)
            throws WikittyException {
        Connection connection = getConnection(conf);
        try {
            for (WikittyExtension ext : extensions) {
                // extension id is extension name with version
                String id = ext.getId();
                //select all the data with the id "id"
                String query = String.format(conf.getProperty(QUERY_SELECT_WHERE),
                        COL_VERSION, TABLE_EXTENSION_ADMIN, COL_ID);
                PreparedStatement statement = connection.prepareStatement(query);
                statement.setString(1, id);
                ResultSet versionResultSet = statement.executeQuery();
                
                //if the data is not already recorded
                if (!versionResultSet.next()) {
                    doQuery(connection, conf.getProperty(QUERY_INSERT_EXTENSION_ADMIN),
                            ext.getId(),
                            ext.getName(),
                            ext.getVersion(),
                            ext.getRequires(),
                            WikittyUtil.tagValuesToString(ext.getTagValues()));
                    for (String fieldName : ext.getFieldNames()) {
                        FieldType type = ext.getFieldType(fieldName);
                        doQuery(connection, conf.getProperty(QUERY_INSERT_EXTENSION_DATA),
                                ext.getId(), fieldName, type.toDefinition(fieldName));
                    }
                } else {
                    if (log.isInfoEnabled()) {
                        log.info("The extension is found " + id);
                    }
                }
            }

            // FIXME poussin 20100114 put that in abstractService after commit, because must remove cache only when tx is commited
            // modification in extension, remove the cache
            lastVersion = null;
            // nothing to put in result
            UpdateResponse result = new UpdateResponse();
            return result;
        } catch (SQLException eee) {
            throw new WikittyException(eee);
        } finally {
            WikittyJDBCUtil.closeQuietly(connection);
        }
    }

    @Override
    public boolean exists(WikittyTransaction transaction, String id) {
        Connection connection = getConnection(conf);
        try {
            //select the data with teh id "id" in the admin table
            String q = String.format(conf.getProperty(QUERY_SELECT_WHERE),
                    COL_ID, TABLE_EXTENSION_ADMIN, COL_ID);
            PreparedStatement statement = connection.prepareStatement(q);
            statement.setString(1, id);
            ResultSet resultSet = statement.executeQuery();

            boolean result = resultSet.next();
            return result;
        } catch (SQLException eee) {
            throw new WikittyException(eee);
        } finally {
            WikittyJDBCUtil.closeQuietly(connection);
        }
    }

    @Override
    public List<String> getAllExtensionIds(WikittyTransaction transaction) {
        Connection connection = getConnection(conf);
        try {
            List<String> result = new ArrayList<String>();
            Statement statement = connection.createStatement();
            //get all extensions names and versions
            ResultSet resultSet = statement.executeQuery(
                    String.format(conf.getProperty(QUERY_SELECT),
                    COL_ID, TABLE_EXTENSION_ADMIN));

            while (resultSet.next()) {
                String id = resultSet.getString(COL_ID);
                result.add(id);
            }
            return result;
        } catch (SQLException eee) {
            throw new WikittyException(eee);
        } finally {
            WikittyJDBCUtil.closeQuietly(connection);
        }
    }

    @Override
    public List<String> getAllExtensionsRequires(WikittyTransaction transaction,
            String extensionName) {
        Connection connection = getConnection(conf);
        try {
            List<String> result = new ArrayList<String>();
            String q = String.format(conf.getProperty(QUERY_SELECT_WHERE),
                    COL_ID, TABLE_EXTENSION_ADMIN, COL_REQUIRES);
            
            PreparedStatement statement = connection.prepareStatement(q);
            statement.setString(1, extensionName);
            ResultSet resultSet = statement.executeQuery();
            
            while (resultSet.next()) {
                String id = resultSet.getString(COL_ID);
                result.add(id);
            }
            
            return result;
        } catch (SQLException eee) {
            throw new WikittyException(eee);
        } finally {
            WikittyJDBCUtil.closeQuietly(connection);
        }
    }

    /**
     * return last version available for specified extension name
     * @param name name of extension
     * @return last version availble for this version, or null if extension
     * doesn't exist
     */
    @Override
    public String getLastVersion(WikittyTransaction transaction,
            String extName) {
        if (lastVersion == null) {
            // create cache for futur call
            lastVersion = new HashMap<String, String>();
            Connection connection = getConnection(conf);
            try {
                Statement statement = connection.createStatement();
                //get all extensions names and versions
                ResultSet resultSet = statement.executeQuery(
                        String.format(conf.getProperty(QUERY_SELECT),
                        COL_NAME + "," + COL_VERSION, TABLE_EXTENSION_ADMIN));
                while (resultSet.next()) {
                    String name = resultSet.getString(COL_NAME);
                    String version = resultSet.getString(COL_VERSION);
                    String prevVersion = lastVersion.get(name);
                    //if the version or this row is greater than the versions of teh already visited rows
                    if (prevVersion == null ||
                            WikittyUtil.versionGreaterThan(version, prevVersion)) {
                        lastVersion.put(name, version);
                    }
                }
            } catch (SQLException eee) {
                throw new WikittyException(eee);
            } finally {
                WikittyJDBCUtil.closeQuietly(connection);
            }
        }
        String result = lastVersion.get(extName);
        return result;
    }

    @Override
    public WikittyExtension restore(WikittyTransaction transaction, String name, String version)
            throws WikittyException {
        String id = WikittyExtension.computeId(name, version);
        WikittyExtension result = extensionCache.get(id);
        if (result == null) {
            Connection connection = getConnection(conf);
            try {

                //get the data with the id "id" in the admin table
                String q = String.format(conf.getProperty(QUERY_SELECT_WHERE),
                        "*", TABLE_EXTENSION_ADMIN, COL_ID);
                PreparedStatement statement = connection.prepareStatement(q);
                statement.setString(1, id);
                ResultSet adminResultSet = statement.executeQuery();

                if (adminResultSet.next()) {
                    String extName = adminResultSet.getString(COL_NAME);
                    String extVersion = adminResultSet.getString(COL_VERSION);
                    
                    LinkedHashMap<String, FieldType> fieldTypes =
                            new LinkedHashMap<String, FieldType>();

                    //get the data with the id "id" in the data table
                    String qdata = String.format(conf.getProperty(QUERY_SELECT_WHERE),
                            "*", TABLE_EXTENSION_DATA, COL_ID);
                    PreparedStatement sta = connection.prepareStatement(qdata);
                    sta.setString(1, id);
                    ResultSet dataResultSet = sta.executeQuery();

                    while (dataResultSet.next()) {
                        String fieldDef = dataResultSet.getString(COL_FIELDTYPE);
                        FieldType fieldType = new FieldType();
                        String fieldName = WikittyUtil.parseField(fieldDef, fieldType);
                        fieldTypes.put(fieldName, fieldType);
                    }
                    
                    String extRequires = adminResultSet.getString(COL_REQUIRES);
                    result = new WikittyExtension(extName, extVersion, extRequires, fieldTypes);

                    String extTagValues = adminResultSet.getString(COL_TAGVALUES);
                    result.setTagValues(WikittyUtil.tagValuesToMap(extTagValues));

                    // synchronisation pour supporter le multithreading
                    synchronized (extensionCache) {
                        // FIXME poussin 20100114 put that in abstractService after commit, because must in cache only when tx is commited
                        extensionCache.put(id, result);
                    }
                }
                
            } catch (SQLException eee) {
                throw new WikittyException(String.format("Can't load extension %s", id), eee);
            } finally {
                WikittyJDBCUtil.closeQuietly(connection);
            }
        }
        return result;
    }

    @Override
    public void clear(WikittyTransaction transaction) {
        Connection connection = getConnection(conf);
        try {
            lastVersion = null;
            doQuery(connection, conf.getProperty(QUERY_CLEAR_EXTENSION));

        } catch (Exception eee) {
            throw new WikittyException(eee);
        } finally {
            WikittyJDBCUtil.closeQuietly(connection);
        }
    }
}
