/* *##%
 * Copyright (c) 2010 poussin. All rights reserved.
 * 
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 *##%*/

package org.nuiton.wikitty;

import java.util.Collection;
import java.util.HashMap;
import java.util.Map;

import org.apache.commons.collections.map.ReferenceMap;
import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;

/**
 * Cette classe sert a introduire du cache dans wikitty. Elle sert a centraliser
 * tous les appels au cache pour pouvoir simplement changer de librairie de
 * cache si necessaire, meme si pour l'instant on s'appuie le JDK
 * 
 * @author poussin
 * @version $Revision$
 *
 * Last update: $Date$
 * by : $Author$
 */
public class WikittyCache implements WikittyServiceListener {

    static private Log log = LogFactory.getLog(WikittyCache.class);
    
    /** keys are wikitty ids */
    protected Map<String, Wikitty> wikittyCache;

    /**
     * Create a soft cache.
     */
    public WikittyCache() {
        this(true);
    }

    /**
     * Init cache with specific reference type.
     * 
     * @param soft si vrai utilise un cache avec des reference Soft, sinon
     * utilise des references static (utile pour les transactions)
     */
    public WikittyCache(boolean soft) {
        if (soft) {
            wikittyCache = new ReferenceMap(ReferenceMap.HARD, ReferenceMap.SOFT);
        } else {
            wikittyCache = new HashMap<String, Wikitty>();
        }
    }

    public boolean existsWikitty(String id) {
        Object o = wikittyCache.get(id);
        boolean result = (o != null);
        return result;
    }

    /**
     * Return wikitty object if is in the cache, null otherwize.
     * 
     * @param id
     * @return wikitty object or null
     */
    public Wikitty getWikitty(String id) {
        Wikitty result = wikittyCache.get(id);
        return result;
    }

    /**
     * Only realy put wikitty in cache, if not in cache or version is newer than
     * one in cache
     * @param e
     */
    public void putWikitty(Wikitty e) {
        if (e == null) {
            return;
        }
        Wikitty old = getWikitty(e.getId());
        if (old == null
                || WikittyUtil.versionGreaterThan(e.getVersion(), old.getVersion())) {
            wikittyCache.put(e.getId(), e);

            if (log.isTraceEnabled()) {
                log.trace("Replace cached wikitty : new version " + e.getVersion() +
                        " > old version " + (old == null ? null : old.getVersion()));
            }
        }
        else {
            if (log.isTraceEnabled()) {
                log.trace("Ignoring putWikittyEvent : new version " + e.getVersion() + " < old version " + old.getVersion());
            }
        }
    }

    /**
     * batched put
     * @param all
     */
    public void putAllWikitty(Collection<Wikitty> all) {
        for (Wikitty w : all) {
            putWikitty(w);
        }
    }

    /**
     * Remove wikitty from cache.
     * 
     * @param id wikitty id to remove
     */
    public void removeWikitty(String id) {
        wikittyCache.remove(id);
    }

    /**
     * batched remove
     * @param ids
     */
    public void removeAllWikitty(Collection<String> ids) {
        for (String id : ids) {
            removeWikitty(id);
        }
    }

    /**
     * Clear all cache.
     */
    public void clearWikitty() {
        wikittyCache.clear();
    }

    /*
     * @see org.nuiton.wikitty.WikittyServiceListener#clearWikitty()
     */
    @Override
    public void clearWikitty(WikittyServiceEvent event) {
        clearWikitty();
    }

    /*
     * @see org.nuiton.wikitty.WikittyServiceListener#putWikitty(org.nuiton.wikitty.Wikitty[])
     */
    @Override
    public void putWikitty(WikittyServiceEvent event) {
        
        Collection<String> ids = event.getIds();
        Map<String, String> idsVersions = event.getIdVersions();

        for (String id : ids) {
            // put on server, remove it from local cache
            Wikitty currentWikitty = getWikitty(id);
            if (currentWikitty != null) {
                String version = idsVersions.get(id);
                if (WikittyUtil.versionGreaterThan(version, currentWikitty.getVersion())) {
                    removeWikitty(id);
                }
            }
        }

    }

    /*
     * @see org.nuiton.wikitty.WikittyServiceListener#removeWikitty(java.lang.String[])
     */
    @Override
    public void removeWikitty(WikittyServiceEvent event) {
        for (String id : event.getIds()) {
            removeWikitty(id);
        }
    }

    /*
     * @see org.nuiton.wikitty.WikittyServiceListener#putExtension(org.nuiton.wikitty.WikittyExtension[])
     */
    @Override
    public void putExtension(WikittyServiceEvent event) {
        
    }

    /*
     * @see org.nuiton.wikitty.WikittyServiceListener#removeExtension(java.lang.String[])
     */
    @Override
    public void removeExtension(WikittyServiceEvent event) {
        
    }

    /*
     * @see org.nuiton.wikitty.WikittyServiceListener#clearExtension()
     */
    @Override
    public void clearExtension(WikittyServiceEvent event) {
        
    }
}
