/* *##% 
 * ToPIA :: SOA
 * Copyright (C) 2004 - 2009 CodeLutin
 *
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Lesser General Public License as
 * published by the Free Software Foundation, either version 3 of the
 * License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Lesser Public License for more details.
 *
 * You should have received a copy of the GNU General Lesser Public
 * License along with this program.  If not, see
 * <http://www.gnu.org/licenses/lgpl-3.0.html>.
 * ##%*/
package org.nuiton.topia.generator;

import org.nuiton.eugene.models.object.*;
import static org.nuiton.topia.generator.TopiaGeneratorUtil.hasUnidirectionalRelationOnAbstractType;

import java.io.File;
import java.io.IOException;
import java.io.Writer;
import java.util.Collection;
import java.util.Iterator;
import java.util.Set;

import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.nuiton.eugene.models.object.ObjectModelGenerator;
import org.nuiton.eugene.GeneratorUtil;
import org.nuiton.topia.service.TopiaApplicationServiceAbstract;
import org.apache.commons.lang.StringUtils;


/**
* ServiceAbstractGenerator.java
*
* Created: 19 juin 2007
*
* @author dupont
* @version $Revision: 1732 $
*
* Le service abstrait etend TopiaApplicationServiceAbstract et implemente
* l'interface du service.
* 
* Genere l'implantation des methodes qui utilisent les methodes des DAO 
* et celles des entites.
*
* @see ServiceInterfaceGenerator
* @see TopiaApplicationServiceAbstract
*
* Mise a jour: $Date: 2009-12-20 17:29:38 +0100 (dim., 20 déc. 2009) $
* par : $Author: tchemit $
* @plexus.component role="org.nuiton.eugene.Template" role-hint="org.nuiton.topia.generator.ServiceAbstractGenerator"
*/
public class ServiceAbstractGenerator extends ObjectModelGenerator {
	
    private static final Log log = LogFactory
            .getLog(ServiceAbstractGenerator.class);

    @Override
    public String getFilenameForInterface(ObjectModelInterface clazz) {
        return clazz.getQualifiedName().replace('.', File.separatorChar) + "Abstract.java";
    }

    @Override
    public void generateFromInterface(Writer output, ObjectModelInterface interfacez) throws IOException {
    	 
    	if (!interfacez.hasStereotype(TopiaGeneratorUtil.STEREOTYPE_SERVICE)) {
            return;
        }
    	
    	// generer l'entete du service abstrait
    	generateInterfaceHeader(output, interfacez);

    	// generer les signatures des methodes metier du service
    	generateInterfaceOperations(output, interfacez);
	        
    	// generer les methodes etendues en modifiant les signatures pour eviter les doublons (findAllByAddress heritee de Person et Employee)
    	generateInheritedInterfaceOperations(output, interfacez);
	
    	// generer les signatures des methodes metier des entites
    	generateAssociatedClassOperations(output, interfacez);
	       
    	// generer les signatures des methodes des entites (getter/setter)
    	generateMethodsGetter(output, interfacez);

output.write("\n");
output.write("} //"+interfacez.getName()+"\n");
output.write("");
    }

    private void generateInterfaceHeader(Writer output, ObjectModelClassifier classifier) throws IOException {
        String copyright = TopiaGeneratorUtil.getCopyright(model);
        String classifierName = classifier.getName();
        if (TopiaGeneratorUtil.notEmpty(copyright)) {
output.write(""+copyright+"\n");
output.write("");
        }
output.write("package "+classifier.getPackageName()+";\n");
output.write("\n");
output.write("import java.util.Arrays;\n");
output.write("import java.util.List;\n");
output.write("import org.nuiton.topia.TopiaException;\n");
output.write("import org.nuiton.topia.service.TopiaApplicationServiceAbstract;\n");
output.write("import org.nuiton.topia.framework.TopiaContextImplementor;");
    	
    	classifier.getInterfaces();
    	// ajouter les imports des interfaces de DAO
    	for (ObjectModelInterface parent : classifier.getInterfaces()) {
    		if(parent.hasStereotype(TopiaGeneratorUtil.STEREOTYPE_DAO)){            	
            	// récupérer la classe de l'object model correspondant à la DAO
    			String className = parent.getQualifiedName().replace("DAO", "");
            	String interfaceName = parent.getQualifiedName()+"Abstract";
output.write("\n");
output.write("import "+interfaceName+";\n");
output.write("import "+className+";");
    		}
    	}
            	
        if (TopiaGeneratorUtil.hasDocumentation(classifier)) {
output.write("\n");
output.write("/**\n");
output.write(" *\n");
output.write(" * "+classifier.getDocumentation()+"\n");
output.write(" */\n");
output.write("");
        }
output.write("\n");
output.write("\n");
output.write("public abstract class "+classifierName+"Abstract extends TopiaApplicationServiceAbstract implements "+classifierName+" {\n");
output.write("");
    }

    private void generateMethodsGetter(Writer output, ObjectModelClassifier classifier) throws IOException {
output.write("\n");
output.write("    public String[] getMethods(){\n");
output.write("    	return methods;\n");
output.write("    }\n");
output.write("");
    }
    
    private void generateInterfaceOperations(Writer output, ObjectModelClassifier classifier) throws IOException {
        for (ObjectModelOperation op : classifier.getOperations()) {
        	String opName = op.getName();
        	String opVisibility = op.getVisibility();
        	String opType = op.getReturnType();
output.write("\n");
output.write("    /**\n");
output.write("     * Implementation a la charge du developpeur\n");
output.write("");
            if (TopiaGeneratorUtil.hasDocumentation(op)) {
            	String opDocumentation = op.getDocumentation();
output.write("     * "+opName+" : "+opDocumentation+"\n");
output.write("");
            }
            Collection<ObjectModelParameter> params = op.getParameters();
            for(ObjectModelParameter param : params) {
            	String paramName = param.getName();
            	String paramDocumentation = param.getDocumentation();
output.write("     * @param "+paramName+" "+paramDocumentation+"\n");
output.write(""); 
            }
output.write("     */\n");
output.write("    "+opVisibility+" abstract "+opType+" "+opName+"(");
            String comma = "";
            for(ObjectModelParameter param : params){
            	String paramName = param.getName();
            	String paramType = param.getType();
output.write(""+comma+""+paramType+" "+paramName+"");
                comma = ", ";
            }
output.write(")");
            Set<String> exceptions = op.getExceptions();
            comma = " throws ";
            for (String exception : exceptions) {
output.write(""+comma+""+exception+"");
                comma = ", ";
            }
output.write(";\n");
output.write("");
        }
    }
    
    private void generateInheritedInterfaceOperations(Writer output, ObjectModelClassifier classifier) throws IOException {
    	
        for (ObjectModelInterface parent : classifier.getInterfaces()) {
            // pour tous les DAOInterface
            if(parent.hasStereotype(TopiaGeneratorUtil.STEREOTYPE_DAO)){
            	
            	// récupérer la classe de l'object model correspondant à la DAO
            	String entityClassName = parent.getQualifiedName().replace("DAO", "");
            	if(getModel().hasClass(entityClassName)){
            		ObjectModelClass clazz =  getModel().getClass(entityClassName);
            		String clazzName = clazz.getName();
            		String clazzFQN = clazz.getQualifiedName();
            		
            		if (!clazz.hasStereotype(TopiaGeneratorUtil.STEREOTYPE_ENTITY)) {
            			return;  
            		}
output.write("\n");
output.write("    /**\n");
output.write("     * Supprime l'entite "+clazzName+" passee en parametre\n");
output.write("     * @param entity l'entite a supprimer\n");
output.write("     */\n");
output.write("	public void delete"+clazzName+"("+clazzFQN+" entity) throws TopiaException{\n");
output.write("		TopiaContextImplementor usedContextImpl = (TopiaContextImplementor) topiaContext;\n");
output.write("		"+clazzName+"DAOAbstract dao = ("+clazzName+"DAOAbstract)usedContextImpl.getDAO("+clazzName+".class);\n");
output.write("		dao.delete(entity);\n");
output.write("	}\n");
output.write("\n");
output.write("	/**\n");
output.write("     * Creer l'entite "+clazzName+" avec les proprietes passees en parametre\n");
output.write("     * @param properties les proprietes de l'entite a creer\n");
output.write("     */\n");
output.write("    public "+clazzFQN+" create"+clazzName+"(Object ... properties) throws TopiaException{\n");
output.write("        TopiaContextImplementor usedContextImpl = (TopiaContextImplementor) topiaContext;\n");
output.write("        "+clazzName+"DAOAbstract dao = ("+clazzName+"DAOAbstract)usedContextImpl.getDAO("+clazzName+".class);\n");
output.write("        "+clazzName+" entity = ("+clazzName+") dao.create(properties);\n");
output.write("        return entity;\n");
output.write("    }\n");
output.write("    \n");
output.write("    /**\n");
output.write("     * Mise a jour de l'entite "+clazzName+" passee en parametre\n");
output.write("     * @param entity l'entite a mettre a jour\n");
output.write("     */\n");
output.write("    public "+clazzFQN+" update"+clazzName+"("+clazzFQN+" entity) throws TopiaException{\n");
output.write("        TopiaContextImplementor usedContextImpl = (TopiaContextImplementor) topiaContext;\n");
output.write("        "+clazzName+"DAOAbstract dao = ("+clazzName+"DAOAbstract)usedContextImpl.getDAO("+clazzName+".class);\n");
output.write("        "+clazzName+" entit = ("+clazzName+") dao.update(entity);\n");
output.write("        return entit;\n");
output.write("    }\n");
output.write("    \n");
output.write("	/**\n");
output.write("     * Retourne tous les "+clazzName+" \n");
output.write("     * @return une liste\n");
output.write("     */\n");
output.write("    public List<"+clazzFQN+"> findAll"+clazzName+"() throws TopiaException {\n");
output.write("		TopiaContextImplementor usedContextImpl = (TopiaContextImplementor) topiaContext;\n");
output.write("		"+clazzName+"DAOAbstract dao = ("+clazzName+"DAOAbstract)usedContextImpl.getDAO("+clazzName+".class);\n");
output.write("		List<"+clazzName+"> result = dao.findAll(); \n");
output.write("    	return result;\n");
output.write("    }\n");
output.write("    \n");
output.write("    /**\n");
output.write("     * Retourne le "+clazzName+" par son TopiaId \n");
output.write("     * @return le "+clazzName+"\n");
output.write("     */\n");
output.write("    public "+clazzFQN+" find"+clazzName+"ByTopiaId(String v) throws TopiaException {\n");
output.write("    	TopiaContextImplementor usedContextImpl = (TopiaContextImplementor) topiaContext;\n");
output.write("		"+clazzName+"DAOAbstract dao = ("+clazzName+"DAOAbstract)usedContextImpl.getDAO("+clazzName+".class);\n");
output.write("		"+clazzName+" entity = ("+clazzName+")dao.findByTopiaId(v); \n");
output.write("    	return entity;\n");
output.write("    }   \n");
output.write("");
            	    for (ObjectModelAttribute attr : clazz.getAttributes()) {
            		    if (!attr.isNavigable()) {
            		 	   continue;
                        }

            	        if (!GeneratorUtil.isNMultiplicity(attr)) {
            	            generateNoNMultiplicity(output, attr, clazz, false);
            	        } else {
            	            generateNMultiplicity(output, attr, clazz, false);
            	        }
            	    }

            	    if (clazz instanceof ObjectModelAssociationClass) {
            	        ObjectModelAssociationClass assocClass = (ObjectModelAssociationClass)clazz;
            	        for (ObjectModelAttribute attr : assocClass.getParticipantsAttributes()) {
            	            if (attr != null) {
            	                if (!GeneratorUtil.isNMultiplicity(attr)) {
            	                   generateNoNMultiplicity(output, attr, clazz, true);
            	                } else {
            	                   generateNMultiplicity(output, attr, clazz, true);
            	                }
            	            }
            	        }
            		} 	    	    
            	}
            }
        }
    }
    
    protected void generateNoNMultiplicity(Writer output, ObjectModelAttribute attr, ObjectModelClass clazz, boolean isAssoc) throws IOException {
	    String propertyName = attr.getName();
	    if (!isAssoc && attr.hasAssociationClass()) {
	        propertyName = TopiaGeneratorUtil.toLowerCaseFirstLetter(attr.getAssociationClass().getName()) + "." + propertyName;
	    }
	    String clazzName = clazz.getName();
	    String clazzFQN = clazz.getQualifiedName();
output.write("\n");
output.write("    /**\n");
output.write("     * Retourne le premier element trouve ayant comme valeur pour l'attribut\n");
output.write("     * "+attr.getName()+" le parametre\n");
output.write("     * @param v la valeur que doit avoir "+attr.getName()+"\n");
output.write("     * @return un element ou null\n");
output.write("     */\n");
output.write("    public "+clazzFQN+" find"+clazzName+"By"+StringUtils.capitalize(attr.getName())+"("+attr.getType()+" v) throws TopiaException{\n");
output.write("		TopiaContextImplementor usedContextImpl = (TopiaContextImplementor) topiaContext;\n");
output.write("		"+clazzName+"DAOAbstract dao = ("+clazzName+"DAOAbstract)usedContextImpl.getDAO("+clazzName+".class);\n");
output.write("		"+clazzName+" entity = dao.findBy"+StringUtils.capitalize(attr.getName())+"(v);\n");
output.write("    	return entity;\n");
output.write("    };\n");
output.write("    \n");
output.write("    /**\n");
output.write("     * Retourne les elements ayant comme valeur pour l'attribut\n");
output.write("     * "+attr.getName()+" le parametre\n");
output.write("     * @param v la valeur que doit avoir "+attr.getName()+"\n");
output.write("     * @return une liste\n");
output.write("     */\n");
output.write("    public List<"+clazzFQN+"> findAll"+clazzName+"By"+StringUtils.capitalize(attr.getName())+"("+attr.getType()+" v) throws TopiaException{\n");
output.write("		TopiaContextImplementor usedContextImpl = (TopiaContextImplementor) topiaContext;\n");
output.write("		"+clazzName+"DAOAbstract dao = ("+clazzName+"DAOAbstract)usedContextImpl.getDAO("+clazzName+".class);\n");
output.write("		List<"+clazzName+"> entityList = dao.findAllBy"+StringUtils.capitalize(attr.getName())+"(v);\n");
output.write("    	return entityList; \n");
output.write("    };\n");
output.write("");
        if (attr.hasAssociationClass()) {
output.write("\n");
output.write("    /**\n");
output.write("     * Retourne le premier element trouve ayant comme valeur pour l'attribut\n");
output.write("     * "+TopiaGeneratorUtil.toLowerCaseFirstLetter(attr.getAssociationClass().getName())+" le parametre\n");
output.write("     * @param value la valeur que doit avoir "+TopiaGeneratorUtil.toLowerCaseFirstLetter(attr.getAssociationClass().getName())+"\n");
output.write("     * @return un element ou null\n");
output.write("     */\n");
output.write("    public "+clazzFQN+" find"+clazzName+"By"+StringUtils.capitalize(attr.getAssociationClass().getName())+"("+attr.getAssociationClass().getQualifiedName()+" value) throws TopiaException{\n");
output.write("		TopiaContextImplementor usedContextImpl = (TopiaContextImplementor) topiaContext;\n");
output.write("		"+clazzName+"DAOAbstract dao = ("+clazzName+"DAOAbstract)usedContextImpl.getDAO("+clazzName+".class);\n");
output.write("		"+clazzName+" entity = dao.findBy"+StringUtils.capitalize(attr.getName())+"(v);\n");
output.write("    	return entity; \n");
output.write("    };\n");
output.write("    \n");
output.write("    /**\n");
output.write("     * Retourne les elements ayant comme valeur pour l'attribut\n");
output.write("     * "+TopiaGeneratorUtil.toLowerCaseFirstLetter(attr.getAssociationClass().getName())+" le param�tre\n");
output.write("     * @param value la valeur que doit avoir "+TopiaGeneratorUtil.toLowerCaseFirstLetter(attr.getAssociationClass().getName())+"\n");
output.write("     * @return une liste\n");
output.write("     */\n");
output.write("    public List<"+clazzFQN+"> findAll"+clazzName+"By"+StringUtils.capitalize(attr.getAssociationClass().getName())+"("+attr.getAssociationClass().getQualifiedName()+" value) throws TopiaException{\n");
output.write("		TopiaContextImplementor usedContextImpl = (TopiaContextImplementor) topiaContext;\n");
output.write("		"+clazzName+"DAOAbstract dao = ("+clazzName+"DAOAbstract)usedContextImpl.getDAO("+clazzName+".class);\n");
output.write("		List<"+clazzName+"> entityList = dao.findAllBy"+StringUtils.capitalize(attr.getName())+"(v);\n");
output.write("    	return entityList;\n");
output.write("    };\n");
output.write("");
	    }
	}        
	        
    protected void generateNMultiplicity(Writer output, ObjectModelAttribute attr, ObjectModelClass clazz, boolean isAssoc) throws IOException {
	    String clazzName = clazz.getName();
	    String clazzFQN = clazz.getQualifiedName();
output.write("    \n");
output.write("    /**\n");
output.write("     * Retourne le premier element trouve dont l'attribut\n");
output.write("     * "+attr.getName()+" contient le parametre\n");
output.write("     * @param v la valeur que doit contenir "+attr.getName()+"\n");
output.write("     * @return un element ou null\n");
output.write("     */\n");
output.write("    public "+clazzFQN+" find"+clazzName+"Contains"+StringUtils.capitalize(attr.getName())+"("+attr.getType()+" ... v) throws TopiaException{\n");
output.write("		TopiaContextImplementor usedContextImpl = (TopiaContextImplementor) topiaContext;\n");
output.write("		"+clazzName+"DAOAbstract dao = ("+clazzName+"DAOAbstract)usedContextImpl.getDAO("+clazzName+".class);\n");
output.write("		"+clazzName+" entity = ("+clazzName+") dao.findContainsProperties(\""+TopiaGeneratorUtil.toLowerCaseFirstLetter(attr.getName())+"\", Arrays.asList(v)); \n");
output.write("    	return entity; \n");
output.write("    };\n");
output.write("    \n");
output.write("    /**\n");
output.write("     * Retourne les elements trouve dont l'attribut\n");
output.write("     * "+attr.getName()+" contient le parametre\n");
output.write("     * @param v la valeur que doit contenir "+attr.getName()+"\n");
output.write("     * @return une liste\n");
output.write("     */\n");
output.write("    public List<"+clazzFQN+"> findAll"+clazzName+"Contains"+StringUtils.capitalize(attr.getName())+"("+attr.getType()+" ... v) throws TopiaException{\n");
output.write("		TopiaContextImplementor usedContextImpl = (TopiaContextImplementor) topiaContext;\n");
output.write("		"+clazzName+"DAOAbstract dao = ("+clazzName+"DAOAbstract)usedContextImpl.getDAO("+clazzName+".class);\n");
output.write("		List<"+clazzName+"> entityList = dao.findAllContainsProperties(\""+TopiaGeneratorUtil.toLowerCaseFirstLetter(attr.getName())+"\", Arrays.asList(v)); \n");
output.write("    	return entityList;\n");
output.write("    };\n");
output.write("");        
	}


    private void generateAssociatedClassOperations(Writer output, ObjectModelClassifier classifier) throws IOException {
        for (ObjectModelInterface parent : classifier.getInterfaces()) {
            // pour tous les DAOInterface
            if(parent.hasStereotype(TopiaGeneratorUtil.STEREOTYPE_DAO)){
            	
            	// r�cup�rer la classe de l'object model correspondant � la DAO
            	String entityClassName;
            	entityClassName = parent.getQualifiedName().replace("DAO", "");
            	if(getModel().hasClass(entityClassName)){
            		ObjectModelClass clazz =  getModel().getClass(entityClassName);
                    generateFromDAOClass(output, clazz);
            	}
            }		
        }
    }
    
    private void generateFromDAOClass(Writer output, ObjectModelClass clazz) throws IOException {
        if (!clazz.hasStereotype(TopiaGeneratorUtil.STEREOTYPE_ENTITY)) {
            return;
        }

        String clazzName = clazz.getName();
	    String clazzFQN = clazz.getQualifiedName();

        for (ObjectModelAttribute attr : clazz.getAttributes()) {
            ObjectModelAttribute reverse = attr.getReverseAttribute();
            String capitalizedAttrName = StringUtils.capitalize(attr.getName());
            if (!attr.isNavigable()
                    && !hasUnidirectionalRelationOnAbstractType(reverse, model)) {
                continue;
            }
            if (!GeneratorUtil.isNMultiplicity(attr)) {
                if (!attr.hasAssociationClass()) {
output.write("    /**\n");
output.write("");
                    if (TopiaGeneratorUtil.hasDocumentation(attr)) {
output.write("     * "+attr.getName()+" : "+attr.getDocumentation()+"\n");
output.write("");
                    }
output.write("     * @param "+GeneratorUtil.toLowerCaseFirstLetter(attr.getName())+" La valeur de l'attribut "+attr.getName()+" a positionner.\n");
output.write("     */\n");
output.write("    public void set"+capitalizedAttrName+"Of"+clazzName+"(String topiaId, "+attr.getType()+" "+GeneratorUtil.toLowerCaseFirstLetter(attr.getName())+") throws TopiaException{\n");
output.write("        TopiaContextImplementor usedContextImpl = (TopiaContextImplementor) topiaContext;\n");
output.write("		"+clazzName+"DAOAbstract dao = ("+clazzName+"DAOAbstract)usedContextImpl.getDAO("+clazzName+".class);\n");
output.write("		"+clazzName+" entity = ("+clazzName+")dao.findByTopiaId(topiaId); \n");
output.write("    	entity.set"+StringUtils.capitalize(attr.getName())+"("+GeneratorUtil.toLowerCaseFirstLetter(attr.getName())+");\n");
output.write("    	usedContextImpl.commitTransaction();\n");
output.write("    }\n");
output.write("\n");
output.write("");
output.write("    /**\n");
output.write("");
                    if (TopiaGeneratorUtil.hasDocumentation(attr)) {
output.write("     * "+attr.getName()+" : "+attr.getDocumentation()+"\n");
output.write("");
                    }
output.write("     * @return La valeur de l'attribut "+attr.getName()+".\n");
output.write("     */\n");
output.write("    public "+attr.getType()+" get"+StringUtils.capitalize(attr.getName())+"Of"+clazzName+"(String topiaId) throws TopiaException{\n");
output.write("        TopiaContextImplementor usedContextImpl = (TopiaContextImplementor) topiaContext;\n");
output.write("		"+clazzName+"DAOAbstract dao = ("+clazzName+"DAOAbstract)usedContextImpl.getDAO("+clazzName+".class);\n");
output.write("		"+clazzName+" entity = ("+clazzName+")dao.findByTopiaId(topiaId); \n");
output.write("    	return entity.get"+StringUtils.capitalize(attr.getName())+"();\n");
output.write("    }\n");
output.write("\n");
output.write("");
                } else {
                    String assocAttrName = TopiaGeneratorUtil.getAssocAttrName(attr);
                    if (log.isTraceEnabled()) { log.trace("assocAttrName: " + assocAttrName); }
output.write("    /**\n");
output.write("     * @param "+GeneratorUtil.toLowerCaseFirstLetter(attr.getAssociationClass().getName())+" La valeur de l'attribut "+attr.getAssociationClass().getName()+" � positionner.\n");
output.write("     */\n");
output.write("    public void set"+StringUtils.capitalize(assocAttrName)+"Of"+clazzName+"(String topiaId, "+attr.getAssociationClass().getQualifiedName()+" "+GeneratorUtil.toLowerCaseFirstLetter(attr.getAssociationClass().getName())+") throws TopiaException{\n");
output.write("        TopiaContextImplementor usedContextImpl = (TopiaContextImplementor) topiaContext;\n");
output.write("		"+clazzName+"DAOAbstract dao = ("+clazzName+"DAOAbstract)usedContextImpl.getDAO("+clazzName+".class);\n");
output.write("		"+clazzName+" entity = ("+clazzName+")dao.findByTopiaId(topiaId); \n");
output.write("        entity.set"+StringUtils.capitalize(assocAttrName)+"("+GeneratorUtil.toLowerCaseFirstLetter(attr.getAssociationClass().getName())+");\n");
output.write("    }\n");
output.write("\n");
output.write("    /**\n");
output.write("     * @return La valeur de l'attribut "+attr.getAssociationClass().getName()+".\n");
output.write("     */\n");
output.write("    public "+attr.getAssociationClass().getQualifiedName()+" get"+StringUtils.capitalize(assocAttrName)+"Of"+clazzName+"(String topiaId) throws TopiaException{\n");
output.write("        TopiaContextImplementor usedContextImpl = (TopiaContextImplementor) topiaContext;\n");
output.write("		"+clazzName+"DAOAbstract dao = ("+clazzName+"DAOAbstract)usedContextImpl.getDAO("+clazzName+".class);\n");
output.write("		"+clazzName+" entity = ("+clazzName+")dao.findByTopiaId(topiaId); \n");
output.write("    	return entity.get"+StringUtils.capitalize(assocAttrName)+"();\n");
output.write("    }\n");
output.write("\n");
output.write("");
                }
            } else { //NMultiplicity
                if (!attr.hasAssociationClass()) { //M�thodes remplac�es par des add/set sur les classes d'assoc
output.write("    /**\n");
output.write("");
                    if (TopiaGeneratorUtil.hasDocumentation(attr)) {
output.write("     * "+attr.getName()+" : "+attr.getDocumentation()+"\n");
output.write("");
                    }
output.write("     * @param "+GeneratorUtil.toLowerCaseFirstLetter(attr.getName())+" L'instance de "+attr.getName()+" � ajouter.\n");
output.write("     */\n");
output.write("    public void add"+StringUtils.capitalize(attr.getName())+"Of"+clazzName+"(String topiaId, "+attr.getType()+" "+GeneratorUtil.toLowerCaseFirstLetter(attr.getName())+") throws TopiaException{\n");
output.write("        TopiaContextImplementor usedContextImpl = (TopiaContextImplementor) topiaContext;\n");
output.write("		"+clazzName+"DAOAbstract dao = ("+clazzName+"DAOAbstract)usedContextImpl.getDAO("+clazzName+".class);\n");
output.write("		"+clazzName+" entity = ("+clazzName+")dao.findByTopiaId(topiaId); \n");
output.write("    	entity.add"+StringUtils.capitalize(attr.getName())+"("+GeneratorUtil.toLowerCaseFirstLetter(attr.getName())+");\n");
output.write("    }\n");
output.write("\n");
output.write("    /**\n");
output.write("");
                    if (TopiaGeneratorUtil.hasDocumentation(attr)) {
output.write("     * "+attr.getName()+" : "+attr.getDocumentation()+"\n");
output.write("");
                    }
output.write("     * @param "+GeneratorUtil.toLowerCaseFirstLetter(attr.getName())+" Les instances de "+attr.getName()+" � ajouter.\n");
output.write("     */\n");
output.write("    public void addAll"+StringUtils.capitalize(attr.getName())+"Of"+clazzName+"(String topiaId, "+(attr.isOrdered()?"java.util.List":"java.util.Collection")+"<"+attr.getType()+"> "+GeneratorUtil.toLowerCaseFirstLetter(attr.getName())+") throws TopiaException{\n");
output.write("    	TopiaContextImplementor usedContextImpl = (TopiaContextImplementor) topiaContext;\n");
output.write("		"+clazzName+"DAOAbstract dao = ("+clazzName+"DAOAbstract)usedContextImpl.getDAO("+clazzName+".class);\n");
output.write("		"+clazzName+" entity = ("+clazzName+")dao.findByTopiaId(topiaId);\n");
output.write("    	entity.addAll"+StringUtils.capitalize(attr.getName())+"("+GeneratorUtil.toLowerCaseFirstLetter(attr.getName())+");\n");
output.write("    }\n");
output.write("\n");
output.write("    /**\n");
output.write("");
                    if (TopiaGeneratorUtil.hasDocumentation(attr)) {
output.write("     * "+attr.getName()+" : "+attr.getDocumentation()+"\n");
output.write("");
                    }
output.write("     * @param "+GeneratorUtil.toLowerCaseFirstLetter(attr.getName())+" La Collection de "+attr.getName()+" � positionner.\n");
output.write("     */\n");
output.write("    public void set"+StringUtils.capitalize(attr.getName())+"Of"+clazzName+"(String topiaId, "+(attr.isOrdered()?"java.util.List":"java.util.Collection")+"<"+attr.getType()+"> "+GeneratorUtil.toLowerCaseFirstLetter(attr.getName())+") throws TopiaException{\n");
output.write("    	TopiaContextImplementor usedContextImpl = (TopiaContextImplementor) topiaContext;\n");
output.write("		"+clazzName+"DAOAbstract dao = ("+clazzName+"DAOAbstract)usedContextImpl.getDAO("+clazzName+".class);\n");
output.write("		"+clazzName+" entity = ("+clazzName+")dao.findByTopiaId(topiaId);\n");
output.write("    	entity.set"+StringUtils.capitalize(attr.getName())+"("+GeneratorUtil.toLowerCaseFirstLetter(attr.getName())+");\n");
output.write("    }\n");
output.write("\n");
output.write("    /**\n");
output.write("");
                    if (TopiaGeneratorUtil.hasDocumentation(attr)) {
output.write("     * "+attr.getName()+" : "+attr.getDocumentation()+"\n");
output.write("");
                    }
output.write("     * @param "+GeneratorUtil.toLowerCaseFirstLetter(attr.getName())+" L'instance de "+attr.getName()+" � retirer.\n");
output.write("     */\n");
output.write("    public void remove"+StringUtils.capitalize(attr.getName())+"Of"+clazzName+"(String topiaId, "+attr.getType()+" "+GeneratorUtil.toLowerCaseFirstLetter(attr.getName())+") throws TopiaException{\n");
output.write("    	TopiaContextImplementor usedContextImpl = (TopiaContextImplementor) topiaContext;\n");
output.write("		"+clazzName+"DAOAbstract dao = ("+clazzName+"DAOAbstract)usedContextImpl.getDAO("+clazzName+".class);\n");
output.write("		"+clazzName+" entity = ("+clazzName+")dao.findByTopiaId(topiaId);\n");
output.write("    	entity.remove"+StringUtils.capitalize(attr.getName())+"("+GeneratorUtil.toLowerCaseFirstLetter(attr.getName())+");\n");
output.write("    }\n");
output.write("\n");
output.write("    /**\n");
output.write("");
                    if (TopiaGeneratorUtil.hasDocumentation(attr)) {
output.write("     * "+attr.getName()+" : "+attr.getDocumentation()+"\n");
output.write("");
                    }
output.write("     * Vide la Collection de "+attr.getName()+".\n");
output.write("     */\n");
output.write("    public void clear"+StringUtils.capitalize(attr.getName())+"Of"+clazzName+"(String topiaId) throws TopiaException{\n");
output.write("    	TopiaContextImplementor usedContextImpl = (TopiaContextImplementor) topiaContext;\n");
output.write("		"+clazzName+"DAOAbstract dao = ("+clazzName+"DAOAbstract)usedContextImpl.getDAO("+clazzName+".class);\n");
output.write("		"+clazzName+" entity = ("+clazzName+")dao.findByTopiaId(topiaId);\n");
output.write("    	entity.clear"+StringUtils.capitalize(attr.getName())+"();\n");
output.write("    }\n");
output.write("\n");
output.write("");
                } else {
                    String assocAttrName = TopiaGeneratorUtil.getAssocAttrName(attr);
                    if (log.isTraceEnabled()) { log.trace("assocAttrName: " + assocAttrName); }
output.write("    /**\n");
output.write("     * @param "+GeneratorUtil.toLowerCaseFirstLetter(attr.getAssociationClass().getName())+" L'instance de "+attr.getAssociationClass().getName()+" � ajouter.\n");
output.write("     */\n");
output.write("    public void add"+StringUtils.capitalize(assocAttrName)+"Of"+clazzName+"(String topiaId, "+attr.getAssociationClass().getQualifiedName()+" "+GeneratorUtil.toLowerCaseFirstLetter(attr.getAssociationClass().getName())+") throws TopiaException{\n");
output.write("    	TopiaContextImplementor usedContextImpl = (TopiaContextImplementor) topiaContext;\n");
output.write("		"+clazzName+"DAOAbstract dao = ("+clazzName+"DAOAbstract)usedContextImpl.getDAO("+clazzName+".class);\n");
output.write("		"+clazzName+" entity = ("+clazzName+")dao.findByTopiaId(topiaId);\n");
output.write("    	entity.add"+StringUtils.capitalize(assocAttrName)+"("+GeneratorUtil.toLowerCaseFirstLetter(attr.getAssociationClass().getName())+");\n");
output.write("    }\n");
output.write("\n");
output.write("    /**\n");
output.write("     * @param "+GeneratorUtil.toLowerCaseFirstLetter(attr.getAssociationClass().getName())+" Les instances de "+attr.getAssociationClass().getName()+" � ajouter.\n");
output.write("     */\n");
output.write("    public void addAll"+StringUtils.capitalize(assocAttrName)+"Of"+clazzName+"(String topiaId, "+(attr.isOrdered()?"java.util.List":"java.util.Collection")+"<"+attr.getAssociationClass().getQualifiedName()+"> "+GeneratorUtil.toLowerCaseFirstLetter(attr.getAssociationClass().getName())+") throws TopiaException{\n");
output.write("    	TopiaContextImplementor usedContextImpl = (TopiaContextImplementor) topiaContext;\n");
output.write("		"+clazzName+"DAOAbstract dao = ("+clazzName+"DAOAbstract)usedContextImpl.getDAO("+clazzName+".class);\n");
output.write("		"+clazzName+" entity = ("+clazzName+")dao.findByTopiaId(topiaId);\n");
output.write("    	entity.addAll"+StringUtils.capitalize(assocAttrName)+"("+GeneratorUtil.toLowerCaseFirstLetter(attr.getAssociationClass().getName())+");\n");
output.write("    }\n");
output.write("\n");
output.write("    /**\n");
output.write("     * @param "+GeneratorUtil.toLowerCaseFirstLetter(attr.getAssociationClass().getName())+" La Collection de "+attr.getAssociationClass().getName()+" � positionner.\n");
output.write("     */\n");
output.write("    public void set"+StringUtils.capitalize(assocAttrName)+"Of"+clazzName+"(String topiaId, "+(attr.isOrdered()?"java.util.List":"java.util.Collection")+"<"+attr.getAssociationClass().getQualifiedName()+"> "+GeneratorUtil.toLowerCaseFirstLetter(attr.getAssociationClass().getName())+") throws TopiaException{\n");
output.write("    	TopiaContextImplementor usedContextImpl = (TopiaContextImplementor) topiaContext;\n");
output.write("		"+clazzName+"DAOAbstract dao = ("+clazzName+"DAOAbstract)usedContextImpl.getDAO("+clazzName+".class);\n");
output.write("		"+clazzName+" entity = ("+clazzName+")dao.findByTopiaId(topiaId);\n");
output.write("    	entity.set"+StringUtils.capitalize(assocAttrName)+"("+GeneratorUtil.toLowerCaseFirstLetter(attr.getAssociationClass().getName())+");\n");
output.write("    }\n");
output.write("\n");
output.write("    /**\n");
output.write("     * @param "+GeneratorUtil.toLowerCaseFirstLetter(attr.getAssociationClass().getName())+" L'instance de "+attr.getAssociationClass().getName()+" � retirer.\n");
output.write("     */\n");
output.write("    public void remove"+StringUtils.capitalize(assocAttrName)+"Of"+clazzName+"(String topiaId, "+attr.getAssociationClass().getQualifiedName()+" "+GeneratorUtil.toLowerCaseFirstLetter(attr.getAssociationClass().getName())+") throws TopiaException{\n");
output.write("    	TopiaContextImplementor usedContextImpl = (TopiaContextImplementor) topiaContext;\n");
output.write("		"+clazzName+"DAOAbstract dao = ("+clazzName+"DAOAbstract)usedContextImpl.getDAO("+clazzName+".class);\n");
output.write("		"+clazzName+" entity = ("+clazzName+")dao.findByTopiaId(topiaId);\n");
output.write("    	entity.remove"+StringUtils.capitalize(assocAttrName)+"("+GeneratorUtil.toLowerCaseFirstLetter(attr.getAssociationClass().getName())+");\n");
output.write("    }\n");
output.write("\n");
output.write("    /**\n");
output.write("     * Vide la Collection de "+attr.getAssociationClass().getName()+".\n");
output.write("     */\n");
output.write("    public void clear"+StringUtils.capitalize(assocAttrName)+"Of"+clazzName+"(String topiaId) throws TopiaException{\n");
output.write("    	TopiaContextImplementor usedContextImpl = (TopiaContextImplementor) topiaContext;\n");
output.write("		"+clazzName+"DAOAbstract dao = ("+clazzName+"DAOAbstract)usedContextImpl.getDAO("+clazzName+".class);\n");
output.write("		"+clazzName+" entity = ("+clazzName+")dao.findByTopiaId(topiaId);\n");
output.write("    	entity.clear"+StringUtils.capitalize(assocAttrName)+"();\n");
output.write("    }\n");
output.write("\n");
output.write("");
                }

                if (!attr.hasAssociationClass()) {
output.write("    /**\n");
output.write("");
                    if (TopiaGeneratorUtil.hasDocumentation(attr)) {
output.write("     * "+attr.getName()+" : "+attr.getDocumentation()+"\n");
output.write("");
                    }
output.write("     * @return La Liste de "+attr.getName()+".\n");
output.write("     */\n");
output.write("    public "+(attr.isOrdered()?"java.util.List":"java.util.Collection")+"<"+attr.getType()+"> get"+StringUtils.capitalize(attr.getName())+"Of"+clazzName+"(String topiaId) throws TopiaException{\n");
output.write("    	TopiaContextImplementor usedContextImpl = (TopiaContextImplementor) topiaContext;\n");
output.write("		"+clazzName+"DAOAbstract dao = ("+clazzName+"DAOAbstract)usedContextImpl.getDAO("+clazzName+".class);\n");
output.write("		"+clazzName+" entity = ("+clazzName+")dao.findByTopiaId(topiaId);\n");
output.write("    	return entity.get"+StringUtils.capitalize(attr.getName())+"();\n");
output.write("    }\n");
output.write("\n");
output.write("    /**\n");
output.write("     * @return Le nombre d'�l�ments de la collection "+attr.getName()+".\n");
output.write("     */\n");
output.write("    public int size"+StringUtils.capitalize(attr.getName())+"Of"+clazzName+"(String topiaId) throws TopiaException{\n");
output.write("    	TopiaContextImplementor usedContextImpl = (TopiaContextImplementor) topiaContext;\n");
output.write("		"+clazzName+"DAOAbstract dao = ("+clazzName+"DAOAbstract)usedContextImpl.getDAO("+clazzName+".class);\n");
output.write("		"+clazzName+" entity = ("+clazzName+")dao.findByTopiaId(topiaId);\n");
output.write("    	return entity.size"+StringUtils.capitalize(attr.getName())+"();\n");
output.write("    }\n");
output.write("\n");
output.write("");
                } else {
                    String assocAttrName = TopiaGeneratorUtil.getAssocAttrName(attr);
                    if (log.isTraceEnabled()) { log.trace("assocAttrName: " + assocAttrName); }
output.write("    /**\n");
output.write("     * @return La liste des attributs "+attr.getAssociationClass().getName()+".\n");
output.write("     */\n");
output.write("    public "+(attr.isOrdered()?"java.util.List":"java.util.Collection")+"<"+attr.getAssociationClass().getQualifiedName()+"> get"+StringUtils.capitalize(assocAttrName)+"Of"+clazzName+"(String topiaId) throws TopiaException{\n");
output.write("    	TopiaContextImplementor usedContextImpl = (TopiaContextImplementor) topiaContext;\n");
output.write("		"+clazzName+"DAOAbstract dao = ("+clazzName+"DAOAbstract)usedContextImpl.getDAO("+clazzName+".class);\n");
output.write("		"+clazzName+" entity = ("+clazzName+")dao.findByTopiaId(topiaId);\n");
output.write("    	return entity.get"+StringUtils.capitalize(assocAttrName)+"();\n");
output.write("    }\n");
output.write("\n");
output.write("    /**\n");
output.write("     * @return L'attribut "+attr.getAssociationClass().getName()+" associ� � la valeur <code>value</code> de l'attribut "+attr.getName()+".\n");
output.write("     */\n");
output.write("    public "+attr.getAssociationClass().getQualifiedName()+" get"+StringUtils.capitalize(assocAttrName)+"Of"+clazzName+"(String topiaId, "+attr.getType()+" value) throws TopiaException{\n");
output.write("    	TopiaContextImplementor usedContextImpl = (TopiaContextImplementor) topiaContext;\n");
output.write("		"+clazzName+"DAOAbstract dao = ("+clazzName+"DAOAbstract)usedContextImpl.getDAO("+clazzName+".class);\n");
output.write("		"+clazzName+" entity = ("+clazzName+")dao.findByTopiaId(topiaId);\n");
output.write("    	return entity.get"+StringUtils.capitalize(assocAttrName)+"(("+attr.getType()+") value);\n");
output.write("    }\n");
output.write("\n");
output.write("    /**\n");
output.write("     * @return Le nombre d'�l�ments de la collection "+attr.getName()+".\n");
output.write("     */\n");
output.write("    public int size"+StringUtils.capitalize(assocAttrName)+"Of"+clazzName+"(String topiaId) throws TopiaException{\n");
output.write("    	TopiaContextImplementor usedContextImpl = (TopiaContextImplementor) topiaContext;\n");
output.write("		"+clazzName+"DAOAbstract dao = ("+clazzName+"DAOAbstract)usedContextImpl.getDAO("+clazzName+".class);\n");
output.write("		"+clazzName+" entity = ("+clazzName+")dao.findByTopiaId(topiaId);\n");
output.write("    	return entity.size"+StringUtils.capitalize(assocAttrName)+"();\n");
output.write("    }\n");
output.write("\n");
output.write("");
                }
            }
        }

        //M�thodes d'acc�s aux attributs d'une classe d'associations
        if (clazz instanceof ObjectModelAssociationClass) {
            ObjectModelAssociationClass assoc = (ObjectModelAssociationClass)clazz;
            for (Iterator i = assoc.getParticipantsAttributes().iterator(); i.hasNext(); ) {
               ObjectModelAttribute attr = (ObjectModelAttribute) i.next();
               if (attr != null) {
                   String type = attr.getType();
                   String name = attr.getName();
                   generateAssociationAccessors(output, clazz, name, type);
                   if (attr.getReverseAttribute() == null) {
                       type = ((ObjectModelClassifier)attr.getDeclaringElement()).getQualifiedName();
                       name = attr.getDeclaringElement().getName();
                       generateAssociationAccessors(output, clazz, name, type);
                   }
               }
            }
        }

        generateInterfaceOperationsOfClass(output, clazz);

    }

    private void generateInterfaceOperationsOfClass(Writer output, ObjectModelClassifier classifier) throws IOException {
    	String classifierName = classifier.getName();
        for (ObjectModelOperation op : classifier.getOperations()) {
        	String opName = op.getName();
        	String opVisibility = op.getVisibility();
        	String opType = op.getReturnType();
output.write("    /**\n");
output.write("");
            if (TopiaGeneratorUtil.hasDocumentation(op)) {
output.write("     * "+opName+" : "+op.getDocumentation()+"\n");
output.write("");
            }
            Collection<ObjectModelParameter> params = (Collection<ObjectModelParameter>)op.getParameters();
            for(ObjectModelParameter param : params) {
            	String paramName = param.getName();
            	String paramDocumentation = param.getDocumentation();
output.write("     * @param "+paramName+" "+paramDocumentation+"\n");
output.write(" "); 
            }
output.write("     */\n");
output.write("    "+opVisibility+" "+opType+" "+opName+"Of"+classifierName+"(String topiaId");
            String comma = ",";
            for(ObjectModelParameter param : params){
            	String paramName = param.getName();
            	String paramType = param.getType();
output.write(""+comma+""+paramType+" "+paramName+"");
            }
output.write(")");
            Set<String> exceptions = (Set<String>)op.getExceptions();
            if(exceptions.isEmpty()){
output.write("throws TopiaException");     	
            }
            comma = " throws ";
            for (String exception : exceptions) {
output.write(""+comma+""+exception+"");
                comma = ", ";
            }
output.write("{\n");
output.write("		TopiaContextImplementor usedContextImpl = (TopiaContextImplementor) topiaContext;\n");
output.write("		"+classifierName+"DAOAbstract dao = ("+classifierName+"DAOAbstract)usedContextImpl.getDAO("+classifierName+".class);\n");
output.write("		"+classifierName+" entity = ("+classifierName+")dao.findByTopiaId(topiaId);\n");
output.write("");            
            if(!op.getReturnType().toString().equalsIgnoreCase("void")){
output.write("        return ");            	
            }
output.write(" entity."+opName+"(");
            comma = "";
            for(ObjectModelParameter param : params){
            	String paramName = param.getName();
output.write(""+comma+""+paramName+"");
                comma = ", ";
            }
output.write(");\n");
output.write("    }\n");
output.write("");

output.write("\n");
output.write("\n");
output.write("");
        }
    }

    private void generateAssociationAccessors(Writer output, ObjectModelClass clazz, String name, String type) throws IOException {
    	String clazzName = clazz.getName();
output.write("    /**\n");
output.write("     * @param value La valeur de l'attribut "+name+" à positionner.\n");
output.write("     */\n");
output.write("    public void set"+StringUtils.capitalize(name)+"Of"+clazzName+"(String topiaId, "+type+" value) throws TopiaException{\n");
output.write("    	TopiaContextImplementor usedContextImpl = (TopiaContextImplementor) topiaContext;\n");
output.write("		"+clazzName+"DAOAbstract dao = ("+clazzName+"DAOAbstract)usedContextImpl.getDAO("+clazzName+".class);\n");
output.write("		"+clazzName+" entity = ("+clazzName+")dao.findByTopiaId(topiaId);\n");
output.write("    	entity.set"+StringUtils.capitalize(name)+"( value);\n");
output.write("    }\n");
output.write("\n");
output.write("    /**\n");
output.write("     * @return La valeur de l'attribut "+name+".\n");
output.write("     */\n");
output.write("    public "+type+" get"+StringUtils.capitalize(name)+"Of"+clazzName+"(String topiaId) throws TopiaException{\n");
output.write("    	TopiaContextImplementor usedContextImpl = (TopiaContextImplementor) topiaContext;\n");
output.write("		"+clazzName+"DAOAbstract dao = ("+clazzName+"DAOAbstract)usedContextImpl.getDAO("+clazzName+".class);\n");
output.write("		"+clazzName+" entity = ("+clazzName+")dao.findByTopiaId(topiaId);\n");
output.write("    	return entity.get"+StringUtils.capitalize(name)+"();\n");
output.write("    }\n");
output.write("\n");
output.write("");
    }

} //ServiceAbstractGenerator
