package org.nuiton.topia;

/*
 * #%L
 * ToPIA :: Persistence
 * $Id: HibernateTopiaJpaSupport.java 2843 2013-10-11 22:40:03Z athimel $
 * $HeadURL: http://svn.nuiton.org/svn/topia/tags/topia-3.0-alpha-4/topia-persistence/src/main/java/org/nuiton/topia/HibernateTopiaJpaSupport.java $
 * %%
 * Copyright (C) 2004 - 2013 CodeLutin
 * %%
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Lesser General Public License as 
 * published by the Free Software Foundation, either version 3 of the 
 * License, or (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Lesser Public License for more details.
 * 
 * You should have received a copy of the GNU General Lesser Public 
 * License along with this program.  If not, see
 * <http://www.gnu.org/licenses/lgpl-3.0.html>.
 * #L%
 */

import java.util.Collection;
import java.util.List;
import java.util.Map;

import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.hibernate.FlushMode;
import org.hibernate.HibernateException;
import org.hibernate.Query;
import org.nuiton.topia.framework.TopiaFiresSupport;
import org.nuiton.topia.framework.TopiaUtil;

/**
 * This class is the Hibernate implementation of TopiaJpaSupport. It realizes the bridge between the JPA specification
 * and the technical choice made for its implementation : Hibernate.
 *
 * @author Arnaud Thimel <thimel@codelutin.com>
 * @since 3.0
 */
public class HibernateTopiaJpaSupport implements TopiaJpaSupport {

    private static final Log log = LogFactory.getLog(HibernateTopiaJpaSupport.class);

    // TODO AThimel 27/09/13 Javadoc
    protected TopiaHibernateSupport hibernateSupport;

    // TODO AThimel 27/09/13 Javadoc
    protected TopiaFiresSupport firesSupport;

    /**
     * This flag permits to use (or not) the flush mode when doing queries.
     * <p/>
     * The normal usage is to says yes (that's why the default value is
     * {@code true}), in that case whebn doing queries (says in method
     * {@link #findAll(String, Object...)} or {@link #find(String, int, int, Object...)})
     * it will use the flush mode {@link FlushMode#AUTO}).
     * <p/>
     * But sometimes, when doing a lot of queries (for some imports for example),
     * we do NOT want the session to be flushed each time we do a find, then you
     * can set this flag to {@code false} using the method {@link #setUseFlushMode(boolean)}
     *
     * @since 2.5
     */
    protected boolean useFlushMode = true;

    public HibernateTopiaJpaSupport(TopiaHibernateSupport hibernateSupport, TopiaFiresSupport firesSupport) {
        this.hibernateSupport = hibernateSupport;
        this.firesSupport = firesSupport;
    }

    @Override
    public void setUseFlushMode(boolean useFlushMode) {
        this.useFlushMode = useFlushMode;
    }

    protected Query prepareQuery(String jpaql, Map<String, Object> parameters) {
        Query query = hibernateSupport.getHibernateSession().createQuery(jpaql);
        for (Map.Entry<String, Object> entry : parameters.entrySet()) {
            String name = entry.getKey();
            Object value = entry.getValue();
            if (value.getClass().isArray()) {
                query.setParameterList(name, (Object[]) value);
            } else if (value instanceof Collection<?>) {
                query.setParameterList(name, (Collection<?>) value);
            } else {
                query.setParameter(name, value);
            }
        }
        // tchemit 2010-11-30 reproduce the same behaviour than before with the dao legacy
        if (useFlushMode) {
            query.setFlushMode(FlushMode.AUTO);
        }
        return query;
    }

    @Override
    public <T> List<T> findAll(String jpaql, Map<String, Object> parameters) {
        try {
            Query query = prepareQuery(jpaql, parameters);

            List result = query.list();
            result = firesSupport.fireEntitiesLoad(this, result);
            return result;
        } catch (HibernateException eee) {
            throw new TopiaException(String.format("An error occurs during query operation: %1$s : %2$s",
                    jpaql, eee.getMessage()), eee);
        } catch (RuntimeException e) {
            if (log.isWarnEnabled()) {
                log.warn("exception occurred for query \n" + jpaql + "\nwith parameters " + parameters);
            }
            throw e;
        }
    }

    @Override
    public <T> T findUnique(String jpaql, Map<String, Object> parameters) {

        // Execute query, and ask for up to 2 results only
        List<T> results = find(jpaql, 0, 1, parameters);

        // If there is more than 1 result, throw an exception
        if (results.size() > 1) {
            String message = String.format(
                    "Query '%s' returns more than 1 unique result", jpaql);
            throw new TopiaException(message);
        }

        // otherwise return the first one, or null
        T result = null;
        if (!results.isEmpty()) {
            result = results.get(0);
        }
        return result;
    }

    @Override
    public <T> List<T> find(String jpaql, int startIndex, int endIndex, Map<String, Object> parameters) {
        try {
            Query query = prepareQuery(jpaql, parameters);

            // Set bounds
            query.setFirstResult(startIndex);
            query.setMaxResults(endIndex - startIndex + 1);

            List result = query.list();
            result = firesSupport.fireEntitiesLoad(this, result);
            return result;
        } catch (HibernateException eee) {
            throw new TopiaException(String.format("An error occurs during query operation: %1$s : %2$s",
                    jpaql, eee.getMessage()), eee);
        } catch (RuntimeException e) {
            if (log.isWarnEnabled()) {
                log.warn("exception occurred for query \n" + jpaql + "\nwith parameters " + parameters);
            }
            throw e;
        }
    }

    @Override
    public int execute(String jpaql, Map<String, Object> parameters) {
        try {
            Query query = prepareQuery(jpaql, parameters);

            int result = query.executeUpdate();
            return result;
        } catch (HibernateException eee) {
            throw new TopiaException(String.format("An error occurs during execute operation: %1$s : %2$s",
                    jpaql, eee.getMessage()), eee);
        } catch (RuntimeException e) {
            if (log.isWarnEnabled()) {
                log.warn("exception occurred for query \n" + jpaql + "\nwith parameters " + parameters);
            }
            throw e;
        }
    }

    @Override
    public <T> List<T> findAll(String jpaql, Object... propertyNamesAndValues) {
        Map<String, Object> parameters = TopiaUtil.convertPropertiesArrayToMap(propertyNamesAndValues);
        List<T> result = findAll(jpaql, parameters);
        return result;
    }

    @Override
    public <T> List<T> find(String jpaql, int startIndex, int endIndex, Object... propertyNamesAndValues) {
        Map<String, Object> parameters = TopiaUtil.convertPropertiesArrayToMap(propertyNamesAndValues);
        List<T> result = find(jpaql, startIndex, endIndex, parameters);
        return result;
    }

    @Override
    public <T> T findUnique(String jpaql, Object... propertyNamesAndValues) {
        Map<String, Object> parameters = TopiaUtil.convertPropertiesArrayToMap(propertyNamesAndValues);
        T result = findUnique(jpaql, parameters);
        return result;
    }

    @Override
    public int execute(String jpaql, Object... propertyNamesAndValues) {
        Map<String, Object> parameters = TopiaUtil.convertPropertiesArrayToMap(propertyNamesAndValues);
        int result = execute(jpaql, parameters);
        return result;
    }

} // HibernateTopiaJpaSupport
