/*
 * *##% 
 * Nuiton processor maven plugin
 * Copyright (C) 2006 - 2009 CodeLutin
 *
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Lesser General Public License as
 * published by the Free Software Foundation, either version 3 of the
 * License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Lesser Public License for more details.
 *
 * You should have received a copy of the GNU General Lesser Public
 * License along with this program.  If not, see
 * <http://www.gnu.org/licenses/lgpl-3.0.html>.
 * ##%*
 */
package org.nuiton.processor.plugin;

import java.io.File;
import java.io.FileReader;
import java.io.FileWriter;
import java.util.Arrays;
import java.util.Map;
import java.util.Map.Entry;
import org.apache.maven.project.MavenProject;
import org.nuiton.plugin.AbstractPlugin;
import org.nuiton.processor.Processor;
import org.nuiton.processor.ProcessorUtil;
import org.nuiton.io.MirroredFileUpdater;
import org.nuiton.plugin.PluginHelper;

/**
 * Maven2 plugin for Nuiton Processor.
 *
 * Created: 14 avril 2006
 *
 * @author ruchaud
 * @version $Revision: 283 $
 *
 * Last update: $Date: 2009-10-10 16:04:48 +0200 (sam., 10 oct. 2009) $
 * by : $Author: tchemit $
 * 
 * @goal process
 * @requiresProject true
 */
public class ProcessorPlugin extends AbstractPlugin {

    /**
     * Dependance du projet.
     *
     * @parameter default-value="${project}"
     * @required
     * @since 1.0.0
     */
    protected MavenProject project;
    /**
     * Répertoire source
     *
     * @parameter expression="${processor.srcDir}" default-value="${basedir}/src/main/java"
     * @since 0.10
     */
    protected File srcDir;
    /**
     * Répertoire cible
     *
     * @parameter expression="${processor.destDir}" default-value="${basedir}/target/processed-sources/java"
     * @since 0.10
     */
    protected File destDir;
    /**
     * Fichiers à inclure.
     *
     * Il s'agit des expressions séparées par des virgules.
     *
     * Exemple :
     *
     * *.java,*.xml
     * 
     * @parameter expression="${processor.includes}"
     * @since 0.10
     */
    protected String includes;
    /**
     * Fichiers à exclure.
     *
     * Il s'agit des expressions spérarées par des virgules.
     *
     * Exemple :
     *
     * *.java,*.xml
     * 
     * @parameter expression="${processor.excludes}"
     * @since 0.10
     */
    protected String excludes;
    /**
     * ???
     * @parameter expression="${processor.fileInPattern}" default-value=""
     * @since 0.10
     */
    protected String fileInPattern = "";
    /**
     * @parameter expression="${processor.fileOutPattern}" default-value=""
     * @since 0.10
     */
    protected String fileOutPattern = "";
    /**
     * Les filtres a utiliser par le processor, séparés par des virgules
     *
     * @parameter expression="${processor.filters}" default-value="org.nuiton.processor.filters.NoActionFilter"
     * @since 0.10
     */
    protected String filters;
    /**
     * Ecrase les fichiers générés
     *
     * @parameter expression="${processor.overwrite}" default-value="false"
     * @since 0.10
     */
    private boolean overwrite;
    /**
     * Permet d'obtenir plus d'information
     *
     * @parameter expression="${processor.verbose}" default-value="${maven.verbose}"
     * @since 0.10
     */
    private boolean verbose;
    /**
     * Les fichiers à traiter
     */
    protected Map<String, String> filesToTreate;

    public ProcessorPlugin() {
        super("all files are up-to-date.");
    }

    @Override
    public boolean isVerbose() {
        return verbose;
    }

    @Override
    public void setVerbose(boolean verbose) {
        this.verbose = verbose;
    }

    @Override
    public MavenProject getProject() {
        return project;
    }

    @Override
    public void setProject(MavenProject project) {
        this.project = project;
    }

    @Override
    protected boolean init() throws Exception {

        if (filters.isEmpty()) {
            getLog().warn("No filters to use, skip execution.");
            return false;
        }

        MirroredFileUpdater updater = overwrite ? null : new MirroredFileUpdater(fileInPattern, fileOutPattern, srcDir, destDir);

        updater.setDestinationDirectory(destDir);
        String[] aIncludes = includes.split(",");
        String[] aExcludes = excludes == null ? null : excludes.split(",");

        filesToTreate = getFilesToTreate(aIncludes, aExcludes, srcDir, updater);

        return true;
    }

    @Override
    protected void doAction() throws Exception {

        if (filesToTreate.isEmpty()) {
            getLog().info("No file to process.");
            return;
        }

        if (isVerbose()) {
            printConfig();
        }

        long t0 = System.nanoTime();

        Processor processor = ProcessorUtil.newProcessor(filters, ",");

        getLog().info("Processing " + filesToTreate.size() + " files(s).");

        for (Entry<String, String> entry : filesToTreate.entrySet()) {
            File srcFile = new File(srcDir, entry.getKey());
            File dstFile = new File(entry.getValue());

            if (isVerbose()) {
                getLog().info("Process " + srcFile);
            }
            // creation du repertoire pour le fichier destination
            dstFile.getParentFile().mkdirs();

            processor.process(new FileReader(srcFile), new FileWriter(dstFile));
        }

        if (isVerbose()) {
            getLog().info("done in " + PluginHelper.convertTime(System.nanoTime() - t0));
        }
        // on indique que le repertoire entrant n'est plus dans le build
        // car sinon on va avoir des classes dupliquées
        removeCompileSourceRoots(srcDir);

        // le repertoire sortant est dans le build de maven
        addCompileSourceRoots(destDir);
    }

    protected void printConfig() {
        getLog().info("config - srcDir         " + srcDir);
        getLog().info("config - destDir        " + destDir);
        getLog().info("config - includes       " + includes);
        getLog().info("config - filters        " + Arrays.asList(PluginHelper.splitAndTrim(filters, ",")));
        if (excludes != null) {
            getLog().info("config - excludes       " + excludes);
        }
        if (!fileInPattern.isEmpty()) {
            getLog().info("config - fileInPattern  " + fileInPattern);
        }
        if (!fileOutPattern.isEmpty()) {
            getLog().info("config - fileOutPattern " + fileOutPattern);
        }
        if (overwrite) {
            getLog().info("config - overwrite      " + overwrite);
        }
    }
}
