/*
 * *##% 
 * JRedmine maven plugin
 * Copyright (C) 2009 CodeLutin
 *
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Lesser General Public License as
 * published by the Free Software Foundation, either version 3 of the
 * License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Lesser Public License for more details.
 *
 * You should have received a copy of the GNU General Lesser Public
 * License along with this program.  If not, see
 * <http://www.gnu.org/licenses/lgpl-3.0.html>.
 * ##%*
 */
package org.nuiton.jredmine.plugin.announcement;

import org.apache.maven.plugin.MojoExecutionException;
import org.apache.maven.plugin.changes.ChangesXML;
import org.apache.velocity.app.VelocityEngine;
import org.apache.velocity.context.Context;
import org.codehaus.plexus.util.ReaderFactory;
import org.codehaus.plexus.util.StringUtils;
import org.codehaus.plexus.velocity.VelocityComponent;
import org.nuiton.jredmine.model.Attachment;
import org.nuiton.jredmine.plugin.AbstractRedmineMojo;
import org.nuiton.plugin.PluginHelper;

import java.io.File;
import java.util.Date;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.regex.Matcher;
import java.util.regex.Pattern;

/**
 * @author chemit
 * @requiresOnline true
 */
public abstract class AbstractAnnouncementMojo extends AbstractRedmineMojo implements AnnouncementGeneratorConfiguration {

    /**
     * Directory where the template file will be generated.
     *
     * @parameter expression="${redmine.templateOutputDirectory}" default-value="${project.build.directory}/generated-sources/announcement"
     * @required
     * @since 1.0.0
     */
    protected File templateOutputDirectory;
    /**
     * The path of the changes.xml file.
     *
     * @parameter expression="${redmine.xmlPath}" default-value="${basedir}/src/changes/changes.xml"
     * @required
     * @since 1.0.0
     */
    protected File xmlPath;
    /**
     * Directory that contains the template.
     * <p>
     * <b>Note:</b> This directory must be a subdirectory of
     * <code>/src/main/resources/ or current project base directory</code>.
     * </p>
     *
     * @parameter expression="${redmine.templateDirectory}" default-value="org/nuiton/jredmine/plugin/announcement"
     * @required
     * @since 1.0.0
     */
    protected String templateDirectory;
    /**
     * The template encoding.
     *
     * @parameter expression="${redmine.templateEncoding}" default-value="${project.build.sourceEncoding}"
     * @since 1.0.0
     */
    protected String templateEncoding;
    /**
     * Map which will be pass to the velocity context
     *
     * @parameter
     * @since 1.0.0
     */
    protected Map<String, Object> announceParameters;
    /**
     * Template strings per system that is used to discover the URL to use to display an attchment. Each key in this
     * map denotes the (case-sensitive) identifier of the issue tracking system and its value gives the URL template.
     * <p>
     * There are 2 template tokens you can use. <code>%URL%</code>: this is computed by getting the
     * <code>&lt;issueManagement&gt;/&lt;url&gt;</code> value from the POM, and removing the last '/'
     * and everything that comes after it. <code>%FILE%</code>: this is the issue number.
     * </p>
     *
     * @parameter expression="${redmine.attachmentLinkTemplate}" default-value="%URL%/attachments/download/%FILE%"
     * @since 1.0.0
     */
    protected String attachmentLinkTemplate;
    /**
     * @parameter expression="${project.groupId}"
     * @readonly
     * @since 1.0.0
     */
    protected String groupId;
    /**
     * @parameter expression="${project.artifactId}"
     * @readonly
     * @since 1.0.0
     */
    protected String artifactId;
    /**
     * Distribution url of the artifact.
     *
     * @parameter expression="${redmine.projectUrl}" default-value="${project.url}"
     * @required
     * @since 1.0.0
     */
    protected String projectUrl;
    /**
     * Packaging structure for the artifact.
     *
     * @parameter expression="${project.packaging}"
     * @readonly
     * @since 1.0.0
     */
    protected String packaging;
    /**
     * The name of the artifact to be used in the announcement.
     *
     * @parameter expression="${redmine.finalName}" default-value="${project.build.finalName}"
     * @required
     * @since 1.0.0
     */
    protected String finalName;
    /**
     * The current project base directory.
     *
     * @parameter expression="${basedir}"
     * @required
     * @since 1.0.0
     */
    protected String basedir;
    /**
     * URL where the artifact can be downloaded. If not specified,
     * no URL is used.
     *
     * @parameter
     * @since 1.0.0
     */
    protected String urlDownload;
    /**
     * Name of the team that develops the artifact.
     *
     * @parameter expression="${redmine.developmentTeam}" default-value="${project.name} team"
     * @required
     * @since 1.0.0
     */
    protected String developmentTeam;
    /**
     * Short description or introduction of the released artifact.
     *
     * @parameter expression="${redmine.introduction}" default-value="${project.description}"
     * @since 1.0.0
     */
    protected String introduction;
    /**
     * A flag to restirct only one run in a build (for multi-module context).
     *
     * @parameter expression="${redmine.runOnce}" default-value="true"
     * @since 1.0.0
     */
    protected boolean runOnce;
    /**
     * @parameter expression="${redmine.artifactsFile}"
     * @since 1.2.1
     */
    protected File artifactsFile;
    /**
     * @parameter expression="${redmine.deploymentUrl}"
     * @readonly
     */
    protected String deploymentUrl;
    /**
     * Velocity Component.
     *
     * @component roleHint="maven-helper-plugin"
     */
    protected VelocityComponent velocity;

    protected Attachment[] attachments;

    protected Map<File, String> artifactUrls;

    protected Map<Attachment, String> attachmentUrls;

    protected List<?> releases;

    protected AnnouncementGenerator generator;

    private static final Pattern ARTIFACT_PATTERN = Pattern.compile("(.+)?--(.+)?");

    /**
     * @return the name fo the template to use
     */
    protected abstract String getAnnouncementTemplate();

    protected AbstractAnnouncementMojo() {
        super(true, false, true);
    }

    ///////////////////////////////////////////////////////////////////////////
    /// AbstractRedmineMojo
    ///////////////////////////////////////////////////////////////////////////

    @Override
    protected boolean isRunOnce() {
        return runOnce;
    }

    @Override
    protected boolean checkRunOnceDone() {

        String template = getAnnouncementTemplate();
        File out = new File(templateOutputDirectory, template);
        Date buildStartTime = session == null ? null : session.getStartTime();
        Date newStartTime = out.exists() ? new Date(out.lastModified()) : null;

        boolean checkRunOnceDone = checkRunOnceDone(runOnce, true, buildStartTime, newStartTime);
        return checkRunOnceDone;
    }

    @Override
    protected void init() throws Exception {

        versionId = PluginHelper.removeSnapshotSuffix(versionId);

        runOnceDone = false;

        if (isRunOnce()) {
            runOnceDone = checkRunOnceDone();
            if (runOnceDone) {
                return;
            }
        }

        if (!xmlPath.exists()) {
            throw new MojoExecutionException("can not find redmine-template at " + xmlPath);
        }

        if (StringUtils.isEmpty(templateEncoding)) {
            templateEncoding = ReaderFactory.FILE_ENCODING;
            getLog().warn(
                    "File encoding has not been set, using platform encoding " + templateEncoding + ", i.e. build is platform dependent!");
        }

        if (StringUtils.isEmpty(introduction)) {
            introduction = project.getUrl();
        }

        if (artifactsFile != null) {
            if (!artifactsFile.exists()) {
                throw new MojoExecutionException("The artifactsFile [" + artifactsFile + "] does not exists");

            }

            File[] files = PluginHelper.getLinesAsFiles(artifactsFile);

            String url = deploymentUrl.trim();

            if (!url.endsWith("/")) {
                deploymentUrl = (url += "/");
            }

            if (isVerbose()) {
                getLog().info("Deploy url = " + url);
            }

            artifactUrls = new HashMap<File, String>();

            for (File f : files) {

                String artifactInfo = f.getParentFile().getName();
                Matcher matcher = ARTIFACT_PATTERN.matcher(artifactInfo);

                if (!matcher.matches()) {
                    getLog().error("no matching for file " + artifactInfo + " (" + f + ")");
                }
                String artifactId = matcher.group(2);
                String groupId = matcher.group(1).replaceAll("\\.", "/") + "/" + artifactId;
                String filename = f.getName();
                if ("pom.xml".equals(filename)) {
                    filename = artifactId + "-" + versionId + ".pom";
                    f = new File(filename);
                }
                String spec = url + groupId + "/" + versionId + "/" + filename;
                if (isVerbose()) {
                    getLog().info("artifact file " + f.getName() + " --> " + spec);
                }
                artifactUrls.put(f, spec);
            }
        }

        super.init();

        generator = new AnnouncementGenerator(this);

        Attachment[] attachments = service.getAttachments(projectId, versionId);

        if (attachments.length == 0) {
            getLog().info("No attachments files");
        } else {

            if (isVerbose()) {
                getLog().info("attachmentLinkTemplate " + attachmentLinkTemplate);
                getLog().info("server url " + url);
            }
            attachmentUrls = generator.getAttachmentsUrls(attachments);

        }
        
        if (isVerbose()) {
            for (Map.Entry<Attachment, String> e : attachmentUrls.entrySet()) {
                Attachment key = e.getKey();
                String value = e.getValue();
                getLog().info("attachment file " + key.getFilename() + " -->  " + value);
            }
        }

        ChangesXML changesXml = new ChangesXML(xmlPath, getLog());

        releases = changesXml.getReleaseList();
    }

    @Override
    protected void doAction() throws Exception {

        //TODO-TC-20091209 must obtain back the full name of dev from the project team on each action
        String template = getAnnouncementTemplate();

        File out = new File(templateOutputDirectory, template);

        getLog().info("Apply template [" + template + "]");
        getLog().info(" from : " + xmlPath);
        getLog().info(" to   : " + out);

        VelocityEngine engine = velocity.getEngine();

        engine.setApplicationAttribute("baseDirectory", basedir);

        Context context = generator.createVelocityContext(releases);

        generator.doGenerate(engine, context, out, templateDirectory + "/" + template, templateEncoding);

        getLog().debug("Created  announcement [" + template + "] in  " + out);
    }

    ///////////////////////////////////////////////////////////////////////////
    /// AnnouncementGeneratorConfiguration
    ///////////////////////////////////////////////////////////////////////////
    @Override
    public String getAttachmentLinkTemplate() {
        return attachmentLinkTemplate;
    }

    @Override
    public String getUrl() {
        return url.toString();
    }

    @Override
    public String getArtifactId() {
        return artifactId;
    }

    @Override
    public Map<File, String> getArtifactUrls() {
        return artifactUrls;
    }

    @Override
    public Map<Attachment, String> getAttachmentUrls() {
        return attachmentUrls;
    }

    @Override
    public String getBasedir() {
        return basedir;
    }

    @Override
    public String getDeploymentUrl() {
        return deploymentUrl;
    }

    @Override
    public String getDevelopmentTeam() {
        return developmentTeam;
    }

    @Override
    public String getFinalName() {
        return finalName;
    }

    @Override
    public String getGroupId() {
        return groupId;
    }

    @Override
    public String getIntroduction() {
        return introduction;
    }

    @Override
    public String getPackaging() {
        return packaging;
    }

    @Override
    public String getProjectUrl() {
        return projectUrl;
    }

    @Override
    public String getUrlDownload() {
        return urlDownload;
    }

    @Override
    public Map<String, Object> getAnnounceParameters() {
        return announceParameters;
    }
}
