/*
 * #%L
 * JAXX :: Widgets
 * $Id: ListSelectorModel.java 2455 2012-07-30 18:47:43Z tchemit $
 * $HeadURL: http://svn.nuiton.org/svn/jaxx/tags/jaxx-2.5.28/jaxx-widgets/src/main/java/jaxx/runtime/swing/ListSelectorModel.java $
 * %%
 * Copyright (C) 2008 - 2012 CodeLutin
 * %%
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Lesser General Public License as 
 * published by the Free Software Foundation, either version 3 of the 
 * License, or (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Lesser Public License for more details.
 * 
 * You should have received a copy of the GNU General Lesser Public 
 * License along with this program.  If not, see
 * <http://www.gnu.org/licenses/lgpl-3.0.html>.
 * #L%
 */
package jaxx.runtime.swing;

import com.google.common.collect.Lists;
import java.beans.PropertyChangeEvent;
import java.beans.PropertyChangeListener;
import java.beans.PropertyChangeSupport;
import java.util.Collection;
import jaxx.runtime.swing.model.GenericListListener;
import jaxx.runtime.swing.model.GenericListModel;

/**
 * @author sletellier <letellier@codelutin.com>
 */
public class ListSelectorModel<B> {

    protected final PropertyChangeSupport pcs = new PropertyChangeSupport(this);

    public static final String PROPERTY_ADD_ENABLED = "addEnabled";
    public static final String PROPERTY_REMOVE_ENABLED = "removeEnabled";

    public static final String PROPERTY_SELECTED_VALUES = "selectedValues";
    public static final String PROPERTY_VALUES = "values";

    protected boolean addEnabled;
    protected boolean removeEnabled;

    protected GenericListModel<B> fromModel;
    protected GenericListModel<B> toModel;

    protected ListSelectorModel() {

        fromModel = new GenericListModel<B>();
        fromModel.addPropertyChangeListener(GenericListModel.PROPERTY_SELECTED_VALUE, new PropertyChangeListener() {
            @Override
            public void propertyChange(PropertyChangeEvent evt) {
                setAddEnabled(fromModel.hasSelectedIndex());
            }
        });

        toModel = new GenericListModel<B>();
        toModel.addPropertyChangeListener(GenericListModel.PROPERTY_SELECTED_VALUE, new PropertyChangeListener() {
            @Override
            public void propertyChange(PropertyChangeEvent evt) {
                setRemoveEnabled(toModel.hasSelectedIndex());
            }
        });
    }

    public boolean isAddEnabled() {
        return addEnabled;
    }

    public void setAddEnabled(boolean addEnabled) {
        boolean oldValue = isAddEnabled();
        this.addEnabled = addEnabled;
        firePropertyChange(PROPERTY_ADD_ENABLED, oldValue, addEnabled);
    }

    public boolean isRemoveEnabled() {
        return removeEnabled;
    }

    public void setRemoveEnabled(boolean removeEnabled) {
        boolean oldValue = isRemoveEnabled();
        this.removeEnabled = removeEnabled;
        firePropertyChange(PROPERTY_REMOVE_ENABLED, oldValue, removeEnabled);
    }

    public GenericListModel<B> getFromModel() {
        return fromModel;
    }

    public GenericListModel<B> getToModel() {
        return toModel;
    }

    public Collection<B> getValues() {
        return fromModel.getElements();
    }

    public void setValues(Collection<B> values) {
        Collection<B> oldValues = getValues();
        toModel.clearElements();
        fromModel.setElements(values);
        firePropertyChange(PROPERTY_VALUES, oldValues, values);
    }

    public void addValues(Collection<B> values) {
        Collection<B> oldValues = getValues();
        toModel.removeElements(values);
        fromModel.removeElements(values);
        fromModel.addElements(values);
        firePropertyChange(PROPERTY_VALUES, oldValues, values);
    }

    public void removeValues(Collection<B> values) {
        Collection<B> oldValues = getValues();
        toModel.removeElements(values);
        fromModel.removeElements(values);
        firePropertyChange(PROPERTY_VALUES, oldValues, values);
    }

    public Collection<B> getSelectedValues() {
        return toModel.getElements();
    }

    public void setSelectedValues(Collection<B> selectedValues) {
        Collection<B> oldValue = getSelectedValues();
        fromModel.removeElements(selectedValues);
        toModel.removeElements(selectedValues);
        toModel.setElements(selectedValues);
        firePropertyChange(PROPERTY_SELECTED_VALUES, oldValue, selectedValues);
    }

    public void add() {
        Collection<B> oldValues = getValues();
        Collection<B> oldSelectedValue = getSelectedValues();
        moveSelect(fromModel, toModel);
        firePropertyChange(PROPERTY_VALUES, oldValues, getValues());
        firePropertyChange(PROPERTY_SELECTED_VALUES, oldSelectedValue, getSelectedValues());
    }

    public void remove() {
        moveSelect(toModel, fromModel);
    }

    protected void moveSelect(GenericListModel<B> from, GenericListModel<B> to) {
        Collection<B> selectedValues = Lists.newArrayList(from.getSelectedValues());
        from.removeElements(selectedValues);
        to.addElements(selectedValues);
    }

    public void addGenericListListener(GenericListListener l) {
        toModel.addGenericListListener(l);
    }

    public void removeGenericListListener(GenericListListener l) {
        toModel.removeGenericListListener(l);
    }

    public void addPropertyChangeListener(PropertyChangeListener listener) {
        pcs.addPropertyChangeListener(listener);
    }

    public void addPropertyChangeListener(String propertyName, PropertyChangeListener listener) {
        pcs.addPropertyChangeListener(propertyName, listener);
    }

    public void removePropertyChangeListener(PropertyChangeListener listener) {
        pcs.removePropertyChangeListener(listener);
    }

    public void removePropertyChangeListener(String propertyName, PropertyChangeListener listener) {
        pcs.removePropertyChangeListener(propertyName, listener);
    }

    protected void firePropertyChange(String propertyName, Object oldValue, Object newValue) {
        pcs.firePropertyChange(propertyName, oldValue, newValue);
    }

    protected void fireIndexedPropertyChange(String propertyName, int index, Object oldValue, Object newValue) {
        pcs.fireIndexedPropertyChange(propertyName, index, oldValue, newValue);
    }
}
