package jaxx.runtime.swing.editor.bean;

/*
 * #%L
 * JAXX :: Widgets
 * $Id: BeanDoubleListModel.java 2719 2013-09-20 12:51:47Z tchemit $
 * $HeadURL: http://svn.nuiton.org/svn/jaxx/tags/jaxx-2.5.26/jaxx-widgets/src/main/java/jaxx/runtime/swing/editor/bean/BeanDoubleListModel.java $
 * %%
 * Copyright (C) 2008 - 2012 CodeLutin
 * %%
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Lesser General Public License as 
 * published by the Free Software Foundation, either version 3 of the 
 * License, or (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Lesser Public License for more details.
 * 
 * You should have received a copy of the GNU General Lesser Public 
 * License along with this program.  If not, see
 * <http://www.gnu.org/licenses/lgpl-3.0.html>.
 * #L%
 */

import com.google.common.collect.Lists;
import jaxx.runtime.swing.model.JaxxDefaultListModel;
import jaxx.runtime.swing.model.JaxxFilterableListModel;
import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.jdesktop.beans.AbstractSerializableBean;

import javax.swing.ListModel;
import java.util.ArrayList;
import java.util.List;

/**
 * The model of the {@link BeanDoubleList} widget
 *
 * @param <O> the type of the object in the list
 * @author kmorin <morin@codelutin.com>
 * @see BeanDoubleList
 * @since 2.5.8
 */
public class BeanDoubleListModel<O> extends AbstractSerializableBean {

    private static final Log log = LogFactory.getLog(BeanDoubleListModel.class);

    public static final String PROPERTY_ADD_ENABLED = "addEnabled";

    public static final String PROPERTY_REMOVE_ENABLED = "removeEnabled";

    public static final String PROPERTY_SELECTED_UP_ENABLED = "selectedUpEnabled";

    public static final String PROPERTY_SELECTED_DOWN_ENABLED = "selectedDownEnabled";

    private static final long serialVersionUID = 1L;

    /** List of all the available items */
    protected List<O> universe = new ArrayList<O>();

    /** Model containing the remaining available items */
    protected JaxxFilterableListModel<O> universeModel = new JaxxFilterableListModel<O>();

    /** List of the selected items */
    protected List<O> selected = new ArrayList<O>();

    /** Model containing the selected items */
    protected JaxxDefaultListModel<O> selectedModel = new JaxxDefaultListModel<O>();

    protected boolean addEnabled;

    protected boolean removeEnabled;

    protected boolean selectedUpEnabled;

    protected boolean selectedDownEnabled;

    /**
     * To get the selected items.
     *
     * @return a list of O
     */
    public List<O> getSelected() {
        return selected;
    }

    /**
     * Sets the list of selected items.
     * It fills the model of the list of the selected items with these items
     * and removes them from the model of list of the universe.
     *
     * @param selected a list of O
     */
    public void setSelected(List<O> selected) {
        //reset all the universe in the universe list
        resetUniverse();

        if (selected == null) {
            selected = Lists.newArrayList();
        }
        this.selected.clear();
        selectedModel.clear();
        addToSelected(selected);
    }

    /**
     * To get all the available items.
     *
     * @return a list of O
     */
    public List<O> getUniverse() {
        return universe;
    }

    /**
     * Sets the list of the available items.
     * It fills the model of the universe list with these items.
     *
     * @param universe a list of O
     */
    public void setUniverse(List<O> universe) {
        if (universe == null) {
            this.universe = Lists.newArrayList();
        } else {
            this.universe = Lists.newArrayList(universe);
        }
        resetUniverse();
    }

    protected void resetUniverse() {
        universeModel.setAllElements(universe);
    }

    public ListModel getSelectedModel() {
        return selectedModel;
    }

    public ListModel getUniverseModel() {
        return universeModel;
    }

    /**
     * Adds an item to the selected items.
     *
     * @param item the item to select
     */
    public void addToSelected(O item) {
        selected.add(item);
        selectedModel.addElement(item);
        universeModel.removeElement(item);
    }

    /**
     * Adds a list of items to the selected items.
     *
     * @param items the list of the items to select
     */
    public void addToSelected(List<O> items) {
        selected.addAll(items);
        for (O item : items) {
            selectedModel.addElement(item);
            universeModel.removeElement(item);
        }
    }

    /**
     * Removes an item from the selected items.
     *
     * @param item the item to unselect
     */
    public void removeFromSelected(O item) {
        selected.remove(item);
        selectedModel.removeElement(item);
        addToUniverseList(item);
    }

    /**
     * Removes a list of items from the list of selected items.
     *
     * @param items the list of the items to unselect
     */
    public void removeFromSelected(List<O> items) {
        selected.removeAll(items);
        for (O item : items) {
            selectedModel.removeElement(item);
            addToUniverseList(item);
        }
    }

    /**
     * Move up a selected item.
     *
     * @param item the selected item
     * @since 2.5.26
     */
    public void moveUpSelected(O item) {
        int i = selected.indexOf(item);
        selected.remove(item);
        selected.add(i - 1, item);
        selectedModel.removeElement(item);
        selectedModel.insertElementAt(item, i - 1);
    }

    /**
     * Move down a selected item.
     *
     * @param item the selected item
     * @since 2.5.26
     */
    public void moveDownSelected(O item) {
        int i = selected.indexOf(item);
        selected.remove(item);
        selected.add(i + 1, item);
        selectedModel.removeElement(item);
        selectedModel.insertElementAt(item, i + 1);
    }

    public boolean isAddEnabled() {
        return addEnabled;
    }

    public void setAddEnabled(boolean addEnabled) {
        boolean oldValue = isAddEnabled();
        this.addEnabled = addEnabled;
        firePropertyChange(PROPERTY_ADD_ENABLED, oldValue, addEnabled);
    }

    public boolean isRemoveEnabled() {
        return removeEnabled;
    }

    public void setRemoveEnabled(boolean removeEnabled) {
        boolean oldValue = isRemoveEnabled();
        this.removeEnabled = removeEnabled;
        firePropertyChange(PROPERTY_REMOVE_ENABLED, oldValue, removeEnabled);
    }

    public boolean isSelectedUpEnabled() {
        return selectedUpEnabled;
    }

    public void setSelectedUpEnabled(boolean selectedUpEnabled) {
        boolean oldValue = isSelectedUpEnabled();
        this.selectedUpEnabled = selectedUpEnabled;
        firePropertyChange(PROPERTY_SELECTED_UP_ENABLED, oldValue, selectedUpEnabled);
    }

    public boolean isSelectedDownEnabled() {
        return selectedDownEnabled;
    }

    public void setSelectedDownEnabled(boolean selectedDownEnabled) {
        boolean oldValue = isSelectedDownEnabled();
        this.selectedDownEnabled = selectedDownEnabled;
        firePropertyChange(PROPERTY_SELECTED_DOWN_ENABLED, oldValue, selectedDownEnabled);
    }

    /**
     * Adds an item to the available items list at the right index
     * to keep always the same order.
     *
     * @param item the item to add to the universe list
     */
    protected void addToUniverseList(O item) {
        // the maximum index where we should insert the item is its index
        // in the list of all the available items 
        int index = universe.indexOf(item);
        // if the index is upper than the size of the list of the remaining available items,
        // get the size of this list
        int insertionIndex = Math.min(index, universeModel.getSize());

        // we decrease the index to insert until we meet an item whose index
        // in the list of all the available index is lower than the one of the item
        // we want to insert
        while (insertionIndex > 0) {
            O o = universeModel.get(--insertionIndex);
            int oIndex = universe.indexOf(o);
            if (oIndex < index) {
                insertionIndex++;
                break;
            }
        }

        universeModel.add(insertionIndex, item);
    }

    public int getSelectedListSize() {
        return selected.size();
    }
}
