/*
 * *##% 
 * JAXX Runtime
 * Copyright (C) 2008 - 2009 CodeLutin
 *
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Lesser General Public License as
 * published by the Free Software Foundation, either version 3 of the
 * License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Lesser Public License for more details.
 *
 * You should have received a copy of the GNU General Lesser Public
 * License along with this program.  If not, see
 * <http://www.gnu.org/licenses/lgpl-3.0.html>.
 * ##%*
 */
package jaxx.runtime.validator.field;

import com.opensymphony.xwork2.validator.ValidationException;
import com.opensymphony.xwork2.validator.validators.FieldValidatorSupport;

import java.io.File;

/**
 * <!-- START SNIPPET: javadoc -->
 * NotExistingFileFieldValidator checks that a File field as a file does not exist. *
 * <!-- END SNIPPET: javadoc -->
 * <p/>
 * <p/>
 * <!-- START SNIPPET: parameters -->
 * <ul>
 * <li>fieldName - The field name this validator is validating. Required if using Plain-Validator Syntax otherwise not required</li>
 * </ul>
 * <!-- END SNIPPET: parameters -->
 * <p/>
 * <p/>
 * <pre>
 * <!-- START SNIPPET: examples -->
 *     &lt;validators&gt;
 *         &lt;!-- Plain-Validator Syntax --&gt;
 *         &lt;validator type="notExistingFile"&gt;
 *             &lt;param name="fieldName"&gt;tmp&lt;/param&gt;
 *             &lt;message&gt;tmp is an existing file&lt;/message&gt;
 *         &lt;/validator&gt;
 * <p/>
 *         &lt;!-- Field-Validator Syntax --&gt;
 *         &lt;field name="tmp"&gt;
 *         	  &lt;field-validator type="notExistingFile"&gt;
 *                 &lt;message&gt;tmp is an existing file&lt;/message&gt;
 *            &lt;/field-validator&gt;
 *         &lt;/field&gt;
 *     &lt;/validators&gt;
 * <!-- END SNIPPET: examples -->
 * </pre>
 *
 * @author chemit
 */
public class NotExistingFileFieldValidator extends FieldValidatorSupport {

    @Override
    public void validate(Object object) throws ValidationException {
        String fieldName = getFieldName();
        Object value = this.getFieldValue(fieldName, object);
        if (value==null) {
            // no value defined
            addFieldError(fieldName, object);            
            return;
        }
        File f;
        if (value instanceof File) {
            f = (File) value;
        } else if (value instanceof String) {
            f = new File((String) value);
        } else {
            addFieldError(fieldName, object);
            return;
        }

        if (f.exists() || f.isDirectory()) {
            // f is not a file and exist
            addFieldError(fieldName, object);
        }
    }

    @Override
    public String getValidatorType() {
        return "notExistingFile";
    }
}