/*
 * Copyright 2006 Ethan Nicholas. All rights reserved.
 * Use is subject to license terms.
 */
package jaxx.runtime.swing;

import javax.swing.AbstractListModel;
import javax.swing.ComboBoxModel;
import javax.swing.DefaultListCellRenderer;
import javax.swing.JComboBox;
import javax.swing.JList;
import javax.swing.ListModel;
import java.awt.Component;
import java.awt.event.ItemEvent;
import java.awt.event.ItemListener;
import java.beans.PropertyChangeEvent;
import java.beans.PropertyChangeListener;
import java.lang.reflect.InvocationTargetException;
import java.lang.reflect.Method;
import java.util.ArrayList;
import java.util.Collection;
import java.util.List;

public class JAXXComboBox extends JComboBox {

    private static final long serialVersionUID = 1L;

    public class JAXXComboBoxModel extends AbstractListModel implements ComboBoxModel {

        private List<Item> items;
        private Object selectedItem;
        private static final long serialVersionUID = -8940733376638766414L;

        public JAXXComboBoxModel(List<Item> items) {
            this.items = items;

            PropertyChangeListener listener = new PropertyChangeListener() {

                @Override
                public void propertyChange(PropertyChangeEvent e) {
                    if (e.getPropertyName().equals(Item.SELECTED_PROPERTY)) {
                        Item item = (Item) e.getSource();
                        int itemIndex = JAXXComboBoxModel.this.items.indexOf(item);
                        // TODO: fix cut-and-pasting badness
                        int[] oldSelection = new int[]{getSelectedIndex()};
                        int[] newSelection;
                        int index = -1;
                        for (int i = 0; i < oldSelection.length; i++) {
                            if (oldSelection[i] == itemIndex) {
                                index = i;
                                break;
                            }
                        }
                        if (item.isSelected()) {
                            if (index != -1) // it was already selected
                            {
                                return;
                            }
                            newSelection = new int[oldSelection.length + 1];
                            System.arraycopy(oldSelection, 0, newSelection, 0, oldSelection.length);
                            newSelection[newSelection.length - 1] = itemIndex;
                        } else {
                            if (index == -1) // it already wasn't selected
                            {
                                return;
                            }
                            newSelection = new int[oldSelection.length - 1];
                            System.arraycopy(oldSelection, 0, newSelection, 0, index);
                            System.arraycopy(oldSelection, index + 1, newSelection, index, oldSelection.length - 1 - index);
                        }
                        setSelectedIndex(newSelection[0]);
                    } else {
                        // TODO: more cut-and-pasting badness
                        for (int i = 0; i < getSize(); i++) {
                            if (getElementAt(i) == ((Item) e.getSource()).getValue()) {
                                fireContentsChanged(JAXXComboBoxModel.this, i, i);
                                if (getSelectedIndex() == i) {
                                    fireItemStateChanged(new ItemEvent(JAXXComboBox.this, ItemEvent.ITEM_STATE_CHANGED, getElementAt(i), ItemEvent.DESELECTED));
                                }
                                return;
                            }
                        }
                    }
                }
            };
            for (Item item : items) {
                item.addPropertyChangeListener(listener);
            }
        }

        @Override
        public Object getElementAt(int i) {
            return items.get(i).getValue();
        }

        @Override
        public int getSize() {
            return items.size();
        }

        @Override
        public Object getSelectedItem() {
            return selectedItem;
        }

        @Override
        public void setSelectedItem(Object selectedItem) {
            if ((this.selectedItem != null && !this.selectedItem.equals(selectedItem)) ||
                    this.selectedItem == null && selectedItem != null) {
                this.selectedItem = selectedItem;
                fireContentsChanged(this, -1, -1);
            }
        }
    }

    public JAXXComboBox() {
        setRenderer(new DefaultListCellRenderer() {

            private static final long serialVersionUID = 1L;

            @Override
            public Component getListCellRendererComponent(JList list, Object value, int index, boolean isSelected, boolean cellHasFocus) {
                ListModel model = list.getModel();
                if (model instanceof JAXXComboBoxModel) {
                    List/*<Item>*/ items = ((JAXXComboBoxModel) model).items;
                    Item item = null;
                    if (index == -1) {
                        for (Object item1 : items) {
                            Item testItem = (Item) item1;
                            if (testItem.getValue() == value) {
                                item = testItem;
                                break;
                            }
                        }
                    } else {
                        item = (Item) items.get(index);
                    }

                    if (item != null) {
                        String label = item.getLabel();
                        if (label != null) {
                            value = label;
                        }
                    }
                }
                return super.getListCellRendererComponent(list, value, index, isSelected, cellHasFocus);
            }
        });

        addItemListener(new ItemListener() {

            @Override
            public void itemStateChanged(ItemEvent e) {
                ListModel model = getModel();
                if (model instanceof JAXXComboBoxModel) {
                    List<Item> items = ((JAXXComboBoxModel) model).items;
                    for (int i = items.size() - 1; i >= 0; i--) {
                        boolean selected = getSelectedIndex() == i;
                        Item item = items.get(i);
                        if (selected != item.isSelected()) {
                            item.setSelected(selected);
                        }
                    }
                }
            }
        });
    }

    /**
     * Fill a combo box model with some datas, and select after all the given object
     *
     * @param data       data ot inject in combo
     * @param select     the object to select in combo after reflling his model
     * @param methodName method to invoke to display data's name
     */
    public void fillComboBox(Collection<?> data, Object select, String methodName) {
        // prepare method to use
        Method m;
        try {
            m = select.getClass().getMethod(methodName);
            m.setAccessible(true);
        } catch (NoSuchMethodException e) {
            throw new IllegalArgumentException("could not find method " + methodName + " on " + select.getClass());
        }

        List<Item> items = new ArrayList<Item>();
        for (Object o : data) {
            boolean selected = o.equals(select);
            try {
                items.add(new Item(o.toString(), (String) m.invoke(o), o, selected));
            } catch (IllegalAccessException e) {
                // shoudl never happen ?
                throw new RuntimeException(e);
            } catch (InvocationTargetException e) {
                // shoudl never happen ?
                throw new RuntimeException(e);
            }
        }
        setItems(items);
    }

    // this way we can keep it marked protected and still allow code in this file to call it
    @Override
    protected void fireItemStateChanged(ItemEvent e) {
        super.fireItemStateChanged(e);
    }

    public void setItems(List<Item> items) {
        setModel(new JAXXComboBoxModel(items));
        List<Integer> selectedIndexList = new ArrayList<Integer>();
        for (int i = 0; i < items.size(); i++) {
            if (items.get(i).isSelected()) {
                selectedIndexList.add(i);
            }
        }
        int[] selectedIndices = new int[selectedIndexList.size()];
        for (int i = 0; i < selectedIndexList.size(); i++) {
            selectedIndices[i] = selectedIndexList.get(i);
        }
        if (selectedIndices.length > 0) {
            setSelectedIndex(selectedIndices[0]);
        }
    }
}
