/*
 * #%L
 * JAXX :: Compiler
 * 
 * $Id: JRadioButtonHandler.java 1873 2010-05-04 19:48:38Z tchemit $
 * $HeadURL: http://svn.nuiton.org/svn/jaxx/tags/jaxx-2.1.1/jaxx-compiler/src/main/java/jaxx/compiler/tags/swing/JRadioButtonHandler.java $
 * %%
 * Copyright (C) 2008 - 2010 CodeLutin
 * %%
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Lesser General Public License as 
 * published by the Free Software Foundation, either version 3 of the 
 * License, or (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Lesser Public License for more details.
 * 
 * You should have received a copy of the GNU General Lesser Public 
 * License along with this program.  If not, see
 * <http://www.gnu.org/licenses/lgpl-3.0.html>.
 * #L%
 */

package jaxx.compiler.tags.swing;

import jaxx.compiler.CompilerException;
import jaxx.compiler.UnsupportedAttributeException;
import jaxx.compiler.CompiledObject;
import jaxx.compiler.JAXXCompiler;
import jaxx.compiler.reflect.ClassDescriptor;
import jaxx.compiler.reflect.ClassDescriptorHelper;
import jaxx.compiler.tags.DefaultComponentHandler;
import jaxx.compiler.types.TypeManager;
import jaxx.runtime.swing.JAXXButtonGroup;

import javax.swing.AbstractButton;
import javax.swing.event.ChangeListener;

public class JRadioButtonHandler extends DefaultComponentHandler {

    private static final String VALUE_PROPERTY = JAXXButtonGroup.VALUE_CLIENT_PROPERTY.substring(1);
    private static final String BUTTON_GROUP_PROPERTY = JAXXButtonGroup.BUTTON8GROUP_CLIENT_PROPERTY.substring(1);

    public JRadioButtonHandler(ClassDescriptor beanClass) {
        super(beanClass);
        ClassDescriptorHelper.checkSupportClass(getClass(), beanClass, AbstractButton.class);
    }

    @Override
    protected void configureProxyEventInfo() {
        super.configureProxyEventInfo();
        addProxyEventInfo("isSelected", ChangeListener.class, "model");
    }

    @Override
    public ClassDescriptor getPropertyType(CompiledObject object, String name, JAXXCompiler compiler) throws CompilerException {
        if (name.equals(BUTTON_GROUP_PROPERTY)) {
            return null; // accepts either a String or a ButtonGroup
        } else if (name.equals(VALUE_PROPERTY)) {
            return ClassDescriptorHelper.getClassDescriptor(Object.class);
        } else {
            return super.getPropertyType(object, name, compiler);
        }
    }

    @Override
    public boolean isMemberBound(String name) throws UnsupportedAttributeException {
        return !(name.equals(BUTTON_GROUP_PROPERTY) || name.equals(VALUE_PROPERTY)) && super.isMemberBound(name);
    }

    // handle buttonGroup assignment in addition block rather than initialization block
    @Override
    public void setProperty(CompiledObject object, String name, Object value, JAXXCompiler compiler) {
        if (name.equals(BUTTON_GROUP_PROPERTY)) {
            object.appendAdditionCode(getSetPropertyCode(object.getJavaCode(), name, TypeManager.getJavaCode(value), compiler));
        } else {
            super.setProperty(object, name, value, compiler);
        }
    }

    @Override
    public String getSetPropertyCode(String id, String name, String valueCode, JAXXCompiler compiler) throws CompilerException {
        if (name.equals(BUTTON_GROUP_PROPERTY)) {
            if (valueCode.startsWith("\"") && valueCode.endsWith("\"")) {
                valueCode = valueCode.substring(1, valueCode.length() - 1);
                CompiledObject buttonGroup = compiler.getCompiledObject(valueCode);
                if (buttonGroup == null) {
                    buttonGroup = new CompiledObject(valueCode, ClassDescriptorHelper.getClassDescriptor(JAXXButtonGroup.class), compiler);
                    compiler.registerCompiledObject(buttonGroup);
                }
            }
            return "{ javax.swing.ButtonGroup $buttonGroup = " + valueCode + "; " + id + ".putClientProperty(\"$buttonGroup\", $buttonGroup); $buttonGroup.add(" + id + "); }\n";
        } else if (name.equals(VALUE_PROPERTY)) {
            return "{ " + id + ".putClientProperty(\"" + JAXXButtonGroup.VALUE_CLIENT_PROPERTY + "\", " + valueCode + ");  Object $buttonGroup = " + id + ".getClientProperty(\"" + JAXXButtonGroup.BUTTON8GROUP_CLIENT_PROPERTY + "\");" +
                    " if ($buttonGroup instanceof jaxx.runtime.swing.JAXXButtonGroup) { ((jaxx.runtime.swing.JAXXButtonGroup) $buttonGroup).updateSelectedValue(); } }\n";
        } else {
            return super.getSetPropertyCode(id, name, valueCode, compiler);
        }
    }
}
